# encoding: utf-8
require "logstash/codecs/base"
require "logstash/util/charset"
require "csv"

class LogStash::Codecs::CSV < LogStash::Codecs::Base

  config_name "csv"

  # Define a list of column names (in the order they appear in the CSV,
  # as if it were a header line). If `columns` is not configured, or there
  # are not enough columns specified, the default column names are
  # "column1", "column2", etc. In the case that there are more columns
  # in the data than specified in this column list, extra columns will be auto-numbered:
  # (e.g. "user_defined_1", "user_defined_2", "column3", "column4", etc.)
  config :columns, :validate => :array, :default => []

  # Define the column separator value. If this is not specified, the default
  # is a comma `,`.
  # Optional.
  config :separator, :validate => :string, :default => ","

  # Define the character used to quote CSV fields. If this is not specified
  # the default is a double quote `"`.
  # Optional.
  config :quote_char, :validate => :string, :default => '"'

  # Treats the first line received as the hearder information, this information will
  # be used to compose the field names in the generated events. Note this information can
  # be reset on demand, useful for example when dealing with new files in the file input
  # or new request in the http_poller. Default => false
  config :include_headers, :validate => :boolean, :default => false

  # Define whether column names should autogenerated or not.
  # Defaults to true. If set to false, columns not having a header specified will not be parsed.
  config :autogenerate_column_names, :validate => :boolean, :default => true

  # Define whether empty columns should be skipped.
  # Defaults to false. If set to true, columns containing no value will not get set.
  config :skip_empty_columns, :validate => :boolean, :default => false

  # Define a set of datatype conversions to be applied to columns.
  # Possible conversions are integer, float, date, date_time, boolean
  #
  # # Example:
  # [source,ruby]
  #     filter {
  #       csv {
  #         convert => { "column1" => "integer", "column2" => "boolean" }
  #       }
  #     }
  config :convert, :validate => :hash, :default => {}

  ##
  # List of valid conversion types used for the convert option
  ##
  VALID_CONVERT_TYPES = [ "integer", "float", "date", "date_time", "boolean" ].freeze


  # The character encoding used in this codec. Examples include "UTF-8" and
  # "CP1252".
  config :charset, :validate => ::Encoding.name_list, :default => "UTF-8"

  def register
    @converter = LogStash::Util::Charset.new(@charset)
    @converter.logger = @logger

    # validate conversion types to be the valid ones.
    @convert.each_pair do |column, type|
      if !VALID_CONVERT_TYPES.include?(type)
        raise LogStash::ConfigurationError, "#{type} is not a valid conversion type."
      end
    end

    @headers = false
    @options = { :col_sep => @separator, :quote_char => @quote_char }
  end

  def decode(data)
    data = @converter.convert(data)
    begin
      values = CSV.parse_line(data, @options)
      if @include_headers && !@headers
        @headers = true
        @options[:headers] = values
      else
        decoded = {}
        values.each_with_index do |fields, index|
          field_name, value = nil, nil
          if fields.is_a?(String) && !( @skip_empty_columns && fields.nil?)  # No headers
            next if ignore_field?(index)
            field_name =  ( !@columns[index].nil? ? @columns[index] : "column#{(index+1)}")
            value      = fields
          elsif fields.is_a?(Array) # Got headers
            field_name = fields[0]
            value      = fields[1]
          end
          next unless field_name
          decoded[field_name] = if should_transform?(field_name)
                                  transform(field_name, value)
                                else
                                  value
                                end
        end
        yield LogStash::Event.new(decoded) if block_given?
      end
    rescue CSV::MalformedCSVError => e
      @logger.info("CSV parse failure. Falling back to plain-text", :error => e, :data => data)
      yield LogStash::Event.new("message" => data, "tags" => ["_csvparsefailure"]) if block_given?
    end
  end

  def encode(event)
    csv_data = CSV.generate_line(event.to_hash.values, @options)
    @on_event.call(event, csv_data)
  end

  def reset
    @headers = false
    @options.delete(:headers)
  end

  private

  def ignore_field?(index)
    !@columns[index] && !@autogenerate_column_names
  end

  def should_transform?(field_name)
    !@convert[field_name].nil?
  end

  def transform(field_name, value)
    transformation = @convert[field_name].to_sym
    converters[transformation].call(value)
  end

  def converters
    @converters ||= {
      :integer => lambda do |value|
        CSV::Converters[:integer].call(value)
      end,
      :float => lambda do |value|
        CSV::Converters[:float].call(value)

      end,
      :date => lambda do |value|
        CSV::Converters[:date].call(value)

      end,
      :date_time => lambda do |value|
        CSV::Converters[:date_time].call(value)
      end,
      :boolean => lambda do |value|
        value = value.strip.downcase
        return false if value == "false"
        return true  if value == "true"
        return value
      end
    }
  end

end # class LogStash::Codecs::Plain
