require 'protocol_buffers/compiler/descriptor.pb'
require 'stringio'

class FileDescriptorToD < Struct.new(:descriptor)

  include Google::Protobuf::FieldDescriptorProto::Type
  include Google::Protobuf::FieldDescriptorProto::Label

  def initialize(descriptor)
    super
    @module = descriptor.package_
    @ns = []
  end

  def module_name
    @module
  end

  def class_name(klass)
    klass
  end

  def write(io)
    @io = io

    @io.write <<HEADER
// Generated by the protocol buffer compiler. DO NOT EDIT!

static import std.string;
import protocol_buffers.message;

HEADER

    descriptor.message_type.each do |message|
      dump_message(message)
    end
  end

  protected

  def dump_message(message)
    in_namespace("final class", class_name(message.name), " : Message") do

      line %{static const #{class_name(message.name)} default_instance;}
      line %{string typeName() { return "#{class_name(message.name)}"; }}

      line %{// nested messages} unless message.nested_type.empty?
      message.nested_type.each { |inner| dump_message(inner) }

      line %{// nested enums} unless message.enum_type.empty?
      message.enum_type.each { |inner| dump_enum(inner) }

      singular_fields = message.field.find_all { |f|
        f.label == LABEL_OPTIONAL || f.label == LABEL_REQUIRED }
      repeated_fields = message.field.find_all { |f| f.label == LABEL_REPEATED }

      line

      singular_fields.each { |f| define_field(f, false) }
      repeated_fields.each { |f| define_field(f, true) }

      line %{enum __tags { #{message.field.map { |f| "#{f.name} = #{f.number}" }.join(", ")} }}

      unless singular_fields.empty?
        in_namespace("private", "") do
          line "ubyte[(#{singular_fields.size}/32)+1] _tags;"
          line %{void _set_bit(uint idx)   { _tags[idx / 32] |= (1u << (idx % 32)); }}
          line %{void _clear_bit(uint idx) { _tags[idx / 32] &= ~(1u << (idx % 32)); }}
          line %{bool _check_bit(uint idx) { return (_tags[idx / 32] & (1u << (idx % 32))) != 0; }}
        end
        line %{enum __bits { #{m=0; singular_fields.map { |f| "#{f.name} = #{m+=1}" }.join(", ")} }} unless singular_fields.empty?
      end

      in_namespace("void clear()", "") do
        message.field.each { |f| line %{clear_#{f.name}();} }
      end

      in_namespace("bool mergeFromString(string s)", "") do
        body = <<-BODY
        uint i;
        while(i < s.length) {
          long nextTag = decodeVarint(s[i++]);
          long tag = nextTag >> 3;
          uint wire_type = nextTag & 0b111;
          switch(tag) {
          }
        }
        return true;
        BODY

        body.each_line { |l| line(l.sub("        ", '').chomp) }
      end

      line
    end
  end

  def field_typename(field)
    (TYPE_MAPPING[field.type] || field.type_name).sub(/^.#{module_name}/, '')
  end

  def define_field(field, repeated)
    typename = field_typename(field)
    if repeated
      line "private #{typename}[] _#{field.name};"
      line "#{typename}[] #{field.name}() { return _#{field.name}; }"
    else
      line "private #{typename} _#{field.name};"
      line "#{typename} #{field.name}() { return _#{field.name}; }"
    end

    if !repeated
      line "static const #{typename} default_#{field.name} = #{default_value(field)};"
      line "bool has_#{field.name}() { return _check_bit(__bits.#{field.name}); }"
      line "void #{field.name}(#{typename} val) { _set_bit(__bits.#{field.name}); _#{field.name} = val; }"
    end

    if repeated
      line "void clear_#{field.name}() { _#{field.name} = []; }"
    else
      line "void clear_#{field.name}() { _clear_bit(__bits.#{field.name}); _#{field.name} = default_#{field.name}; }"
    end
    line
  end

  def dump_enum(enum)
    in_namespace("enum", enum.name) do
      enum.value.each do |value|
        line %{#{value.name} = #{value.number},}
      end
    end
  end

  TYPE_MAPPING = {
    TYPE_DOUBLE => "double",
    TYPE_FLOAT => "float",
    TYPE_INT64 => "long",
    TYPE_UINT64 => "ulong",
    TYPE_INT32 => "int",
    TYPE_FIXED64 => "long",
    TYPE_FIXED32 => "int",
    TYPE_BOOL => "bool",
    TYPE_STRING => "string",
    TYPE_BYTES => "ubyte[]",
    TYPE_UINT32 => "uint",
    TYPE_SFIXED32 => "int",
    TYPE_SFIXED64 => "long",
    TYPE_SINT32 => "int",
    TYPE_SINT64 => "long",
  }

  def default_value(field)
    typename = field_typename(field)

    if field.default_value && field.default_value != ""
      case field.type
      when TYPE_STRING, TYPE_BYTES
        %{"#{field.default_value}"}
      when TYPE_BOOL
        field.default_value || "false"
      when TYPE_ENUM
        %{#{typename}.#{field.default_value}}
      else
        field.default_value || "0"
      end
    else
        %{(#{typename}).init}
    end
  end

  def in_namespace(type, name, rest = "")
    if !name
      yield
    else
      line "#{type} #{capfirst(name)}#{rest} {"
      @ns.push name
      yield
      @ns.pop
      line "}"
    end
  end

  def capfirst(s)
    "#{s[0,1].capitalize}#{s[1..-1]}" if s
  end

  def line(str = nil)
    if str
      @ns.size.times { @io.write("  ") }
      @io.write(str)
    end
    @io.write("\n")
  end

end
