<?php defined('BASEPATH') OR exit('No direct script access allowed');

class Alerts extends LS_Controller
{
    function __construct()
    {
        parent::__construct();

        // Make sure that user is authenticated no matter what page they are on
        if (!$this->users->logged_in()){
            $this->users->redirect_to_login();
        }

        if (!user_has_permission('alerts', array('view'))) {
            redirect('');
        }

        $this->page = "alerts";
        $this->load->model('alert');
    }

    public function index()
    {
        $this->init_page(_("Alerting"), $this->page);
        if (!user_has_permission('alerts', array('view'))) {
            redirect('alerts');
        }

        $query = '';
        if ($this->input->post('search')) {
            $search = $this->input->post('search');
            $query = 'name:'.$search.'*~';
            $this->data['search'] = $search;
        }

        // Set username so that users can only manage their own alerts (unless set to all)
        $username = $this->session->userdata('username');
        $this->data['username'] = $username;
        $user_id = $this->session->userdata('user_id');
        $this->data['user_id'] = $user_id;
        $this->data['config_edit_allowed'] = user_has_permission('configure', array('all')) && !is_demo_mode();
        $this->data['contact_edit_allowed'] = user_has_permission('contacts', array('edit'));
        $this->data['alert_edit_allowed'] = user_has_permission('alerts', array('edit'));

        $this->data['alerts'] = $this->alert->get_all($query);

        // Flash message
        $this->data['msg'] = $this->session->flashdata('msg');
        $this->data['msg_type'] = $this->session->flashdata('msg_type');
        $this->data['csrf_name'] = $this->security->get_csrf_token_name();
        $this->data['csrf_value'] = $this->security->get_csrf_hash();

        $this->data['leftbar'] = $this->load->view('alerts/leftbar', array("tab" => "alerts"), true);
        $this->load->view('alerts/home', $this->data);
    }

    public function run($id='')
    {
        if (empty($id)) { redirect('alerts'); }
        if (!user_has_permission('alerts', array('view'))) {
            redirect('alerts');
        }

        // Run the alert and then redirect
        $this->load->model('alert');
        $this->alert->run($id);

        redirect('alerts');
    }

    /**
     * Show alert history
    [16] => Array
        (
            [alert_id] => AVsRF7zcQeQUH0d1mUZ5
            [name] => testsetset
            [check_interval] => 5m
            [lookback_period] => 5m
            [warning] => 0
            [critical] => 0
            [start] => 1491854306
            [end] => 1491854606
            [ran] => 1491854606
            [status] => 2
            [output] => CRITICAL: 50 matching entries found |logs=50;0;0
            [id] => AVtZdsAMD_TzZ4ef4nzK
        )
     */
    public function history()
    {
        $this->load->library('pagination');
        $this->init_page(_("Alert History"), $this->page);
        $this->data['halerts'] = array();
        $query = array();

        if (!user_has_permission('alerts', array('view'))) {
            redirect('alerts');
        }

        $page_size_option = get_username() . '_page_size';
        $page_size = intval(get_option($page_size_option, 25));

        // Check if we are setting pagination settings
        if ($this->input->method() == "post") {
            $pp = $this->input->post('pp');
            if (!empty($pp)) {
                $page_size = intval($pp);
                set_option($page_size_option, $page_size);
            }
        }

        $history = new Elasticsearch(array('index' => 'nagioslogserver_history'));

        $alert_type_map = array();
        
        $deleted_alerts = $this->elasticsearch->query_wresultSize('deleted_alert');
        $this->data['deleted_alerts'] = array();
        if ($deleted_alerts['hits']['total'] > 0) {
            $this->data['deleted_alerts'] = $deleted_alerts['hits']['hits'];
            foreach ($deleted_alerts['hits']['hits'] as $deleted_alert) {
                $alert_type_map[$deleted_alert['name']] = $deleted_alert['alert_type'];
            }
        }

        $alerts = $this->alert->get_all();
        foreach ($alerts as $alert) {
            $alert_type_map[$alert['name']] = $alert['alert_type'];
        }
        $this->data['alerts'] = $alerts;

        // Filters
        $this->data['alert_id'] = $this->input->get('alert');
        if (!$this->data['alert_id']) {
            $this->data['alert_id'] = $this->input->get('deleted_alert');
        }
        if (!empty($this->data['alert_id']) && $this->data['alert_id'] !== '[showing-history-for-all-alerts]') {
            $query[] = '+alert_id:' . $this->data['alert_id'];
        }


        // Set date/time filter
        $this->data['time'] = $this->input->get('time');
        if (empty($this->data['time'])) { $this->data['time'] = '24h'; }
        if ($this->data['time'] == '24h') {
            $last = time() - 24*60*60;
        } else if ($this->data['time'] == '7d') {
            $last = time() - 7*24*60*60;
        } else if ($this->data['time'] == '30d') {
            $last = time() - 30*24*60*60;
        } else {
            $this->data['start'] = $this->input->get('start');
            $this->data['end'] = $this->input->get('end');
            if (!empty($this->data['start'])) {
                $start = strtotime($this->data['start']);
            }
            if (!empty($this->data['end'])) {
                $end = strtotime($this->data['end']);
            }
        }

        // Apply status filter (must check specifically for empty string)
        $this->data['status'] = $this->input->get('status');
        if ($this->data['status'] != "") {
            $query[] = '+status:' . intval($this->data['status']);
        }

        // Apply time filter
        if (!empty($last)) {
            $query[] = '+ran:>=' . $last;
        } else {
            $query[] = '+ran:>=' . $start . ' +ran:<=' . $end;
        }
        $query = implode(' ', $query);

        // Get alert total
        $record_total = 0;
        $from = $this->input->get('per_page');
        $temp = $history->count('alert', $query);
        if (!empty($temp['count'])) {
            $record_total = $temp['count'];
        }
        $total = $record_total;

        // Limit to 10,000 range
        $max_limit = 10000;
        if ($record_total >= $max_limit) {
            $total = $max_limit;
            $this->data['limiting'] = _('Limited to') . ' ' . number_format($max_limit) . ' ' . _('results');
        }

        // Set up pagination
        $pcfg['base_url'] = site_url('alerts/history');
        $pcfg['total_rows'] = $total;
        $pcfg['per_page'] = $page_size;
        $pcfg['num_links'] = 3;
        $pcfg['page_query_string'] = true;
        $pcfg['reuse_query_string'] = true;
        $this->pagination->initialize($pcfg);
        $this->data['pagination'] = $this->pagination->create_links();

        // Apply paging
        $page = 1;
        if ($from != null) {
            $page += ceil($from / $page_size);
        }
        $this->data['total_pages'] = ceil($total / $page_size);
        $this->data['page'] = $page;
        $this->data['page_size'] = $page_size;
        $this->data['record_total'] = $record_total;

        // Run query for alert result set
        $alerts = $history->query_wresultSize('alert', $query, $page_size, $from, 'ran:desc');
        if (!empty($alerts['hits']['total']) && $alerts['hits']['total'] > 0) {
            foreach ($alerts['hits']['hits'] as $hit) {
                $tmp = $hit['_source'];
                $tmp['id'] = $hit['_id'];
                if (!array_key_exists('alert_type', $tmp)) {
                    $tmp['alert_type'] = $alert_type_map[$tmp['name']];
                }
                $this->data['halerts'][] = $tmp;
            }
        }

        $this->data['leftbar'] = $this->load->view('alerts/leftbar', array("tab" => "history"), true);
        $this->load->view('alerts/history', $this->data);
    }

    // Show alert history value
    public function history_show($id='')
    {
        if (empty($id)) { redirect('alerts/history'); }

        $es = new Elasticsearch(array('index' => 'nagioslogserver_history'));
        $alert = $es->get('alert', $id);

        // If no alert exists we should redirect
        if (empty($alert)) {
            redirect('alerts/history');
        }

        $this->load->helper('multitenancy');

        $query = $alert['_source']['query'];
        $query = multitenancy_limit_query($query);

        $indexes = $alert['_source']['indexes'];

        $es = new Elasticsearch(array('index' => $indexes));
        $data = $es->backend_call("_search", 'POST', $query);

        print json_pretty(json_encode($data), true);
    }

    // Export an alert history to CSV based on alert run ID
    public function history_export($id='')
    {
        if (empty($id)) { redirect('alerts/history'); }

        $es = new Elasticsearch(array('index' => 'nagioslogserver_history'));
        $alert = $es->get('alert', $id);

        // If no alert exists we should redirect
        if (empty($alert)) {
            redirect('alerts/history');
        }

        $this->load->helper('multitenancy');
        $query = $alert['_source']['query'];
        $query = multitenancy_limit_query($query);

        $indexes = $alert['_source']['indexes'];

        $es = new Elasticsearch(array('index' => $indexes));
        $data = $es->backend_call("_search", 'POST', $query);

        $fields = array();
        foreach ($data['hits']['hits'] as $hit) {
            foreach ($hit['_source'] as $field => $value) {
                if (!in_array($field, $fields)) {
                    $fields[] = $field;
                }
            }
        }

        // Output header for csv
        header("Content-type: application/octet-stream");
        header("Content-Disposition: attachment; filename=\"export.csv\"");

        // Output CSV format
        print implode(',', $fields) . "\n";
        foreach ($data['hits']['hits'] as $hit) {
            $tmp = array();
            foreach ($fields as $i) {
                if (isset($hit['_source'][$i])) {
                    $tmp[] = '"'.trim(str_replace(array("\r", "\n", "&apos;"), array(" ", " ", "'"), html_entity_decode($hit['_source'][$i]))).'"';
                } else {
                    $tmp[] = '';
                }
            }
            print implode(',', $tmp) . "\n";
        }
    }

    /**
     * Show alert in dashboard
     */
    public function show($id='')
    {
        if (empty($id)) { redirect('alerts'); }

        $s_to = grab_request_var('s_to', "");
        $s_from = grab_request_var('s_from', "");

        // Get the data from ES index based on $id
        $result = $this->elasticsearch->get('alert', $id);

        // Redirect to custom alert info if there is no dashboard query
        if (empty($result['found'])) {
            redirect('alerts');
        } else if (empty($result['_source']['dash_query'])) {
            $url = 'alerts/show_custom/'.$id;
            redirect($url);
        }

        $lookback_period = $result['_source']['lookback_period'];
        $url = 'dashboard#/dashboard/script/logserver.js?from='.urlencode($lookback_period).'&a='.$id;

        if ($s_to != "") {
            $url .= "&s_to=" . $s_to;
        }
        if ($s_from != "") {
            $url .= "&s_from=" . $s_from;
        }

        redirect($url);
    }

    public function show_query($id='')
    {
        if (empty($id)) { redirect('alerts'); }
        redirect('dashboard#/dashboard/script/logserver.js?q='.$id);
    }

    /**
     * Takes an alert ID and gives out the custom query with update start and end date/time
     * @return string JSON formatted query string
     */
    public function show_custom($id='')
    {
        if (empty($id)) { redirect('alerts'); }

        $this->load->helper('alert');
        $this->load->helper('multitenancy');
        $alert = $this->elasticsearch->get('alert', $id);

        if (empty($alert['found'])) {
            redirect('alerts');
        }

        $end = time();
        $json = $alert['_source']['query'];
        $lookback_period = $alert['_source']['lookback_period'];
        $start = $end - get_alert_seconds($lookback_period);

        $json = set_start_end_times($json, $start, $end);
        $json = multitenancy_limit_query($json);
        $indexes = get_logstash_indices_in_range($start, $end);

        // Do the query
        $backend = new Elasticsearch(array('index' => $indexes));
        $data = $backend->backend_call("_search", 'POST', $json);

        print json_pretty(json_encode($data), true);
    }

    /**
     * Expects arguments that have been preprocessed by Backend.php, uses them to regenerate the results of the backend call.
     * Arguments are passed in by POST.
     * @param $index The elasticsearch index we'll be querying
     * @param $request_method The original $_SERVER['REQUEST_METHOD'] that was used for this backend call.
     * @param $path The modified url path generated by Backend.php
     * @param $data A string representing the JSON query we'll be executing.
     */
    public function raw_custom()
    {
        // If a user can edit alerts, they'd be able to run all of this in some other way, so we're not adding any additional risk.
        if (!user_has_permission('alerts', array('edit'))) {
            redirect('alerts');
        }

        $index = grab_request_var('index', '');
        $request_method = grab_request_var('request_method', '');
        $path = grab_request_var('path', '');
        $data = grab_request_var('data', '{}');

        $backend = new Elasticsearch(array('index' => $index));

        $result = $backend->backend_call($path, $request_method, $data);

        if (strpos($index,'logstash') !== false  && get_option('log_user_queries') == 1) {
            $ci =& get_instance();
            $ci->load->helper('auditlog');

            $log = array('type' => 'SECURITY',
                     'message' => _('Re-ran query from Audit Log') . ' ' . show_arguments_link($path, $data),
                     'method' => $_SERVER['REQUEST_METHOD'],
                     'query_index' => "$index",
                     'path' => "$path",
                     'data' => "$data",
                     'source' => 'Backend API',
                     'actions' => json_encode(rerun_query_action($index, $request_method, $path, $data))
                    );
            $this->logger->log($log);
        }

        // json_encode($result, JSON_PRETTY_PRINT) would work, except that we're using <br>s and &nbsp;s here.
        print json_pretty(json_encode($result), true);
    }

    public function deactivate($id='', $real_time=false)
    {
        if (empty($id) || !user_has_permission('alerts', array('edit'))) { redirect('alerts'); } 

        // Deactivate the alert
        $alert = $this->alert->get_by_id($id);
        $data = array("active" => 0);
        $this->elasticsearch->update('alert', $data, $id);
        $this->elasticsearch->refresh();

        $msg = _('Alert deactivated: ') . $alert['name'];
        $log = array('type' => 'ALERT', 'message' => $msg);
        $logged = $this->logger->log($log);

        $apply_now = $this->input->post_get('apply_now');
        $can_apply = user_has_permission('configure', array('all'));

        if ($real_time) {
            if (!$apply_now || !$can_apply) {

                $this->session->set_flashdata('msg', _('Alert deactivated. Monitoring will not be affected until configuration is applied.'));
                $this->session->set_flashdata('msg_type', 'info');
                if ($apply_now) {
                    $this->session->set_flashdata('msg_type', 'danger');
                }
            }
            $this->deactivate_real_time_alert_config($id);
        }


        if ($apply_now && $can_apply) {
            redirect('configure/apply_to_instances');
        }

        redirect('alerts');
    }

    public function activate($id='', $real_time=false)
    {
        if (empty($id) || !user_has_permission('alerts', array('edit'))) { redirect('alerts'); }

        // Activate the alert
        $alert = $this->alert->get_by_id($id);
        $data = array("active" => 1);
        $this->elasticsearch->update('alert', $data, $id);
        $this->elasticsearch->refresh();

        $msg = _('Alert activated: ') . $alert['name'];
        $log = array('type' => 'ALERT', 'message' => $msg);
        $logged = $this->logger->log($log);

        $apply_now = $this->input->post_get('apply_now');
        $can_apply = user_has_permission('configure', array('all'));

        if ($real_time) {
            if (!$apply_now || !$can_apply) {

                $this->session->set_flashdata('msg', _('Alert activated. Monitoring will not be affected until configuration is applied.'));
                $this->session->set_flashdata('msg_type', 'info');
                if ($apply_now) {
                    $this->session->set_flashdata('msg_type', 'danger');
                }
            }
            $this->activate_real_time_alert_config($id);
        }

        if ($apply_now && $can_apply) {
            redirect('configure/apply_to_instances');
        }

        redirect('alerts');
    }

    private function activate_real_time_alert_config($alert_id) {

        $this->load->model('ls_configure');

        $alert = $this->elasticsearch->get('alert', $alert_id);
        $alert = $alert['_source'];

        $alert['criteria_display'] = $this->ls_configure->build_conditional_from_arrays($alert['criteria_fields'], $alert['criteria_comparators'], $alert['criteria_values'], $alert['boolean_ops']);

        $this->ls_configure->upsert_real_time_alert($alert, $alert['name']);

        // Apply Config happens after this post request, so this might get changed back (almost) immediately.
        set_option('configuration_required', 1);
    }

    private function deactivate_real_time_alert_config($alert_id) {
        $this->load->model('ls_configure');

        $alert = $this->elasticsearch->get('alert', $alert_id);

        $this->ls_configure->delete_real_time_alert($alert['_source']['name']);

        // Apply Config happens after this post request, so this might get changed back (almost) immediately.
        set_option('configuration_required', 1);
    }

    // Displays the NRDP Servers page, allowing users to manage their NRDP servers
    // which includes Nagios XI and Nagios Core
    public function nrdp()
    {
        $this->init_page(_("Nagios Servers (NRDP)"), $this->page);
        if (!user_has_permission('alerts', array('view'))) {
            redirect('alerts');
        }

        // Get a list of all alerts that have host/service associations
        $saved = array();
        $alerts = $this->alert->get_all();
        foreach ($alerts as $alert) {
            if ($alert['method']['type'] == "nrdp") {
                $server = $this->elasticsearch->get('nrdp_server', $alert['method']['server_id']);
                $alert['method']['server_name'] = $server['_source']['name'];
                $saved[] = $alert;
            }
        }
        $this->data['alerts'] = $saved;

        $this->data['leftbar'] = $this->load->view('alerts/leftbar', array("tab" => "nrdp"), true);
        $this->load->view('alerts/nrdp', $this->data);
    }

    // Displays SNMP trap receivers management page
    public function snmp()
    {
        $this->init_page(_("SNMP Trap Receivers"), $this->page);
        if (!user_has_permission('alerts', array('view'))) {
            redirect('alerts');
        }

        $this->data['leftbar'] = $this->load->view('alerts/leftbar', array("tab" => "snmp"), true);
        $this->load->view('alerts/snmp', $this->data);
    }

    // Display the linking of Nagios Reactor servers
    public function reactor()
    {
        $this->init_page(_("Nagios Reactor Servers"), $this->page);
        if (!user_has_permission('alerts', array('view')) || !show_nagios_reactor()) {
            redirect('alerts');
        }

        $this->data['leftbar'] = $this->load->view('alerts/leftbar', array("tab" => "reactor"), true);
        $this->load->view('alerts/reactor', $this->data);
    }

    public function delete($id='')
    {
        if (empty($id) || !user_has_permission('alerts', array('delete'))) { redirect('alerts'); }

        if (is_demo_mode()) {
            $this->session->set_flashdata('msg', _("This function is not available in demo mode."));
            $this->session->set_flashdata('msg_type', 'danger');
            redirect('alerts');
        }

        $alert = $this->alert->get_by_id($id);
        if (array_key_exists('found', $alert) && $alert['found'] === false) {
            // Alert was already deleted?
            redirect('alerts');
        }

        $is_real_time = $this->input->post_get('real_time');
        $apply_now = $this->input->post_get('apply_now');
        $can_apply = user_has_permission('configure', array('all'));

        if ($is_real_time) {
            if (!$apply_now || !$can_apply) {

                $this->session->set_flashdata('msg', _('Alert deleted. Monitoring will not be affected until configuration is applied.'));
                $this->session->set_flashdata('msg_type', 'info');
                if ($apply_now) {
                    $this->session->set_flashdata('msg_type', 'danger');
                }
            }
            // Real-time alerts need to have their filters/outputs removed.
            $this->load->model('ls_configure');
            set_option('configuration_required', 1);
            $this->ls_configure->delete_real_time_alert($alert['name']);
        }

        // A record of deleted alert names are needed for the Alert History page
        $this->elasticsearch->add('deleted_alert', array('id' => $id, 'name' => $alert['name'], 'alert_type' => $alert['alert_type']));

        // Delete the check from the database
        $result = $this->elasticsearch->delete('alert', $id);
        $this->elasticsearch->refresh();
        $msg = _('Alert deleted: ') . $alert['name'];
        $log = array('type' => 'ALERT', 'message' => $msg);
        $logged = $this->logger->log($log);

        if ($apply_now && $can_apply) {
            redirect('configure/apply_to_instances');
        }

        redirect('alerts');
    }

    //
    // Alert Email Templates
    //

    public function templates()
    {
        $this->init_page(_('Email Templates'), $this->page);
        if (!user_has_permission('alerts', array('all'))) {
            redirect('alerts');
        }

        // Grab the current default email template
        $dtpl = get_default_email_tpl();
        $this->data['default_template'] = $dtpl['name'];
        $this->data['default_template_subject'] = $dtpl['subject'];
        $this->data['default_template_body'] = $dtpl['body'];

        $this->data['success'] = $this->session->flashdata('msg');
        $this->data['leftbar'] = $this->load->view('alerts/leftbar', array("tab" => "templates"), true);
        $this->load->view('alerts/templates', $this->data);
    }

}
