<?php  if ( ! defined('BASEPATH')) exit('No direct script access allowed');

class Dashboard extends LS_Controller
{
    function __construct()
    {
        parent::__construct();

        // Make sure that user is authenticated no matter what page they are on
        require_install();
        if (!$this->users->logged_in()) { $this->users->redirect_to_login(); }
    }

    // Logs out user
    function logout()
    {
        // Log the user out
        $logout = $this->users->logout();

        // Redirect them to the login page
        $this->session->set_flashdata('message', $this->users->message);
        redirect('login');
    }

    // Basic dashboard
    public function index()
    {
        $this->init_page(_("Home"), 'home', array('/js/highcharts/highcharts.js', '/js/highcharts/modules/no-data-to-display.js'), array(), 'dash');

        // Fix key value timezone offset to match server time
        $x = date_default_timezone_get();
        if ($x) {
            $dt = new DateTime("now", new DateTimeZone($x));
            $offset = $dt->getOffset()*1000;
            if (isset($this->data['hc_count_data'])) {
                foreach ($this->data['hc_count_data'] as &$obj) {
                    $obj['key'] = $obj['key'] + $offset;
                }
            }
        }

        // Get the amount of unique hosts that sent data
        $start = time()-(24*60*60);
        $end = time();
        $this->data['unique_hosts'] = count(get_hosts($start, $end));

        // get number of nodes in cluster
        $es = new Elasticsearch(array('index' => '/'));
        $data = $es->backend_call('_cluster/health?human');
        $this->data['nodes'] = $data['number_of_nodes'];

        // Get current number of alerts defined
        $es = new Elasticsearch(array('index' => 'nagioslogserver/alert'));
        $counts = $es->backend_call('_search?search_type=count&q=active:1', 'GET');
        $this->data['alerts'] = $counts['hits']['total'];
        $this->load->view('home', $this->data);
    }

    public function show_dash()
    {
        $css = array('/css/dashboard.css', '/../css/animate.min.css', '/../css/timepicker.css');
        $this->init_page(_("Dashboard"), 'dashboard', array(), $css);

        $this->data['user'] = $this->users->get_user();

        $this->load->view('dashboard', $this->data);
    }
    
    public function dashlet()
    {
        $this->data = array();
        $this->load->view('dashlet/dashlet', $this->data);
    }
    
    public function do_update_check()
    {
        $this->init_page();

        // Do an update check
        if (is_admin()) {
            $xml = do_update_check();
        }
        $update_check = simplexml_load_string(base64_decode($xml));

        $html = '<strong>';
        if (isset($update_check->update_available) && intval($update_check->update_available)) {
            $html .= '<img src="media/icons/exclamation.png" style="vertical-align: text-top;"> ' . _("New version available!") . ' <a href="https://library.nagios.com/library/products/nagios-log-server/downloads">' . _("Download") . ' <i class="icon-share"></i></a>';
        } else {
            $html .= '<img src="media/icons/accept.png" style="vertical-align: text-top;"> ' . _("You're running the latest version of Nagios Log Server");
        }
        if (isset($update_check->update_available) && intval($update_check->update_available)) {
            $html .= _(" Version ") . strval($update_check->update_version->version) . _(" is the latest version available");
        }
        $html .= '.</strong>';

        print $html;
    }

    // Basic user profile (can't do much here - requires auth to change certain things)
    // - for api key mostly
    public function profile()
    {
        $this->init_page(_("My Profile"), 'profile');
        $this->data['csrf'] = array(
            'name' => $this->security->get_csrf_token_name(),
            'hash' => $this->security->get_csrf_hash()
        );

        $this->data['user'] = $this->users->get_user();
        $this->data['languages'] = get_languages();

        $this->load->view('profile', $this->data);
    }

    public function newkey()
    {
        $redirect_to_help = $this->input->post('help', FALSE);
        
        if (!$this->data['demo_mode']) {
            // Generate a new API key
            $newkey = sha1(uniqid() . 'nagioslogserver');

            // Update new key in database
            $user_id = $this->session->userdata('user_id');
            $verify_user_id = $this->input->post('user_id_verify', TRUE);

            if ($user_id == $verify_user_id) {
                // Update the new api key
                $data = array("apikey" => $newkey);
                $this->users->update_user($user_id, $data);
            }

            // Set session data
            $this->session->set_userdata('apikey', $newkey);
        }

        if ($redirect_to_help) {
            redirect('help/api');
        }
        redirect('profile');
    }

    public function setlanguage()
    {
        $user_id = $this->session->userdata('user_id');
        $language = $this->input->post('language');

        // Update db
        $data = array("language" => $language);
		$this->users->update_user($user_id, $data);

        // Update session
        $this->session->set_userdata('language', $language);
    }

    public function export()
    {
        $query = $this->input->get_post('q');
        $limit = $this->input->get_post('limit');
        $fields = $this->input->get_post('fields');
        $fields = explode(',', $fields);

        $csv_use_local_timezone = get_option("csv_use_local_timezone");
        $timezone = new DateTime('now');
        $timezone = $timezone->getTimezone();

        if (empty($query) || empty($fields)) {
            print _("Cannot make CSV with the query and fields given.");
            print "<p>q = $query</p>";
            print "<p>fields = ".print_r($fields, true)."</p>";
            die();
        }

        $this->load->helper('multitenancy');
        $query = str_replace(array(" and ", " or "),array(" AND ", " OR "), $query);
        $query = multitenancy_limit_query($query);

        // Do the query
        $backend = new Elasticsearch(array('index' => '/'));
        $data = $backend->backend_call("_search", 'GET', $query);

        // Output header for csv
        header("Content-type: application/octet-stream");
        header("Content-Disposition: attachment; filename=\"export.csv\"");

        // Output CSV format
        print implode(',', $fields) . "\n";
        foreach ($data['hits']['hits'] as $hit) {
            $tmp = array();
            foreach ($fields as $i) {

                $field = "";
                // Check for a nested field - usually geoip related.
                if (strpos($i, '.') === false) {
                    
                    if (@isset($hit['_source'][$i])) {
                        $field = $hit['_source'][$i];
                    } else {
                        $field = $hit[$i];
                    }
                }
                else {
                    // Field is nested - recursively navigate to the actual value
                    $keys = explode('.', $i);
                    $field = $hit['_source'];
                    for ($j = 0; $j < count($keys); $j++) {
                        if (@isset($field[$keys[$j]])) {
                            $field = $field[$keys[$j]];
                        }
                    }
                }

                // This converts the timestamp from UTC to the system timezone (if the global option is set)
                if ($csv_use_local_timezone && $i === "@timestamp") {
                    $as_datetime = new DateTime($field);
                    $as_datetime->setTimezone($timezone);
                    $field = $as_datetime->format('c');
                }

                $tmp[] = '"'.trim(str_replace(array("\r", "\n", "&apos;", "\""), array(" ", " ", "'", "\"\""), html_entity_decode($field))).'"';
            }
            print implode(',', $tmp) . "\n";
        }
    }

    public function get_natural_language_query_status() {
        $natural_language_query = get_option("natural_language_query");
        echo json_encode(['value' => $natural_language_query]);
    }

    public function natural_language_to_query() {
        $input = isset($_GET['input']) ? $_GET['input'] : '';
        $input = escapeshellarg($input);
        $current_fields = isset($_GET['current_fields']) ? $_GET['current_fields'] : '';
        $current_fields = escapeshellarg($current_fields);
    
        $api_key = escapeshellarg(trim(get_option("ai_api_key")));
        $model = escapeshellarg(trim(get_option("ai_provider")));
        $self_host_ip_address = escapeshellarg(trim(get_option("self_host_ip_address")));
        $self_host_port = escapeshellarg(trim(get_option("ai_port")));
        
        $script =  '/usr/local/nagioslogserver/scripts/generate_log_query.py';
    
        if ($model == "self_hosted") {
            $command = "python3.9 $script --model \"$model\" --provider_address \"$self_host_ip_address\" --provider_port \"$self_host_port\" --natural_language_query \"$input\" --current_fields \"$current_fields\"";
        } else {
            $command = "python3.9 $script --api_key \"$api_key\" --model \"$model\" --natural_language_query \"$input\" --current_fields \"$current_fields\"";
        }
    
        $output = shell_exec($command);
        
        $lucene_query = trim($output);

        $es = new Elasticsearch(array('index' => '/'));
        $query = '{
            "query": {
                "query_string": {
                    "query": "' . $lucene_query . '"
                }
            }
        }';

        $result = $es->backend_call('_validate/query', 'GET', $query);
        $is_valid = isset($result['valid']) ? $result['valid'] : false;
        // holy shit this works https://www.elastic.co/guide/en/elasticsearch/reference/current/search-validate.html
        header('Content-Type: application/json');
        echo json_encode([
            'query' => $lucene_query,
            'is_valid' => $is_valid
        ]);
    }
}
