<?php

/* DO NOT REMOVE THIS LINE! sourceguardian-check */

/**
*  functions that can be called from cmdsubsys
**/

/************************************************
    CMDSUBS FUNCTIONS 
*************************************************/ 

/**
 * Run an update check in the background
 */
function update_check($args=array())
{
    // Run an update check (forced)
    do_update_check(false);

    // Run a forced maintenance check (forced)
    do_check_maintenance(true);
}

/**
 * Start logstash or elasticsearch
 **/
function start_service($args=array())
{
    $ci =& get_instance();
    $ci->load->model('Systemstat');
    $subsystem = $args[0];

    $log = array('type' => 'JOBS',
                 'message' => $subsystem . _(' is being started'),
                 'node' => NODE
                );
    $logged = $ci->logger->log($log);

    return json_encode($ci->Systemstat->start($subsystem));
}

/**
 * Stop logstash or elasticsearch
 **/
function stop_service($args=array())
{
    $ci =& get_instance();
    $ci->load->model('Systemstat');
    $subsystem = $args[0];

    $log = array('type' => 'JOBS',
                 'message' => $subsystem . _(' is being stopped'),
                 'node' => NODE
                );
    $logged = $ci->logger->log($log);

    return json_encode($ci->Systemstat->stop($subsystem));
}

/**
 * Retart logstash or elasticsearch
 **/
function restart_service($args=array())
{
    $ci =& get_instance();
    $ci->load->model('Systemstat');
    $subsystem = $args[0];

    $log = array('type' => 'JOBS',
                 'message' => $subsystem . _(' is being restarted'),
                 'node' => NODE
                );
    $logged = $ci->logger->log($log);

    return json_encode($ci->Systemstat->restart($subsystem));
}

/************************************************
    CLEANUP FUNCTIONS
*************************************************/ 

/**
 * Clean up the cmdsubsys commands
 */
function cleanup()
{
    $ci =& get_instance();
    $time = time() - 24*60*60; // Keep the 
    $result = $ci->elasticsearch->query('commands', "status:completed AND run_time:<$time");
    if ($result['hits']['total'] > 0) {
        $commands = $result['hits']['hits'];
        foreach ($commands as $cmd) {
            $ci->elasticsearch->delete('commands', $cmd['_id']);
        }
        $log = array('type' => 'JOBS',
                     'message' => count($commands) . " " . _('jobs removed that were completed jobs more than 24 hours old')
        );
        $logged = $ci->logger->log($log);
    }
}

/************************************************
    BACKUP / MAINTENANCE FUNCTIONS 
*************************************************/ 

/**
 * Run maintance jobs
 **/
function do_maintenance($args=array())
{
    $ci =& get_instance();

    $maintenance_settings = unserialize(get_option('maintenance_settings', array()));

    if(empty($maintenance_settings)){
        set_option('maintenance_settings', serialize(array('active', 1,
                                                           'optimize_time' => 2, 
                                                           'bloom_time' => 1, 
                                                           'close_time' => 30,
                                                           'delete_time' => 0,
                                                           'audit_retention' => 0,
                                                           'alert_retention' => 0,
                                                           )));
        $maintenance_settings = get_option('maintenance_settings');
    }

    if(!$maintenance_settings['active'])
        return _('Maintenance and Backup jobs are disabled');

    # Trim audit log
    if ($maintenance_settings['audit_retention'])
        do_maintenance_trim_audit_index($maintenance_settings['audit_retention']);

    # Trim alert history
    if ($maintenance_settings['alert_retention'])
        do_maintenance_trim_alert_index($maintenance_settings['alert_retention']);

    # Optimize indexes
    if ($maintenance_settings['optimize_time'])
        do_maintenance_optimize($maintenance_settings['optimize_time']);

        # Close all indexes within timeframe
    if ($maintenance_settings['close_time'])
        do_maintenance_close($maintenance_settings['close_time']);

    # Delete old indexes
    if ($maintenance_settings['delete_time'])
        do_maintenance_delete($maintenance_settings['delete_time']);

    # Give elasticsearch enough time to index if necessary
    sleep(1);

    # Create new snapshot in repository
    if (!empty($maintenance_settings['repository']))
        do_maintenance_create_snapshot($maintenance_settings['repository']);

    # Delete old snapshots if there are any that are past expiration date
    if ($maintenance_settings['delete_snapshot_time'] && !empty($maintenance_settings['repository'])) {
        do_maintenance_delete_snapshot($maintenance_settings['delete_snapshot_time'], $maintenance_settings['repository']);
    }

    return _('Maintenance and Backup jobs are being executed');
}

/**
 * Disables Bloom Filter Cache on indexes older than $bloom_time
 **/
function do_maintenance_bloom($bloom_time)
{
    $ci =& get_instance();
    
    $cmd = $ci->config->item('scripts_dir') . "/curator.sh bloom indices --older-than ".$bloom_time." --time-unit days --timestring %Y.%m.%d";
    exec($cmd . "");

    // Print output to log
    echo "-----\n";
    echo "Running cmd: $cmd\n";
    echo "Return: $return_var\n";
    echo implode("\n", $output);
    echo "\n-----\n";

    $msg = _('Disabling bloom filter cache on indexes ') . $bloom_time . _(' day(s) old.');

    $log = array('type' => 'MAINTENANCE',
                 'message' => $msg
    );
    $logged = $ci->logger->log($log);

    return $msg;
}

function do_maintenance_trim_audit_index($trim_time)
{
    $ci =& get_instance();
    $trim_date = (time() - (60 * 60 * 24 * $trim_time));
    $trim_date = $trim_date * 1000;
    $command = $queries[$mode] . $trim_date;
    $cmd = "http://localhost:9200/nagioslogserver_log/_query?q=created:<=".$trim_date;

    $ch = curl_init($cmd);
    curl_setopt($ch, CURLOPT_CUSTOMREQUEST, 'DELETE');
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    $response = curl_exec($ch);
    curl_close($ch);

    $msg = _('Trimmed Audit log index ') . $trim_time . _(' day(s) old.');
    $log = array('type' => 'MAINTENANCE',
                 'message' => $msg
    );
    $logged = $ci->logger->log($log);

    return $msg;
}

function do_maintenance_trim_alert_index($trim_time)
{
    $ci =& get_instance();
    $trim_date = (time() - (60 * 60 * 24 * $trim_time));
    $command = $queries[$mode] . $trim_date;
    $cmd = "http://localhost:9200/nagioslogserver_history/_query?q=ran:<=".$trim_time;

    $ch = curl_init($cmd);
    curl_setopt($ch, CURLOPT_CUSTOMREQUEST, 'DELETE');
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    $response = curl_exec($ch);
    curl_close($ch);

    $msg = _('Trimmed alert history index ') . $trim_time . _(' day(s) old.');
    $log = array('type' => 'MAINTENANCE',
                 'message' => $msg
    );
    $logged = $ci->logger->log($log);

    return $msg;
}

/**
 * Optimizing indexes older than $optimize_time
 **/

function do_maintenance_optimize($optimize_time)
{
    $ci =& get_instance();
    $backend_dir = $ci->config->item('backend_dir');

    $cmd = $ci->config->item('scripts_dir') . "/curator.sh optimize indices --older-than ".$optimize_time." --time-unit days --timestring %Y.%m.%d";
    exec($cmd , $output, $return_var);

    // Print output to log
    echo "-----\n";
    echo "Running cmd: $cmd\n";
    echo "Return: $return_var\n";
    echo implode("\n", $output);
    echo "\n-----\n";

    $msg = _('Optimizing indexes ') . $optimize_time . _(' day(s) old.');

    $log = array('type' => 'MAINTENANCE',
                 'message' => $msg
    );
    $logged = $ci->logger->log($log);

    return $msg;
}

/**
 * Closing indexes older than $close_time
 **/
function do_maintenance_close($close_time)
{
    $ci =& get_instance();
    $backend_dir = $ci->config->item('backend_dir');

    $cmd = $ci->config->item('scripts_dir') . "/curator.sh close indices --older-than ".$close_time." --time-unit days --timestring %Y.%m.%d";
    exec($cmd , $output, $return_var);

    // Print output to log
    echo "-----\n";
    echo "Running cmd: $cmd\n";
    echo "Return: $return_var\n";
    echo implode("\n", $output);
    echo "\n-----\n";

    $msg = _('Closing indexes ') . $close_time . _(' day(s) old.');

    $log = array('type' => 'MAINTENANCE',
                 'message' => $msg
    );
    $logged = $ci->logger->log($log);

    return $msg;
}

/**
 * Delete indexes older than $delete_time
 **/
function do_maintenance_delete($delete_time)
{
    $ci =& get_instance();
    $backend_dir = $ci->config->item('backend_dir');

    $cmd = $ci->config->item('scripts_dir') . "/curator.sh delete indices --older-than ".$delete_time." --time-unit days --timestring %Y.%m.%d";
    exec($cmd , $output, $return_var);

    // Print output to log
    echo "-----\n";
    echo "Running cmd: $cmd\n";
    echo "Return: $return_var\n";
    echo implode("\n", $output);
    echo "\n-----\n";

    $msg = _('Deleting indexes more than ') . $delete_time . _(' day(s) old.');
    $log = array('type' => 'MAINTENANCE',
                 'message' => $msg
    );
    $logged = $ci->logger->log($log);

    return $msg;
}

/**
 * Delete snapshots more than than $delete_snapshot_time
 **/
function do_maintenance_delete_snapshot($delete_snapshot_time, $repository)
{
    $ci =& get_instance();
    $backend_dir = $ci->config->item('backend_dir');

    $cmd = $ci->config->item('scripts_dir') . "/curator.sh delete snapshots --older-than " . $delete_snapshot_time . ' --time-unit days --timestring %Y%m%d --repository "' . $repository . '"';
    exec($cmd , $output, $return_var);

    // Print output to log
    echo "-----\n";
    echo "Running cmd: $cmd\n";
    echo "Return: $return_var\n";
    echo implode("\n", $output);
    echo "\n-----\n";

    $msg = _('Deleting snapshots more than ') . $delete_snapshot_time . _(' day(s) old from ' . $repository);
    $log = array('type' => 'MAINTENANCE',
                 'message' => $msg
    );
    $logged = $ci->logger->log($log);

    return $msg;
}

/**
 * Create snapshots in $repository
 **/
function do_maintenance_create_snapshot($repository)
{
    $ci =& get_instance();
    $backend_dir = $ci->config->item('backend_dir');

    $cmd = $ci->config->item('scripts_dir') . '/curator.sh snapshot --repository "' . $repository . '" --ignore_unavailable indices --older-than 1 --time-unit days --timestring %Y.%m.%d';
    exec($cmd , $output, $return_var);

    // Print output to log
    echo "-----\n";
    echo "Running cmd: $cmd\n";
    echo "Return: $return_var\n";
    echo implode("\n", $output);
    echo "\n-----\n";

    $msg = _('Creating snapshots for indexes more than 1 day old in the repository ' . $repository);

    $log = array('type' => 'BACKUP',
                 'message' => $msg
    );
    $logged = $ci->logger->log($log);

    return $msg;
}

/************************************************
    CONFIGURATION FUNCTIONS
*************************************************/

/**
* Apply the configuration to the node...
**/
function apply_config($args=array())
{
    $ci =& get_instance();
    $ci->load->model('ls_configure');

    // Create a new snapshot before we apply
    if (!empty($args)) {
        $ci->ls_configure->create_snapshot($args['sh_id'], $args['sh_created'], "applyconfig.snapshot");
    }

    // Apply the configuration
    $result = $ci->ls_configure->apply(NODE);

    if ($result){
        $return = "success";
        $message = _("New configuration was applied.");
    } else {
        $return = "failed";
        $message = _("New configuration failed to apply.");
    }

    $log = array('type' => 'CONFIG',
                 'message' => $message,
                 'node' => NODE
    );
    $logged = $ci->logger->log($log);

    return $return;
}

/**
 * Create a snapshot on the node
 **/
function create_snapshot($args=array())
{
    $ci =& get_instance();
    $ci->load->model('ls_configure');
    $success = $ci->ls_configure->create_snapshot($args['id'], $args['created']);
    
    $log = array('type' => 'CONFIG',
                 'message' => _("New configuration snapshot was created."),
                 'node' => NODE
    );
    if (!$success) {
        $log['message'] = _("Error: New configuration snapshot could not be created.");
    }
    $logged = $ci->logger->log($log);
}

/**
 * Deletes a snapshot
 */
function delete_snapshot($args=array())
{
    $ci =& get_instance();
    $ci->load->model('ls_configure');
    $ci->ls_configure->delete_snapshot($args['path']);
    
    $log = array('type' => 'CONFIG',
                 'message' => _("Configuration snapshot was deleted from ". $args['path']),
                 'node' => NODE
    );
    $logged = $ci->logger->log($log);
}

/**
 * Restores a snapshot on the node
 **/
function restore_snapshot($args=array())
{
    $ci =& get_instance();
    $ci->load->model('ls_configure');
    $ci->ls_configure->restore_snapshot($args['id']);
    
    $log = array('type' => 'CONFIG',
                 'message' => _("Configuration snapshot was restored from "). $args['id'],
                 'node' => NODE
    );
    $logged = $ci->logger->log($log);
}

/************************************************
    SYSTEM BACKUPS
*************************************************/

function do_backups($args=array())
{
    $ci =& get_instance();
    $ci->load->model('cmdsubsys');
    $backup_rotation = get_option('backup_rotation');

    if ($backup_rotation > 0) {
        $result = $ci->elasticsearch->query('node', array("size" => 2000));
        foreach ($result['hits']['hits'] as $node) {
            if ($node['_id'] == "global") { continue; }
            $data = array("command" => 'create_backup', "node" => $node['_id'], "args" => '');
            $job = $ci->cmdsubsys->create($data);
        }
    }

    $log = array('type' => 'JOBS',
                 'message' => _("Created 'create_backup' jobs for all nodes.")
    );
    $logged = $ci->logger->log($log);
}

// Ran per-node to create a new backup on the specified node
function create_backup($args=array())
{
    $ci =& get_instance();
    $scripts_dir = $ci->config->item('scripts_dir');
    $ci->load->model('cmdsubsys');

    // Run the create backup script...
    $cmd = $scripts_dir.'/create_backup.sh > /tmp/backups.log';
    exec($cmd, $output, $return_var);

    // Print output to log
    echo "-----\n";
    echo "Running cmd: $cmd\n";
    echo "Return: $return_var\n";
    echo implode("\n", $output);
    echo "\n-----\n";

    // Remove any backups that need to be trimmed
    $backup_rotation = get_option('backup_rotation');
    $backup_dir = '/store/backups/nagioslogserver';
    $count = file_count($backup_dir);
    if ($count > $backup_rotation) {
        while ($count > $backup_rotation) {
            remove_oldest_file($backup_dir);
            $count--;
        }
    }

    // Send out logs to the audit log
    $log = array('type' => 'JOBS',
                 'message' => _("Created LS backup."),
                 'node' => NODE
    );
    if ($return_var > 0) {
        $log['message'] = _("Error creating LS backup. Check permissions of backup directory /store/backups/nagioslogserver and disk space.");
    }
    $logged = $ci->logger->log($log);

    // If we failed, update the global command
    if ($return_var > 0) {
        $job = $ci->cmdsubsys->get('backups');
        $job->update(array('last_run_status' => _("FAILED: Check audit log for JOBS")));
    }
}

// Removes the oldest file in the currently specified directory
function remove_oldest_file($dir)
{
    $oldest_file = '';
    $oldest = time();
    foreach(glob($dir . "/*.tar.gz") as $entry) {
        if (filemtime($entry) < $oldest) {
            $oldest_file = $entry;
            $oldest = filemtime($entry);
        }
    }
    unlink($oldest_file);

    return true;
}

function file_count($dir)
{
    $arr = glob($dir . "/*.tar.gz");
    $count = count($arr);
    return $count;
}

/************************************************
    ALERTING FUNCTIONS
*************************************************/

/**
 * Runs all the alerts that need to be ran
 **/
function run_alerts($args=array())
{
    $ci =& get_instance();
    $ci->load->helper('alert');
    $ci->load->model('alert');
    $alerts = $ci->alert->get_all();

    // Loop through the alerts and actually run them if necessary
    $ci->elasticsearch->refresh();
    foreach ($alerts as $alert) {
        if (verify_alert_run($alert['id'])) {
            $ci->alert->run($alert['id']);
        }
    }
}

/************************************************
    REPORTING FUNCTIONS
*************************************************/

function clear_old_scheduled_report($args=array())
{
    $ci =& get_instance();
    $ci->load->model('ScheduledReport');
    $ci->ScheduledReport->clear_old_cron_job($args['sr_id']);
}

function do_report_generation($args=array())
{
    exec($args['cmd']);
}

/************************************************
    HOMEPAGE FUNCTIONS
*************************************************/

/**
 * Gets index usage statistics for homepage graph
 **/
function run_index_usage($args = array())
{
    // Create an elasticsearch model for 'nagioslogserver', where we'll store our data.
    $main_index = new Elasticsearch();

    // Create an elasticsearch model for the current day
    $current_day = new Elasticsearch(array('index' => 'logstash-'.gmdate('Y.m.d')));

    // Retrieve _stats for the current day - contains the total usage of the index.
    $current_stats = $current_day->get('_stats');
    $current_total = $current_stats['_all']['total']['store']['size_in_bytes'];
    $time_unix = time();

    // Retrieve the most recent recorded value
    $most_recent_value_query = array(
        'query' => array('match_all' => array()),
        'size' => '1',
        'sort' => array(
            array('time_unix' => array('order' => 'desc'))
    ));

    $most_recent_value_data = $main_index->advancedquery('index_usage', $most_recent_value_query);
    
    // This value should only be used for calculating the very first entry.
    $most_recent_value = array('total' => 0, 'delta' => 0, 'created' => date('c', 0), 'time_delta' => 0, 'time_unix' => 0);
    if (!empty($most_recent_value_data['hits']['hits'])) {
        $most_recent_value = $most_recent_value_data['hits']['hits'][0]['_source'];
    }
    $current_delta = $current_total - $most_recent_value['total'];

    $time_delta = $time_unix - (array_key_exists('time_unix', $most_recent_value) ? $most_recent_value['time_unix'] : 0);

    // Logstash stores ISO time, so we should be consistent with that to make frontend stuff easier
    $current_time = date('c', $time_unix);

    // Record the total and the delta, store in database.
    $insert_data = array(
        'created' => $current_time,
        'time_unix' => $time_unix,
        'total' => $current_total,
        'delta' => $current_delta,
        'time_delta' => $time_delta
    );
    $main_index->add('index_usage', $insert_data);
}

/************************************************
    SYSTEM FUNCTIONS
*************************************************/

/**
 * Runs the change_timezone.sh script
 **/
function change_timezone($args=array())
{
    $ci =& get_instance();
    $ci->load->model('Systemstat');
    $scripts_dir = $ci->config->item('scripts_dir');

    $cmd = "sudo $scripts_dir/change_timezone.sh -z '".$args['timezone']."'";
    exec(escapeshellcmd($cmd), $o, $r);

    // Restart NCPA since it reports timezone.
    $ci->Systemstat->restart('ncpa_listener');

    return true;
}
