<?php

// Turns a query ($data from the Backend API) into one which is restricted by a user's multitenancy settings
// If the user is not logged in, call get_user() beforehand and pass that array into the function.
function multitenancy_limit_query($query, $user = false)
{
    $ci =& get_instance();

    if ($user === false) {
        $ci->load->model('Users');
        $user = $ci->users->get_user();
    }

    $hosts = isset($user['hosts']) ? $user['hosts'] : array();
    $host_lists = isset($user['host_lists']) ? $user['host_lists'] : array();

    if ($query) {
        // Note: decode to an object because ES queries might include objects like '{ "0": "some_string" }',
        // which get re-encoded to '["some_string"]' unless we force ALL arrays to objects.
        $query_as_object = json_decode($query);
    }

    $ci->load->model('Host_list');
    $all_hosts = $hosts;

    foreach ($host_lists as $hl_id) {
        $host_list = $ci->Host_list->get_by_id($hl_id);
        if ($host_list === false || $host_list === null) {
            continue;
        }
        $all_hosts = array_merge($all_hosts, $host_list['hosts']);
    }
    foreach ($all_hosts as $idx => $host) {
        if (empty($host)) {
            unset($all_hosts[$idx]);
        }
    }
    if (count($all_hosts) == 0) {
        // No hosts/host lists defined for the user, don't limit the query.
        return $query;
    }

    $filter = multitenancy_generate_filter($all_hosts);

    if (empty($query_as_object) && is_object($query_as_object)) {
        return array('query' => array('filtered' => $filter));
    }

    $query_keys = multitenancy_key_bfs($query_as_object, 'query');

    if ($query_keys === false) {
        // 'query' is not in the raw post data, so don't change anything
        return $query;
    }

    $query_location =& $query_as_object;
    for ($i = 0; $i < count($query_keys); $i++) {
        if (is_object($query_location)) {
            $query_location =& $query_location->$query_keys[$i];
        }
        else {
            $query_location =& $query_location[$query_keys[$i]];
        }
    }

    if (is_object($query_location)) {
        $query_location->query = array('filtered' => array('query' => $query_location->query, 'filter' => $filter['filter']));
    }
    else {
        $query_location['query'] = array('filtered' => array('query' => $query_location['query'], 'filter' => $filter['filter']));
    }
    return json_encode($query_as_object);
}

/* Creates a filter that returns elements whose hosts are in the list $hosts.
 * i.e. if $hosts is array('192.168.0.1', '192.168.0.2'), it returns
 * {
 *    'filter': {
 *        'or': {
 *            'filters': [
 *            {
 *                "term": {
 *                    'host': '192.168.0.1'
 *                }
 *            },{
 *                "term": {
 *                    'host': '192.168.0.2'
 *                }
 *            }],
 *            '_cache': true
 *        }
 *    }
 * }
 */
function multitenancy_generate_filter($hosts)
{
    $filter = array('filter' => array('or' => array('filters' => array(), '_cache' => true)));

    foreach ($hosts as $host) {
        $filter['filter']['or']['filters'][] = array('term' => array('host.raw' => $host));
    }

    return $filter;
}

/* Find the highest level reference to $key in query, and return all of the keys leading to that key in an array 
 * e.g. 
 *    multitenancy_key_bfs(
 *        array('a' => array('b' => 'c'), 'b' => 1),
 *        'b'
 *    )
 *
 * should return
 *
 *    array()
 * 
 * while
 *    multitenancy_key_bfs(
 *        array('a' => array('a' => array('b' => 'c'), 'b' => 1)),
 *        'b'
 *    )
 * 
 * should return
 *
 *    array('a').

 * Note: this function assumes that you will never have the same key occur twice at the same 'level', i.e. the output of
 * multitenancy_key_bfs(
 *     array('a' => array('b' => 1), 'c' => array('b' => 2)), 
 *     'b'
 * )
 * is not defined
 */ 
function multitenancy_key_bfs($query, $key)
{
    // Start at the top level of the array (no keys to follow)
    $queue = array(array());
    for ($i = 0; $i < count($queue); $i++) {
        // Start at the top level
        $current = $query;

        // Each of the levels we need to follow to get where we want
        $keys = $queue[$i];

        // Follow the keys to get to the right level.
        for ($j = 0; $j < count($keys); $j++) {
            if (is_object($current)) {
                $current = $current->$keys[$j];
            }
            else {
                $current = $current[$keys[$j]];
            }
            
        }

        // If we're looking at a leaf, leave it alone.
        if (!(is_array($current) || is_object($current))) {
            continue;
        }

        foreach ($current as $candidate_key => $value) {
            // Check if the desired key is the same as any of the candidate keys
            if ($key === $candidate_key) {
                return $keys;
            }
            // If it's not, take the array of keys we used to get here, 
            $one_deeper = $keys;
            // add the current key to that array, 
            $one_deeper[] = $candidate_key;
            // push to the end of the queue.
            $queue[] = $one_deeper;
        }
    }

    return false;
}