#!/bin/bash
#
# Bash script for restoring Nagios Log Server backups
# Copyright (c) 2014-2017 Nagios Enterprises, LLC. All rights reserved.
#
# Restores the backups created for the 3 internal Nagios Log Server databases
#

INDEXNAMES=( "nagioslogserver" "kibana-int" "nagioslogserver_log" "nagioslogserver_history" "easy_buttons")
LOGSERVER_DIR="/usr/local/nagioslogserver"
BACKUP_DIR="/store/backups/nagioslogserver"
PYTHON=$(which python 2>/dev/null || which python3 2>/dev/null || which python2 2>/dev/null)

if [ -z "$1" ];then
    echo "Backup file must be specified as the first argument"
    echo "e.g. ./$0 backup_name.tar.gz"
    exit 1
fi

# Restoring mapping files with the index mapping data
echo "Starting Nagios Log Server Restore"
echo "----------------------------------"

filename=$1
if [[ $filename != *.tar.gz ]]; then
    filename="$filename.tar.gz"
fi
folder=${filename:0:${#filename}-7}
echo "Backup folder: $folder"
echo ""

# Extract the file given and start the actual updating
echo "Extracting the backups."
cd $BACKUP_DIR
tar xf $filename
cd $folder

# Loop through all the indexes and restore them one by one
echo "Creating restore jobs for indexes."
for index in "${INDEXNAMES[@]}"; do
    curl -s -XPOST 'http://localhost:9200/_import/state' > state.json
    count=$($PYTHON -c "import sys; import json; print(json.loads(sys.stdin.read())['count'])" < state.json)
    while [[ $count -gt 0 ]]; do
        echo "Waiting for available slot"
        sleep 1
        curl -s -XPOST 'http://localhost:9200/_import/state' > state.json
        count=$($PYTHON -c "import sys; import json; print(json.loads(sys.stdin.read())['count'])" < state.json)
    done
    echo "Restoring: $index "
    # Delete and restore the index
    if [ -f "$folder/$index.tar.gz" ]; then
        # Delete index
        curl -XDELETE "http://localhost:9200/$index/" > /dev/null 2>&1

        # Restore the index by importing the index tar.gz
        curl -XPOST "http://localhost:9200/$index/_import?path=$folder/$index.tar.gz" > /dev/null 2>&1
    else
        printf "\n\n ERROR: Backup file $filename was not found\n"
        exit 1
    fi
    
    count=1
    while [[ $count -gt 0 ]]; do
        echo -n "."
        sleep 3
        curl -s -XPOST 'http://localhost:9200/_import/state' > state.json
        count=$($PYTHON -c "import sys; import json; print(json.loads(sys.stdin.read())['count'])" < state.json)
    done

    echo -n " done."

done
echo "All indexes restored."

# Reset backend jobs
echo "Resetting backend jobs... "
curl -XPOST 'localhost:9200/nagioslogserver/commands/snapshots_maintenance/_update' -d '{ "doc": { "run_time": 0, "status": "waiting" } }' > /dev/null 2>&1
curl -XPOST 'localhost:9200/nagioslogserver/commands/backups/_update' -d '{ "doc": { "run_time": 0, "status": "waiting" } }' > /dev/null 2>&1
curl -XPOST 'localhost:9200/nagioslogserver/commands/run_update_check/_update' -d '{ "doc": { "run_time": 0, "status": "waiting" } }' > /dev/null 2>&1
curl -XPOST 'localhost:9200/nagioslogserver/commands/run_all_checks/_update' -d '{ "doc": { "run_time": 0, "status": "waiting" } }' > /dev/null 2>&1
curl -XPOST 'localhost:9200/nagioslogserver/commands/cleanup_cmdsubsys/_update' -d '{ "doc": { "run_time": 0, "status": "waiting" } }' > /dev/null 2>&1
echo "done."

# Restore snapshots
cp -r $folder/snapshots/*.tar.gz $LOGSERVER_DIR/snapshots
chown nagios:nagios -R $LOGSERVER_DIR/snapshots

# Apply new configuration of ES
php /var/www/html/nagioslogserver/www/index.php configure apply_to_instances

# Clean up
rm -rf "$folder"

echo ""
echo "Restore Complete!"
