define([
  'angular',
  'jquery',
  'kbn',
  'lodash',
  'config',
  'moment',
  'modernizr',
  'filesaver',
  'blob'
],
function (angular, $, kbn, _, config, moment, Modernizr) {
  'use strict';

  var module = angular.module('kibana.services');

  module.service('dashboard', function(
    $routeParams, $http, $rootScope, $injector, $location, $timeout,
    ejsResource, timer, kbnIndex, alertSrv, esVersion, esMinVersion
  ) {
    // A hash of defaults to use when loading a dashboard

    var _dash = {
      title: "",
      style: "dark",
      editable: true,
      failover: false,
      panel_hints: true,
      rows: [],
      pulldowns: [
        {
          type: 'query',
        },
        {
          type: 'filtering'
        }
      ],
      nav: [
        {
          type: 'timepicker'
        }
      ],
      services: {},
      loader: {
        save_gist: false,
        save_elasticsearch: true,
        save_local: true,
        save_default: true,
        save_temp: true,
        save_temp_ttl_enable: true,
        save_temp_ttl: '30d',
        load_gist: false,
        load_elasticsearch: true,
        load_elasticsearch_size: 20,
        load_local: false,
        hide: false
      },
      index: {
        interval: 'none',
        pattern: '_all',
        default: 'INDEX_MISSING',
        warm_fields: true
      },
      refresh: false
    };

    // An elasticJS client to use
    var ejs = ejsResource(config.elasticsearch);
    var gist_pattern = /(^\d{5,}$)|(^[a-z0-9]{10,}$)|(gist.github.com(\/*.*)\/[a-z0-9]{5,}\/*$)/;

    // Store a reference to this
    var self = this;
    var filterSrv,querySrv;

    this.current = _.clone(_dash);
    this.last = {};
    this.availablePanels = [];

    $rootScope.$on('$routeChangeSuccess',function(){
      // Clear the current dashboard to prevent reloading
      self.current = {};
      self.indices = [];
      esVersion.isMinimum().then(function(isMinimum) {
        if(_.isUndefined(isMinimum)) {
          return;
        }
        if(isMinimum) {
          route();
        } else {
          alertSrv.set('Upgrade Required',"Your version of Elasticsearch is too old. Kibana requires Elasticsearch " + esMinVersion + " or above.", "error");
        }
      });
    });

    var route = function() {
      // Is there a dashboard type and id in the URL?
      if(!(_.isUndefined($routeParams.kbnType)) && !(_.isUndefined($routeParams.kbnId))) {
        var _type = $routeParams.kbnType;
        var _id = $routeParams.kbnId;

        switch(_type) {
        case ('elasticsearch'):
          self.elasticsearch_load('dashboard',_id,_type);
          break;
        case ('temp'):
          self.elasticsearch_load('temp',_id,_type);
          break;
        case ('report'):
          self.elasticsearch_load('report',_id,_type);
          break;
        case ('file'):
          self.file_load(_id);
          break;
        case('script'):
          self.script_load(_id);
          break;
        case('local'):
          self.local_load();
          break;
        default:
          $location.path(config.default_route);
        }
      // No dashboard in the URL
      } else {
        $http.get('api/user/get_user_default_dashboard')
        .success(function(data) {
          if (data.dashboard != "") {
            $location.path(data.dashboard);
          } else {
            $location.path(config.default_route);
          }
        });
      }
    };

    // Since the dashboard is responsible for index computation, we can compute and assign the indices
    // here before telling the panels to refresh
    this.refresh = function() {
      LOAD_ID = setTimeout(function() {
        $("#loading").show();
      }, 500);
      if (self.current.index.interval !== 'none') {
        if (_.isUndefined(filterSrv)) {
          $rootScope.$emit('loaded');
          return;
        }
        if (filterSrv.idsByType('time').length > 0) {
          var _range = filterSrv.timeRange('last');
          kbnIndex.indices(_range.from,_range.to,
            self.current.index.pattern,self.current.index.interval
          ).then(function (p) {
            if (p.length > 0) {
              self.indices = p;
            } else {
              // Option to not failover
              if (self.current.failover) {
                self.indices = [self.current.index.default];
              } else {
                // Do not issue refresh if no indices match. This should be removed when panels
                // properly understand when no indices are present
                alertSrv.set(glv('No results'), glv('There were no results because no indices were found that match your selected time span'), 'info', 5000);
                $rootScope.$emit('loaded');
                return false;
              }
            }
            // Don't resolve queries until indices are updated
            querySrv.resolve().then(function(){
              $rootScope.$broadcast('refresh');
              $rootScope.$emit('loaded');
            });
          });
        } else {
          if (self.current.failover) {
            self.indices = [self.current.index.default];
            querySrv.resolve().then(function(){
              $rootScope.$broadcast('refresh');
              $rootScope.$emit('loaded');
            });
          } else {
            alertSrv.set(glv('No time filter'), glv('Timestamped indices are configured without a failover. Waiting for time filter.'), 'info', 5000);
            $rootScope.$emit('loaded');
          }
        }
        $rootScope.$emit('loaded');
      } else {
        self.indices = [self.current.index.default];
        querySrv.resolve().then(function(){
          $rootScope.$broadcast('refresh');
          $rootScope.$emit('loaded');
        });
      }
    };

    $rootScope.$on('loaded', function(evt, data) {
      clearTimeout(LOAD_ID);
      if ($('#loading').is(":visible")) {
        $('#loading').hide();
      }
    });

    var dash_defaults = function(dashboard) {
      _.defaults(dashboard,_dash);
      _.defaults(dashboard.index,_dash.index);
      _.defaults(dashboard.loader,_dash.loader);
      return _.cloneDeep(dashboard);
    };

    this.dash_load = function(dashboard, dash_type) {
      // Cancel all timers
      timer.cancel_all();

      // Make sure the dashboard being loaded has everything required
      dashboard = dash_defaults(dashboard);

      // If not using time based indices, use the default index
      if(dashboard.index.interval === 'none') {
        self.indices = [dashboard.index.default];
      }

      // Set the current dashboard
      self.current = _.clone(dashboard);
      self.current.dash_type = dash_type;
      if (self.current.dash_type == "report") {
        self.current.report = true;
      }

      // Delay this until we're sure that querySrv and filterSrv are ready
      $timeout(function() {
        // Ok, now that we've setup the current dashboard, we can inject our services
        if(!_.isUndefined(self.current.services.query)) {
          querySrv = $injector.get('querySrv');
          querySrv.init();
        }
        if(!_.isUndefined(self.current.services.filter)) {
          filterSrv = $injector.get('filterSrv');
          filterSrv.init();
        }
      },0).then(function() {
        // Call refresh to calculate the indices and notify the panels that we're ready to roll
        self.refresh();
      });

      if(dashboard.refresh) {
        self.set_interval(dashboard.refresh);
      }

      // Set the available panels for the "Add Panel" drop down
      self.availablePanels = _.difference(config.panel_names,
        _.pluck(_.union(self.current.nav,self.current.pulldowns),'type'));

      // Take out any that we're not allowed to add from the gui.
      self.availablePanels = _.difference(self.availablePanels,config.hidden_panels);

      // Set the saving to whatever it needs to be...
      if ($('#save-dash-button').hasClass('btn-danger')) {
        $('#save-dash-button').removeClass('btn-danger');
      }
      $('#save-dash-button').addClass('btn-success');
      $('.brand div').remove();

      return true;
    };

    this.gist_id = function(string) {
      if(self.is_gist(string)) {
        return string.match(gist_pattern)[0].replace(/.*\//, '');
      }
    };

    this.is_gist = function(string) {
      if(!_.isUndefined(string) && string !== '' && !_.isNull(string.match(gist_pattern))) {
        return string.match(gist_pattern).length > 0 ? true : false;
      } else {
        return false;
      }
    };

    this.to_file = function() {
      var blob = new Blob([angular.toJson(self.current,true)], {type: "application/json;charset=utf-8"});
      // from filesaver.js
      window.saveAs(blob, self.current.title+"-"+new Date().getTime());
      return true;
    };

    this.set_default = function(route) {
      var req = $http({
        method: "post",
        url: "index.php/api/user/set_user_default_dashboard",
        data: {
          "dashboard": route
        }
      });
      return true;
    };

    this.purge_default = function() {
      var req = $http({
        method: "post",
        url: "index.php/api/user/set_user_default_dashboard",
        data: {
          "dashboard": ""
        }
      });
      return true;
    };

    // TOFIX: Pretty sure this breaks when you're on a saved dashboard already
    this.share_link = function(title,type,id) {
      return {
        location  : window.location.href.substr(0, window.location.href.indexOf('#')),
        type      : type,
        id        : id,
        link      : window.location.href.substr(0, window.location.href.indexOf('#'))+"#dashboard/"+type+"/"+encodeURIComponent(id),
        title     : title
      };
    };

    var renderTemplate = function(json,params) {
      var _r;
      _.templateSettings = {interpolate : /\{\{(.+?)\}\}/g};
      var template = _.template(json);
      var rendered = template({ARGS:params});
      try {
        _r = angular.fromJson(rendered);
      } catch(e) {
        _r = false;
      }
      return _r;
    };

    this.local_load = function() {
      var dashboard = JSON.parse(window.localStorage['dashboard']);
      dashboard.rows.unshift({
        height: "30",
        title: glv("Deprecation Notice"),
        panels: [
          {
            title: glv('WARNING: Legacy dashboard'),
            type: 'text',
            span: 12,
            mode: 'html',
            content: glv('This dashboard has been loaded from the browsers local cache. If you use another brower or computer you will not be able to access it!')+
            '\n\n  <h4>Good news!</h4> Kibana'+
            ' now stores saved dashboards in Elasticsearch. Click the <i class="fa fa-save"></i> '+
            'button in the top left to save this dashboard. Then select "Set as Home" from'+
            ' the "advanced" sub menu to automatically use the stored dashboard as your Kibana '+
            'landing page afterwards'
          }
        ]
      });
      self.dash_load(dashboard);
    };

    this.file_load = function(file) {
      return $http({
        url: "app/dashboards/"+file.replace(/\.(?!json)/,"/")+'?' + new Date().getTime(),
        method: "GET",
        transformResponse: function(response) {
          return renderTemplate(response,$routeParams);
        }
      }).then(function(result) {
        if(!result) {
          return false;
        }
        self.dash_load(dash_defaults(result.data), 'file');
        return true;
      },function() {
        alertSrv.set(glv('Error'), glv('Could not load') + " <i>dashboards/"+file+"</i>. " + glv('Please make sure it exists') ,'error');
        return false;
      });
    };

    this.elasticsearch_load = function(type,id,dash_type) {
      var successcb = function(data) {
        if (data.found) {
          var response = renderTemplate(angular.fromJson(data)._source.dashboard, $routeParams);
          self.dash_load(response, dash_type);
        } else {
          alertSrv.set(glv('No Dashboard'), glv('The dashboard you had selected no longer exists. Try loading a different dashboard.'), 'error');
          return false;
        }
      };
      var errorcb = function(data, status) {
        if (status === 0) {
          alertSrv.set(glv('Error'), glv('Could not contact Elasticsearch at')+" "+ejs.config.server+". "+glv('Please ensure that Elasticsearch is reachable from your system.'), 'error');
        } else {
          alertSrv.set(glv('Error'), glv('Could not find')+" "+id+". "+glv('If you are using a proxy, ensure it is configured correctly'), 'error');
        }
        return false;
      };

      ejs.client.get(
        "/" + config.kibana_index + "/"+type+"/"+id+'?' + new Date().getTime(),
        null, successcb, errorcb);
    };

    this.script_load = function(file) {
      var runscriptcb = function(data) {
        if (data != undefined) {
          if (data._source.services != undefined) {
            $routeParams.services = data._source.services;
          } else if (data._source.dash_query != undefined) {
            $routeParams.services = data._source.dash_query;
          }
        }
        $http({
          url: "app/dashboards/"+file.replace(/\.(?!js)/,"/"),
          method: "GET",
          transformResponse: function(response) {
            var _f = new Function('ARGS','kbn','_','moment','window','document','angular','require','define','$','jQuery',response);
            return _f($routeParams,kbn,_,moment);
          }
        }).then(function(result) {
          if(!result) {
            return false;
          }
          self.dash_load(dash_defaults(result.data), 'file');
          return true;
        },function() {
          alertSrv.set(glv('Error'),
            glv('Could not load')+" <i>scripts/"+file+"</i>. "+glv('Please make sure it exists and returns a valid dashboard'), 'error');
          return false;
        });
      };
      var errorcb = function(data, status) {
        if (status === 0) {
          alertSrv.set(glv('Error'), glv('Could not contact Elasticsearch at')+" "+ejs.config.server+". "+glv('Please ensure that Elasticsearch is reachable from your system.'), 'error');
        } else {
          alertSrv.set(glv('Error'), glv('Could not find')+" "+id+". "+glv('If you are using a proxy, ensure it is configured correctly.'), 'error');
        }
        return false;
      };
      if ($routeParams.q != undefined) {
        var qid = $routeParams.q;
        ejs.client.get("/nagioslogserver/query/"+qid, null, runscriptcb, errorcb);
      } else if ($routeParams.a != undefined) {
        var aid = $routeParams.a;
        ejs.client.get("/nagioslogserver/alert/"+aid, null, runscriptcb, errorcb);
      } else {
        runscriptcb()
      }
    };

    this.elasticsearch_save = function(type,title,ttl,username,groups,update) {
      // Clone object so we can modify it without influencing the existing obejct
      var save = _.clone(self.current);
      delete save.loaded_query;

      // Change title on object clone
      if (type === 'dashboard' || type === 'report') {
        save.title = _.isUndefined(title) ? self.current.title : title;
      }

      // Change username/groups to guest
      if (type == 'temp') {
        username = "guest";
        groups = "";
      }

      if (update == true) {
        var id = $routeParams.kbnId;
        var request = ejs.Document(config.kibana_index,type,id).source({
          user: username,
          group: groups,
          title: save.title,
          dashboard: angular.toJson(save)
        });
      } else {
        var request = ejs.Document(config.kibana_index,type).source({
          user: username,
          group: groups,
          title: save.title,
          dashboard: angular.toJson(save)
        });
      }

        request = type === 'temp' && ttl ? request.ttl(ttl) : request;

        return request.doIndex(
          // Success
          function(result) {
            $('form[name="saveas"] input').val('');
            if (type === 'dashboard') {
              $location.path('/dashboard/elasticsearch/'+result._id);
            }
            return result;
          },
          // Failure
          function() {
            return false;
          }
        );
    };

    this.elasticsearch_delete = function(id, type) {
      return ejs.Document(config.kibana_index,type,id).doDelete(
        // Success
        function(result) {
          return result;
        },
        // Failure
        function() {
          return false;
        }
      );
    };

    this.elasticsearch_list = function(query,count,userdata,type) {
      var request = _.isUndefined(type) ? ejs.Request().indices(config.kibana_index).types('dashboard') : ejs.Request().indices(config.kibana_index).types(type);
      var qq = query + " && (user:" + userdata.username + " OR user:global)";
      return request.query(
        ejs.QueryStringQuery(qq)
        ).size(count).doSearch(
          // Success
          function(result) {
            return result;
          },
          // Failure
          function() {
            return false;
          }
        );
    };

    this.save_gist = function(title,dashboard) {
      var save = _.clone(dashboard || self.current);
      save.title = title || self.current.title;
      return $http({
        url: "https://api.github.com/gists",
        method: "POST",
        data: {
          "description": save.title,
          "public": false,
          "files": {
            "kibana-dashboard.json": {
              "content": angular.toJson(save,true)
            }
          }
        }
      }).then(function(data) {
        return data.data.html_url;
      }, function() {
        return false;
      });
    };

    this.gist_list = function(id) {
      return $http.jsonp("https://api.github.com/gists/"+id+"?callback=JSON_CALLBACK"
      ).then(function(response) {
        var files = [];
        _.each(response.data.data.files,function(v) {
          try {
            var file = JSON.parse(v.content);
            files.push(file);
          } catch(e) {
            return false;
          }
        });
        return files;
      }, function() {
        return false;
      });
    };

    this.start_scheduled_refresh = function (after_ms) {
      timer.cancel(self.refresh_timer);
      self.refresh_timer = timer.register($timeout(function () {
        self.start_scheduled_refresh(after_ms);
        self.refresh();
      }, after_ms));
    };

    this.cancel_scheduled_refresh = function () {
      timer.cancel(self.refresh_timer);
    };

    this.set_interval = function (interval) {
      self.current.refresh = interval;
      if (interval) {
        var _i = kbn.interval_to_ms(interval);
        this.start_scheduled_refresh(_i);
      } else {
        this.cancel_scheduled_refresh();
      }
    };

    // Nagios Log Server special functions

    // Function to create a new alert
    this.create_nls_alert = function(alert) {

      // refresh dashboard (updates angular dashboard to be the same as the user's dashboard)
      self.refresh();
      
      // Verify everything has a value

      if (is_demo_mode) {
        $('#alert_alert').html('<div class="alert alert-danger"><button type="button" class="close" data-dismiss="alert">&times;</button>'+glv('This function is not available in demo mode.')+'</div>');
        return;
      }

      if ($('#a_name').val().trim() == '') {
        $('#alert_alert').html('<div class="alert alert-danger"><button type="button" class="close" data-dismiss="alert">&times;</button>'+glv('You must enter an alert name.')+'</div>');
        $('#alert_alert').show();
        return;
      }

      if ($('#a_ci').val().trim() == '' || $('#a_lp').val().trim() == '') {
        $('#alert_alert').html('<div class="alert alert-danger"><button type="button" class="close" data-dismiss="alert">&times;</button>'+glv('You must enter a check interval and lookback period.')+'</div>');
        $('#alert_alert').show();
        return;
      }

      if ($('#a_w').val().trim() == '' || $('#a_c').val().trim() == '') {
        $('#alert_alert').html('<div class="alert alert-danger"><button type="button" class="close" data-dismiss="alert">&times;</button>'+glv('You must enter a warning and critical threshold.')+'</div>');
        $('#alert_alert').show();
        return;
      }

      // Check interval and lookback period

      var ci = $('#a_ci').val().trim();
      var lp = $('#a_lp').val().trim();
      var valid_time = ['s', 'm', 'h', 'd', 'w', 'm', 'y'];
      if (valid_time.indexOf(ci.substr(-1)) == -1) {
        $('#alert_alert').html('<div class="alert alert-danger"><button type="button" class="close" data-dismiss="alert">&times;</button>'+glv('You must enter a valid time. (Valid suffexes: s, m, h, d)')+'</div>');
        return;
      }
      if (valid_time.indexOf(lp.substr(-1)) == -1) {
        $('#alert_alert').html('<div class="alert alert-danger"><button type="button" class="close" data-dismiss="alert">&times;</button>'+glv('You must enter a valid time. (Valid suffexes: s, m, h, d)')+'</div>');
        return;
      }

      var queries = querySrv.getQueryObjs(self.current.services.query.ids);

      // Create a query for us to use...
      var boolQuery = ejs.BoolQuery();
        _.each(queries,function(q) {
        boolQuery = boolQuery.should(querySrv.toEjsObj(q));
      });
      var x = ejs.QueryFilter(
                ejs.FilteredQuery(
                  boolQuery,
                  filterSrv.getBoolFilter(filterSrv.ids())));
      var qstr = x.toString();

      // Parse json to remove the fquery
      var obj = JSON.parse(qstr);
      var query_str = JSON.stringify(obj.fquery);

      var aco = 0;
      if ($('#a_send_wc_only').is(":checked")) {
        aco = 1;
      }

      // Now that we have the query string let's get the rest of the stuff we need
      var check = { name: $('#a_name').val(),
                    created_by: LS_USERNAME,
                    check_interval: $('#a_ci').val(),
                    lookback_period: $('#a_lp').val(),
                    warning: $('#a_w').val(),
                    critical: $('#a_c').val(),
                    query: query_str,
                    dash_query: JSON.stringify(self.current.services),
                    method: {
                      type: $('#a_type').val()
                    },
                    alert_crit_only: aco
                  }

      // Add more to the alert based on type...
      switch (check.method.type) {
        case 'nrdp':
          check.method.hostname = $('#a_hostname').val();
          check.method.servicename = $('#a_servicename').val();
          check.method.server_id = $('#a_nrdp_servers').val();
          break;
        case 'email':
          var user_ids = [];
          $('#select_users option:selected').each(function(i, opt) {
            user_ids.push($(opt).val());
          });
          check.method.user_ids = user_ids;
          check.method.tpl_id = $('#templates').val();
          break;
        case 'exec':
          check.method.path = $('#a_script').val();
          check.method.args = $('#a_args').val();
          break;
        case 'snmp':
          check.method.snmp_receiver = $('#a_snmp_receivers').val();
          break;
        case 'reactor':
          check.method.reactor_server_id = $('#a_reactor_servers').val();
          check.method.chain_id = $('#a_reactor_chains').val();
          check.method.chain_name = $('#a_reactor_chains').text();
          break;
      }

      // Add check to the elasticsearch database
      $http({
        url: site_url+"api/check/create",
        method: "POST",
        data: JSON.stringify(check)
      }).success(function(data) {
        // Display the "Check Created" message to the user
        $('.alert-body').html('');
        $('.create-alert-btn').hide();
        $('.alert-message').html(glv('Check successfully created. You can')+' <a href="'+site_url+'alerts">'+glv('view it now')+'</a>.');
        $('.alert').removeClass('alert-danger').addClass('alert-success').show();

      }).error(function(data) {
        var message = glv('Could not create check, please try again.');
        if (data.hasOwnProperty('output')) {
          message = data.output;
        }
        $('#alerteditor-initial-message').hide();
        $('.alert-message').html(message);
        $('.alert').addClass('alert-danger').show();
      });

    };

    // Create a new "query"
    this.create_nls_query = function() {

      if (is_demo_mode) {
        $('#query_alert').html('<div class="alert alert-danger"><button type="button" class="close" data-dismiss="alert">&times;</button>'+glv('This function is not available in demo mode.')+'</div>');
        return;
      }

      // Verify that it is filled out with a proper name..
      if ($('#query_name').val().trim() == '') {
        $('#query_alert').html('<div class="alert alert-danger"><button type="button" class="close" data-dismiss="alert">&times;</button>'+glv('Must enter a valid name for the query.')+'</div>');
        $('#query_alert').show();
        return;
      }

      // Grab the query string...
      var queries = querySrv.getQueryObjs(self.current.services.query.ids);

      // Create a query for us to use...
      var boolQuery = ejs.BoolQuery();
        _.each(queries,function(q) {
        boolQuery = boolQuery.should(querySrv.toEjsObj(q));
      });
      var x = ejs.QueryFilter(
                ejs.FilteredQuery(
                  boolQuery,
                  filterSrv.getBoolFilter(filterSrv.ids())));
      var qstr = x.toString();

      // Parse json to remove the fquery
      var obj = JSON.parse(qstr);
      var query_str = JSON.stringify(obj.fquery);

      // Grab the services json
      var services = JSON.stringify(self.current.services);

      var show_to_everyone = 0;
      if ($('#query_show_everyone').is(":checked")) {
        show_to_everyone = 1;
      }

      var query = { name: $('#query_name').val(),
                    raw: query_str,
                    services: services,
                    show: show_to_everyone };

      // Store everything in elasticsearch
      $http({
        url: site_url+"api/check/create_query",
        method: "POST",
        data: JSON.stringify(query)
      }).then(function(data) {
        $('#query_name').val('');
        get_queries_list();
      });

    };

    this.reload_nls_dash = function(id) {
      if (id == undefined || id == '') {
        var id = $('#load_query').val();
      }

      // Load the series data, replace, and refresh
      $http({
        url: site_url+"api/check/get_queries?id="+id,
        method: "GET"
      }).then(function(resp) {
        var query = resp.data;
        self.current.services = JSON.parse(query.services);
        self.current.loaded_query = query.name;
        self.refresh();
        $('.manage-queries').parents('.modal').modal('hide');
        $timeout(function() {
          $.each($('.top-row-close'), function(i, v) {
            if ($(v).find('span').html() == 'filtering') {
              if (!$(v).parent().find('.top-row-open').is(":visible")) {
                $(v).trigger('click');
              }
            }
          });
        });
      });
    };

    this.overwrite_nls_dash = function() {
      var id = $('#overwrite_query').val();

      // Grab the query string...
      var queries = querySrv.getQueryObjs(self.current.services.query.ids);

      // Create a query for us to use...
      var boolQuery = ejs.BoolQuery();
        _.each(queries,function(q) {
        boolQuery = boolQuery.should(querySrv.toEjsObj(q));
      });
      var x = ejs.QueryFilter(
                ejs.FilteredQuery(
                  boolQuery,
                  filterSrv.getBoolFilter(filterSrv.ids())));
      var qstr = x.toString();

      // Parse json to remove the fquery
      var obj = JSON.parse(qstr);
      var query_str = JSON.stringify(obj.fquery);

      // Grab the services json
      var services = JSON.stringify(self.current.services);

      // Update query
      var update_query = { id: id,
                           raw: query_str,
                           services: services };

      // Update the query in elasticsearch
      $http({
        url: site_url+"api/check/update_query",
        method: "POST",
        data: JSON.stringify(update_query)
      }).then(function(data) {
        $('#query_alert').html('<div class="alert alert-success"><button type="button" class="close" data-dismiss="alert">&times;</button>'+glv('Query updated successfully.')+'</div>');
        $('#query_alert').show();
        get_queries_list();
      });

    }

  });
});