<?php

/* DO NOT REMOVE THIS LINE! sourceguardian-check */

/**
*  functions that can be called from cmdsubsys
**/

/************************************************
    CMDSUBS FUNCTIONS 
*************************************************/ 

/**
 * Run an update check in the background
 */
function update_check($args=array())
{
    // Run an update check (forced)
    do_update_check(false);

    // Run a forced maintenance check (forced)
    do_check_maintenance(true);
}

/**
 * Start Logstash or OpenSearch
 **/
function start_service($args=array())
{
    $ci =& get_instance();
    $ci->load->model('Systemstat');
    $subsystem = $args[0];

    $log = array('type' => 'JOBS',
                 'message' => $subsystem . _(' is being started'),
                 'node' => NODE
                );
    $logged = $ci->logger->log($log);

    return json_encode($ci->Systemstat->start($subsystem));
}

/**
 * Stop Logstash or OpenSearch
 **/
function stop_service($args=array())
{
    $ci =& get_instance();
    $ci->load->model('Systemstat');
    $subsystem = $args[0];

    $log = array('type' => 'JOBS',
                 'message' => $subsystem . _(' is being stopped'),
                 'node' => NODE
                );
    $logged = $ci->logger->log($log);

    return json_encode($ci->Systemstat->stop($subsystem));
}

/**
 * Retart Logstash or OpenSearch
 **/
function restart_service($args=array())
{
    $ci =& get_instance();
    $ci->load->model('Systemstat');
    $subsystem = $args[0];

    $log = array('type' => 'JOBS',
                 'message' => $subsystem . _(' is being restarted'),
                 'node' => NODE
                );
    $logged = $ci->logger->log($log);

    return json_encode($ci->Systemstat->restart($subsystem));
}

/************************************************
    CLEANUP FUNCTIONS
*************************************************/ 

/**
 * Clean up the cmdsubsys commands
 */
function cleanup()
{
    $ci =& get_instance();
    $time = time() - 24*60*60; // Keep the 
    $result = $ci->elasticsearch->query('commands', "status:completed AND run_time:<$time");
    if ($result['hits']['total']['value'] > 0) {
        $commands = $result['hits']['hits'];
        foreach ($commands as $cmd) {
            $ci->elasticsearch->delete('commands', $cmd['_id']);
        }
        $log = array('type' => 'JOBS',
                     'message' => count($commands) . " " . _('jobs removed that were completed jobs more than 24 hours old')
        );
        $logged = $ci->logger->log($log);
    }
}

/************************************************
    BACKUP / MAINTENANCE FUNCTIONS 
*************************************************/ 

/**
 * Run maintance jobs
 **/
function do_maintenance($args=array())
{
    $ci =& get_instance();

    $maintenance_settings = unserialize(get_option('maintenance_settings', array()));

    if(empty($maintenance_settings)){
        set_option('maintenance_settings', serialize(array('active', 1,
                                                           'optimize_time' => 2, 
                                                           'close_time' => 30,
                                                           'delete_time' => 0,
                                                           'audit_retention' => 0,
                                                           'alert_retention' => 0,
                                                           )));
        $maintenance_settings = get_option('maintenance_settings');
    }

    if(!$maintenance_settings['active'])
        return _('Maintenance and Backup jobs are disabled');

    # Trim audit log
    if ($maintenance_settings['audit_retention'])
        do_maintenance_trim_audit_index($maintenance_settings['audit_retention']);

    # Trim alert history
    if ($maintenance_settings['alert_retention'])
        do_maintenance_trim_alert_index($maintenance_settings['alert_retention']);

    # Optimize indexes
    if ($maintenance_settings['optimize_time'])
        do_maintenance_optimize($maintenance_settings['optimize_time']);

    # Close Older Indices
    if ($maintenance_settings['close_time'])
        do_maintenance_close_indices($maintenance_settings['close_time']);

    #Delete Older Indices
    if ($maintenance_settings['delete_time'])
        do_maintenance_delete_indices($maintenance_settings['delete_time']);

    # Give OpenSearch enough time to index if necessary
    sleep(1);

    return _('Maintenance and Backup jobs are being executed');
}

function do_maintenance_trim_audit_index($trim_time)
{
    $ci =& get_instance();
    $trim_date = (time() - (60 * 60 * 24 * $trim_time));
    $trim_date = $trim_date * 1000;
    $command = $queries[$mode] . $trim_date;
    $cmd = "https://localhost:9200/nagioslogserver_log/_delete_by_query?q=created:<=".$trim_date;
    $opensearch_user = 'nagioslogserver';
    $opensearch_password = $ci->config->item('opensearch_password');

    syslog(LOG_ERR, "NLS Maintenance Trimming using URL $cmd");

    $ch = curl_init($cmd);
    curl_setopt($ch, CURLOPT_CUSTOMREQUEST, 'POST');
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    curl_setopt($ch, CURLOPT_SSL_VERIFYHOST, false);
    curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false);
    curl_setopt($ch, CURLOPT_USERPWD, "$opensearch_user:$opensearch_password");
    $response = curl_exec($ch);
    curl_close($ch);

    $msg = _('Trimmed Audit log index ') . $trim_time . _(' day(s) old.');
    $log = array('type' => 'MAINTENANCE',
                 'message' => $msg
    );
    $logged = $ci->logger->log($log);
    syslog(LOG_ERR, "NLS Maintenance $msg");

    return $msg;
}

function do_maintenance_trim_alert_index($trim_time)
{
    $ci =& get_instance();
    $trim_date = (time() - (60 * 60 * 24 * $trim_time));
    $command = $queries[$mode] . $trim_date;
    $cmd = "https://localhost:9200/nagioslogserver_history/_delete_by_query?q=ran:<=".$trim_time;

    $ch = curl_init($cmd);
    curl_setopt($ch, CURLOPT_CUSTOMREQUEST, 'POST');
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    $response = curl_exec($ch);
    curl_close($ch);

    $msg = _('Trimmed alert history index ') . $trim_time . _(' day(s) old.');
    $log = array('type' => 'MAINTENANCE',
                 'message' => $msg
    );
    $logged = $ci->logger->log($log);

    return $msg;
}

function get_open_logstash_indices_older_than($index_time) {
    $ci =& get_instance();
    $opensearch_password = $ci->config->item('opensearch_password');

    // Get All Logstash Indices
    $session = curl_init();

    curl_setopt($session, CURLOPT_USERPWD, 'nagioslogserver:' . $opensearch_password);
    curl_setopt($session, CURLOPT_RETURNTRANSFER, true);
    curl_setopt($session, CURLOPT_SSL_VERIFYHOST, false);
    curl_setopt($session, CURLOPT_SSL_VERIFYPEER, false);
    curl_setopt($session, CURLOPT_HTTPHEADER, array('Accept: application/json'));

    $uri = "https://localhost:9200/_cat/indices";
    // Extract Index Names from cluster:
    curl_setopt($session, CURLOPT_URL, $uri);
    $indices = json_decode(curl_exec($session),true);

    $index_older_than_date = date('Y.m.d', strtotime("-$index_time days"));
    $older_indices = array();
    foreach ($indices as $index) {
        if ($index['status'] == 'open' &&
            preg_match('/^logstash-\d\d\d\d\.\d\d\.\d\d/', $index['index']) ) {
            if ($index['index'] < "logstash-$index_older_than_date") {
                $older_indices[] = $index['index'];
            }
        }
    }

    curl_close($session);
    return $older_indices;
}

function do_maintenance_close_indices($close_time) {
    $ci =& get_instance();
    $opensearch_password = $ci->config->item('opensearch_password');

    $close_indices = get_open_logstash_indices_older_than($close_time);
    $session = curl_init();

    $close_results = '';

    if (count($close_indices) > 0) {
        syslog(LOG_INFO, "NLS Maintenance indexes to close: " . json_encode($close_indices));

        $uri = 'https://localhost:9200/' . implode(',', $close_indices) . '/_close';

        curl_setopt($session, CURLOPT_USERPWD, 'nagioslogserver:' . $opensearch_password);
        curl_setopt($session, CURLOPT_RETURNTRANSFER, true);
        curl_setopt($session, CURLOPT_SSL_VERIFYHOST, false);
        curl_setopt($session, CURLOPT_SSL_VERIFYPEER, false);
        curl_setopt($session, CURLOPT_POST, true);
        curl_setopt($session, CURLOPT_URL, $uri);
        $close_results = curl_exec($session);
    }

    $msg = _('Closed indexes ') . $close_time . _(' day(s) old. Results: ' . $close_results);
    $log = array('type' => 'MAINTENANCE',
                 'message' => $msg
    );
    $logged = $ci->logger->log($log);
    curl_close($session);
    return $msg;
}

function do_maintenance_delete_indices($delete_time) {
    $ci =& get_instance();
    $opensearch_password = $ci->config->item('opensearch_password');

    $delete_indices = get_open_logstash_indices_older_than($delete_time);
    $session = curl_init();

    syslog(LOG_INFO, "Deleting Got list of indices: " . implode(',', $delete_indices));

    $close_results = '';

    if (count($delete_indices) > 0) {
        $uri = 'https://localhost:9200/' . implode(',', $delete_indices);

        curl_setopt($session, CURLOPT_USERPWD, 'nagioslogserver:' . $opensearch_password);
        curl_setopt($session, CURLOPT_RETURNTRANSFER, true);
        curl_setopt($session, CURLOPT_SSL_VERIFYHOST, false);
        curl_setopt($session, CURLOPT_SSL_VERIFYPEER, false);
        curl_setopt($session, CURLOPT_CUSTOMREQUEST, "DELETE");
        curl_setopt($session, CURLOPT_URL, $uri);
        syslog(LOG_INFO, "Deleting URL: $uri");
        $delete_results = curl_exec($session);
    }

    $msg = _('Deleted indexes ') . $delete_time . _(' day(s) old. Results: ' . $delete_results);
    $log = array('type' => 'MAINTENANCE',
                 'message' => $msg
    );
    $logged = $ci->logger->log($log);
    syslog(LOG_INFO,"$msg");
    curl_close($session);
    return $msg;
}

/**
 * Optimizing indexes older than $optimize_time
 **/

function do_maintenance_optimize($optimize_time)
{
    $ci =& get_instance();
    $opensearch_password = $ci->config->item('opensearch_password');

    $optimize_indices = get_open_logstash_indices_older_than($optimize_time);

    $optimization_results = '';
    $session = curl_init();

    if (count($optimize_indices) > 0) {
        $uri = 'https://localhost:9200/' . implode(',', $optimize_indices) . '/_forcemerge';
        curl_setopt($session, CURLOPT_USERPWD, 'nagioslogserver:' . $opensearch_password);
        curl_setopt($session, CURLOPT_RETURNTRANSFER, true);
        curl_setopt($session, CURLOPT_SSL_VERIFYHOST, false);
        curl_setopt($session, CURLOPT_SSL_VERIFYPEER, false);
        curl_setopt($session, CURLOPT_POST, true);
        curl_setopt($session, CURLOPT_URL, $uri);
        $optimization_results = curl_exec($session);
    }

    $msg = _('NLSMaintenance Optimized indexes ') . $optimize_time . _(' day(s) old.');
    $log = array('type' => 'MAINTENANCE',
                 'message' => $msg
    );
    $logged = $ci->logger->log($log);
    curl_close($session);
    return $msg;
}

/************************************************
    CONFIGURATION FUNCTIONS
*************************************************/

/**
* Apply the configuration to the node...
**/
function apply_config($args=array())
{
    $ci =& get_instance();
    $ci->load->model('ls_configure');

    // Create a new snapshot before we apply
    if (!empty($args)) {
        $ci->ls_configure->create_snapshot($args['sh_id'], $args['sh_created'], "applyconfig.snapshot");
    }

    // Apply the configuration
    $result = $ci->ls_configure->apply(NODE);

    if ($result){
        $return = "success";
        $message = _("New configuration was applied.");
    } else {
        $return = "failed";
        $message = _("New configuration failed to apply.");
    }

    $log = array('type' => 'CONFIG',
                 'message' => $message,
                 'node' => NODE
    );
    $logged = $ci->logger->log($log);

    return $return;
}

/**
 * Create a snapshot on the node
 **/
function create_snapshot($args=array())
{
    $ci =& get_instance();
    $ci->load->model('ls_configure');
    $success = $ci->ls_configure->create_snapshot($args['id'], $args['created']);
    
    $log = array('type' => 'CONFIG',
                 'message' => _("New configuration snapshot was created."),
                 'node' => NODE
    );
    if (!$success) {
        $log['message'] = _("Error: New configuration snapshot could not be created.");
    }
    $logged = $ci->logger->log($log);
}

/**
 * Deletes a snapshot
 */
function delete_snapshot($args=array())
{
    $ci =& get_instance();
    $ci->load->model('ls_configure');
    $ci->ls_configure->delete_snapshot($args['path']);
    
    $log = array('type' => 'CONFIG',
                 'message' => _("Configuration snapshot was deleted from ". $args['path']),
                 'node' => NODE
    );
    $logged = $ci->logger->log($log);
}

/**
 * Restores a snapshot on the node
 **/
function restore_snapshot($args=array())
{
    $ci =& get_instance();
    $ci->load->model('ls_configure');
    $ci->ls_configure->restore_snapshot($args['id']);
    
    $log = array('type' => 'CONFIG',
                 'message' => _("Configuration snapshot was restored from "). $args['id'],
                 'node' => NODE
    );
    $logged = $ci->logger->log($log);
}

/************************************************
    SYSTEM BACKUPS
*************************************************/

function do_backups($args=array())
{
    $ci =& get_instance();
    $ci->load->model('cmdsubsys');
    $backup_rotation = get_option('backup_rotation');

    if ($backup_rotation > 0) {
        $result = $ci->elasticsearch->query_wresultSize('node', '', 2000);
        foreach ($result['hits']['hits'] as $node) {
            if ($node['_id'] == "global") { continue; }
            $data = array("command" => 'create_backup', "node" => $node['_id'], "args" => '');
            $job = $ci->cmdsubsys->create($data);
        }
    }

    $log = array('type' => 'JOBS',
                 'message' => _("Created 'create_backup' jobs for all nodes.")
    );
    $logged = $ci->logger->log($log);
}

// Ran per-node to create a new backup on the specified node
function create_backup($args=array())
{
    $ci =& get_instance();
    $scripts_dir = $ci->config->item('scripts_dir');
    $ci->load->model('cmdsubsys');

    // Run the create backup script...
    $cmd = $scripts_dir.'/create_backup.sh > /tmp/backups.log';
    exec($cmd, $output, $return_var);

    // Print output to log
    echo "-----\n";
    echo "Running cmd: $cmd\n";
    echo "Return: $return_var\n";
    echo implode("\n", $output);
    echo "\n-----\n";

    // Remove any backups that need to be trimmed
    $backup_rotation = get_option('backup_rotation');
    $backup_dir = '/store/backups/nagioslogserver';
    $count = file_count($backup_dir);
    if ($count > $backup_rotation) {
        while ($count > $backup_rotation) {
            remove_oldest_file($backup_dir);
            $count--;
        }
    }

    // Send out logs to the audit log
    $log = array('type' => 'JOBS',
                 'message' => _("Created LS backup."),
                 'node' => NODE
    );
    if ($return_var > 0) {
        $log['message'] = _("Error creating LS backup. Check permissions of backup directory /store/backups/nagioslogserver and disk space.");
    }
    $logged = $ci->logger->log($log);

    // If we failed, update the global command
    if ($return_var > 0) {
        $job = $ci->cmdsubsys->get('backups');
        $job->update(array('last_run_status' => _("FAILED: Check audit log for JOBS")));
    }
}

// Removes the oldest file in the currently specified directory
function remove_oldest_file($dir)
{
    $oldest_file = '';
    $oldest = time();
    foreach(glob($dir . "/*.tar.gz") as $entry) {
        if (filemtime($entry) < $oldest) {
            $oldest_file = $entry;
            $oldest = filemtime($entry);
        }
    }
    unlink($oldest_file);

    return true;
}

function file_count($dir)
{
    $arr = glob($dir . "/*.tar.gz");
    $count = count($arr);
    return $count;
}

/************************************************
    ALERTING FUNCTIONS
*************************************************/

/**
 * Runs all the alerts that need to be ran
 **/
function run_alerts($args=array())
{
    $ci =& get_instance();
    $ci->load->helper('alert');
    $ci->load->model('alert');
    $alerts = $ci->alert->get_all();

    // Loop through the alerts and actually run them if necessary
    $ci->elasticsearch->refresh('alert');
    foreach ($alerts as $alert) {
        if (verify_alert_run($alert['id'])) {
            $ci->alert->run($alert['id']);
        }
    }
}

/************************************************
    REPORTING FUNCTIONS
*************************************************/

function clear_old_scheduled_report($args=array())
{
    $ci =& get_instance();
    $ci->load->model('ScheduledReport');
    $ci->ScheduledReport->clear_old_cron_job($args['sr_id']);
}

function do_report_generation($args=array())
{
    $output = [];
    $return_var = 0;
    exec($args['cmd'], $output, $return_var);
    //file_put_contents('/usr/local/nagioslogserver/tmp/test.cat', $output);
}

/************************************************
    HOMEPAGE FUNCTIONS
*************************************************/

/**
 * Gets index usage statistics for homepage graph
 **/
function run_index_usage($args = array())
{
    // Create an OpenSearch model for 'nagioslogserver', where we'll store our data.
    $main_index = new Elasticsearch(array('index' => 'nagioslogserver'));

    // Create an OpenSearch model for the current day
    $current_day = new Elasticsearch(array('index' => 'logstash-'.gmdate('Y.m.d')));

    // Retrieve _stats for the current day - contains the total usage of the index.
    $current_stats = $current_day->get_index_statistics();
    $current_total = $current_stats['_all']['total']['store']['size_in_bytes'];
    $time_unix = time();

    // Retrieve the most recent recorded value
    $most_recent_value_query = array(
        'query' => array('match_all' => array()),
        'size' => '1',
        'sort' => array(
            array('time_unix' => array('order' => 'desc'))
    ));

    $most_recent_value_data = $main_index->advancedquery('index_usage', $most_recent_value_query);
    
    // This value should only be used for calculating the very first entry.
    $most_recent_value = array('total' => 0, 'delta' => 0, 'created' => date('c', 0), 'time_delta' => 0, 'time_unix' => 0);
    if (!empty($most_recent_value_data['hits']['hits'])) {
        $most_recent_value = $most_recent_value_data['hits']['hits'][0]['_source'];
    }
    $current_delta = $current_total - $most_recent_value['total'];

    $time_delta = $time_unix - (array_key_exists('time_unix', $most_recent_value) ? $most_recent_value['time_unix'] : 0);

    // Logstash stores ISO time, so we should be consistent with that to make frontend stuff easier
    $current_time = date('c', $time_unix);

    // Record the total and the delta, store in database.
    $insert_data = array(
        'created' => $current_time,
        'time_unix' => $time_unix,
        'total' => $current_total,
        'delta' => $current_delta,
        'time_delta' => $time_delta
    );
    $main_index->add('index_usage', $insert_data);
}

/************************************************
    SYSTEM FUNCTIONS
*************************************************/

/**
 * Runs the change_timezone.sh script
 **/
function change_timezone($args=array())
{
    $ci =& get_instance();
    $ci->load->model('Systemstat');
    $scripts_dir = $ci->config->item('scripts_dir');

    $cmd = "sudo $scripts_dir/change_timezone.sh -z '".$args['timezone']."'";
    exec(escapeshellcmd($cmd), $o, $r);

    // Restart NCPA since it reports timezone.
    $ci->Systemstat->restart('ncpa');

    return true;
}


function start_logdata_migration($args=array()) {
    syslog(LOG_ERR, "Migration start_logdata_migration args: " . json_encode($args));

    $ci =& get_instance();
    $scripts_dir = $ci->config->item('scripts_dir');
    $cmd = "php $scripts_dir/migrate_data.php ";

    $cmd .= ' -s ' . $args['server'];
    if (isset($args['index'])) {
        $cmd .= ' -l -i ' . $args['index'];
    }
    if ($args['force_restart']) {
        $cmd .= ' -r';
    }
    if ($args['migrateLogServerMetaData']) {
        $cmd .= ' -c';
    }

    $cmd = escapeshellcmd($cmd);

    syslog(LOG_INFO, "Nagios Log Server Migration command: $cmd");

    exec($cmd, $o, $r);

    return true;
}

function reconfigure_opensearch($args=array()) {
    $ci =& get_instance();
    $scripts_dir = $ci->config->item('scripts_dir');
    $backend_dir = $ci->config->item('backend_dir');

    $ymlfiles = ['opensearch.yml', 'opensearch-worker.yml'];

    $source_server = $args['server'];

    foreach ($ymlfiles as $ymlfile) {
        $file_name = "$backend_dir/opensearch/config/$ymlfile";
        $opensearch_yml = file_get_contents($file_name);

        if ($opensearch_yml === false) {
            return;
        }

        $opensearch_yml = explode("\n", $opensearch_yml);

        $new_yaml = [];

        $did_set_remote_reindex = false;

        foreach ($opensearch_yml as $line) {
            if (preg_match('/^reindex.remote.allowlist/', $line)) {
                $line = "reindex.remote.allowlist: [\"$source_server:9200\"]\n";
                $did_set_remote_reindex = true;
            }
            $new_yaml[] = $line;
        }

        if (!$did_set_remote_reindex) {
            $new_yaml[] = "reindex.remote.allowlist: [\"$source_server:9200\"]";
        }

        file_put_contents($file_name, implode("\n", $new_yaml));

    }

    $cmd = "sudo systemctl restart opensearch";

    exec($cmd, $o, $r);

    return true;
}

function apply_instance_roles($args=array()) {
    $ci =& get_instance();
    $scripts_dir = $ci->config->item('scripts_dir');
    $backend_dir = $ci->config->item('backend_dir');

    $es = new Elasticsearch(array('index' => 'nagioslogserver'));
    $global = $es->get('node', 'global');
    if (empty($global) || $global['found'] === false) {
        echo "The global node configuration was not found.";
        return false;
    }

    $abcd = json_decode($global['_source']['requested_roles'], true);

    $ymlfiles = ['opensearch.yml'];

    foreach ($ymlfiles as $ymlfile) {
        $file_name = "$backend_dir/opensearch/config/$ymlfile";
        $opensearch_yml = file_get_contents($file_name);

        if ($opensearch_yml === false) {
            return;
        }

        $opensearch_yml = explode("\n", $opensearch_yml);

        $new_yaml = [];
        $node_name = '';

        foreach ($opensearch_yml as $line) {
            if (preg_match('/^node.roles/', $line)) {
                continue;
            }
            if (preg_match('/^node.name/', $line)) {
                $node_name = trim(explode(':', $line, 2)[1]);
            }
            $new_yaml[] = $line;
        }

        if (empty($node_name)) {
            // Try to get the node name from the hostname
            return false;
        }

        $new_roles = [];

        foreach ($abcd['roles'] as $address => $node) {
            if ($node['name'] == $node_name) {
                foreach ($node as $role => $value) {
                    if ($value === true) {
                        $new_roles[] = $role;
                    }
                }
            }
        }
        if (count($new_roles) > 0) {
            $new_roles = json_encode($new_roles);

            // Add the new roles
            $new_yaml[] = "node.roles: " . $new_roles;
            file_put_contents("$file_name", implode("\n", $new_yaml) . "\n");
        }
    }

    $cmd = "sudo systemctl restart opensearch";
    exec($cmd, $o, $r);

    syslog(LOG_INFO, "Instanceroles $cmd produced result code $r with output: " . implode("\n", $o));

    return true;
}
