'use client'

import { useEffect, useState, useRef } from 'react';

import axios, { AxiosError } from 'axios';
import { DashboardQueryProvider } from '@/dashboard/contexts/DashboardQueryContext';
import { DashboardProvider } from '@/dashboard/contexts/DashboardContext';
import { ErrorBoundary } from 'react-error-boundary';
import { ErrorDialog } from '@/ErrorDialog';
import { useTranslation } from 'react-i18next';
import { Token, SimpleSearch } from './SimpleSearch';
import { type TokenInputRef }  from 'react-customize-token-input';
import 'react-customize-token-input/dist/react-customize-token-input.css';

import 'maplibre-gl/dist/maplibre-gl.css';
import { Badge } from './components/shadcn/badge';
import { Button } from "@/components/shadcn/button";
import { Checkbox } from "@/components/shadcn/checkbox";
import {
    Dialog,
    DialogContent,
    DialogFooter,
    DialogTrigger,
    DialogHeader,
    DialogTitle,
    DialogDescription,
    DialogClose
} from '@/components/shadcn/dialog';
import { Label } from "@/components/shadcn/label";
import { Toaster } from '@/components/shadcn/toaster';
import { Search, TextSearch, Sparkles } from "lucide-react";
import Icon from '@mui/material/Icon';
import { easybutton } from './dashboard/easybuttons/types';
import { getUserEasyButtons, defaultEasyButtons } from './dashboard/easybuttons/queries';

import { useNavigate } from "react-router-dom";



function SimpleHome() {
  const tokenInputRef = useRef<TokenInputRef>(null);
  const [tokenValues, setTokenValues] = useState<Token[]>([]);
  const [showHelpDialog, setShowHelpDialog] = useState<boolean>();
  const [preventSubmit, setPreventSubmit] = useState<boolean>(false);
  const [isChecked, setIsChecked] = useState<boolean>();
  const hasLoaded = useRef(false);
  const { t } = useTranslation();
  const navigate = useNavigate();
  const [selectedButtons, setShortcutButtons]  = useState<easybutton[]>(defaultEasyButtons.map((ezbutton) => (Object.assign({}, ezbutton,{name: ezbutton.name + " (default)"})))
                                                                                            .sort(() => Math.random() - 0.5).slice(0, 8));

  // api calls to run on page load
  useEffect(() =>{
    const getShowHelpDialog = async() => {
    await axios.post('/nagioslogserver/api/dashboards/show_simple_help_dialog')
      .then((showHelp) => {
        setShowHelpDialog(showHelp.data);
      })
      .catch(err => console.error('Error fetching data : ', err));
    }

    const getNoSimpleHelp = async() => {
    await axios.post('/nagioslogserver/api/dashboards/request_no_simple_help')
      .then((noHelp) => {
        setIsChecked(noHelp.data);
      })
      .catch(err => console.error('Error fetching data : ', err));
    }

    // gets the user defined queries. shuffles them to the front of the ez buttons list 
    const getUserQueries = async() => {
      await getUserEasyButtons()
          .then((userQueries) => {
            if (userQueries.data.length == 0) {
              return; // do nothing
            }
            const userDefinedQueries = userQueries.data;
            // if new values append to queryList
            console.log("before ", selectedButtons)
            if (userDefinedQueries.length > 8) {
              selectedButtons.splice(0, 8, userDefinedQueries.sort(() => Math.random() - 0.5).slice(0, 8));
              setShortcutButtons(selectedButtons);
            }
            else {
              selectedButtons.splice(0, userDefinedQueries.length, ...userDefinedQueries)
              setShortcutButtons(selectedButtons);
            }

            console.log("after ", selectedButtons)
          })
        .catch(err => console.error('Error fetching data : ', err));
    }
    getUserQueries();

    getShowHelpDialog();
    getNoSimpleHelp();
  }, []);

  let isSimple=true;
  // simple/advanced search buttons
  function searchButtons() {
    return (
      <div className="flex justify-center items-center gap-4 p-4">
        <Button id="simple-search-button" name="dashboardId"
                className={isSimple ? "text-primary-foreground" : "text-secondary-foreground"} 
                style={{ height: "3.25em" }} variant={(isSimple ? "default" : "outline")}>
            <div className="flex items-center gap-2 ">
              <Sparkles className="material-symbols-outlined"></Sparkles>
              { t("Simple Search") }
            </div>
        </Button>
        <Button id="adv-search" name="dashboardId"
                className={isSimple ? "text-secondary-foreground" : "text-primary-foreground"} 
                style={{ height: "3.25em" }} variant={(isSimple ? "outline" : "default")}
                onClick={() => { isSimple = false; }}>
            <div className="flex items-center gap-2">
              <TextSearch className="material-symbols-outlined"></TextSearch>
              { t("Advanced Search") }
            </div>
        </Button>
    </div>)
  }

  // format tokens and submit
  const handleSubmit = (event:React.FormEvent) => {
        event.preventDefault(); 
        if (preventSubmit) { 
          setPreventSubmit(false); 
          return; 
        }
        // format tokens into query
        let detoken;
        detoken = tokenValues.map((t) => {
          if (t.tType == 'query'){
            return t.query;
          }
          return '"' + t.value.replace(/"/g, "'") + '"';
        });
        detoken = encodeURIComponent(detoken.join(" AND "));
        const dashId = isSimple? "simple" : "default";
        // console.log(url + "?dashboardId=" + dashId + "&searchQuery=" + detoken);
        navigate("/nagioslogserver/media/ui/index.html?dashboardId=" + dashId + "&searchQuery=" + detoken + "&simpleTokens=" + JSON.stringify(tokenValues));
  }

  useEffect(() => {
    const setNoSimpleHelp = async() => {
    await axios.post('/nagioslogserver/api/dashboards/request_no_simple_help?' + (isChecked? "true":"false"))
      .then((noHelp) => {
        noHelp.data;
      })
      .catch(err => console.error('Error fetching data : ', err));
    }
    if (hasLoaded.current) {
      setNoSimpleHelp();
    }
    
  }, [isChecked]);

  // pulled out of main return to allow for asynchronous loading of showHelpDialog variable
  const infoDialog = () => {
    if ((showHelpDialog != undefined) && (isChecked != undefined)) {
      const handleCheckClick = () => {
        setIsChecked(!isChecked);
        hasLoaded.current = true;
      }
      return (
        <Dialog defaultOpen={showHelpDialog}>
          <DialogTrigger><u>{t('What is Simple Search?')}</u></DialogTrigger>
          <DialogContent style={{minWidth:"full", top:"30%"}}>
            <DialogHeader>
              <DialogTitle className="flex items-center">
                {t('Simple Search')}
                <Badge className="mx-2" variant='default'>{t('Experimental')}</Badge>
              </DialogTitle>
            </DialogHeader>
            <DialogDescription>
                {t("Simple search allows you to search with one of the ready-made searches below, \
                  or simply the text you type in here. Type a word or words and hit enter to search. \
                  To further refine your search, you can type another word or words, and hit enter again.") }
              </DialogDescription>
            <DialogFooter style={{justifyContent: "space-between"}}>
              <div className="flex items-center">
                <Checkbox id="dont-show-again" defaultChecked={isChecked} onClick={handleCheckClick} />
                <Label htmlFor="dont-show-again" className="px-2">{t("Do not show again")}</Label>
              </div>
              <div>
                <Button className="mx-2" variant="outline" onClick={() => {window.parent.location='/nagioslogserver/profile'}}>
                  {t("Use Advanced")}
                </Button>
                <DialogClose><Button>{t("Get Started")}</Button></DialogClose>
              </div>
            </DialogFooter>
          </DialogContent>
        </Dialog>
      )
    }
    return (
      <u>{t('What is Simple Search?')}</u>
    )
  }

  // easy buttons below search bar
  const easyHomeButtons = () => {
    return (
      <div className="flex flex-wrap flex-row pt-4 my-3 w-1/3 justify-center">
        <link
          rel="stylesheet"
          href="https://fonts.googleapis.com/icon?family=Material+Icons"
        />

        {selectedButtons.map((ezbutton) => (
          <Button className="flex items-center gap-2 m-2"
              id={"shortcut-" + ezbutton.name}
              variant="outline"
              onClick={() => {tokenInputRef.current?.createTokens("~q." + ezbutton.name)}}>
              <Icon className="text-primary">{ezbutton.icon? ezbutton.icon : "saved_search"}</Icon>
             
              {ezbutton.name}
          </Button>
        ))}
      </div>
    )
  }

  // main return of page
  return (
  <div className="flex fixed w-full top-[30%] flex-col items-center justify-evenly gap-4">
      <form id="search-form" onSubmit={handleSubmit} method="get" className="navbar-search items-center gap-4 p-2 w-1/3">
        <div className="flex items-center w-full relative"
            onClick={()=>{tokenInputRef.current?.focus()}}>
          <SimpleSearch tokenInputRef={tokenInputRef} tokenValues={tokenValues} 
              setTokenValues={setTokenValues} 
              setPreventSubmit={setPreventSubmit} />
          <button id="simple-search" type="submit" className="absolute inset-y-0 right-3">
              <Search className="material-symbols-outlined tw-p-3 text-secondary-foreground/70" />
          </button>
        </div>
        {searchButtons()}
      </form>
      {infoDialog()}
      {easyHomeButtons()}
  </div>
  )
}

function Home() {

  const Fallback = ({ error }: { error: AxiosError }) => {
    return <ErrorDialog>{error.message}</ErrorDialog>
  }

  const GenericFallback = ({ error }: { error: AxiosError }) => {
    return <ErrorDialog>{error.message}</ErrorDialog>
  }
    
  return (
    <div className="min-h-screen w-full min-w-screen bg-background">
      <ErrorBoundary FallbackComponent={GenericFallback} onError={(error: Error) => console.log(error)}>
        <DashboardProvider>
          <DashboardQueryProvider>
            <ErrorBoundary FallbackComponent={Fallback} onError={(error: Error) => console.log(error)}>
              <SimpleHome></SimpleHome>
            </ErrorBoundary>
            <Toaster />
          </DashboardQueryProvider>
        </DashboardProvider>
      </ErrorBoundary>
    </div>
  );
}

export default Home;
