'use client'

import React, { useEffect, useState } from 'react';
import { Command,
         CommandItem,
         CommandList
} from "@/components/shadcn/command";
import { useTranslation } from 'react-i18next';
import TokenInput, { type TokenInputRef }  from 'react-customize-token-input';

import { easybutton } from './dashboard/easybuttons/types';
import { getUserEasyButtons, defaultEasyButtons } from './dashboard/easybuttons/queries';

import { SearchCode } from "lucide-react";
import 'maplibre-gl/dist/maplibre-gl.css';
import 'react-customize-token-input/dist/react-customize-token-input.css';
import { Popper } from '@mui/material';


/// token typing. the token type affects how the token is displayed and managed on submit
export enum TokenType {query = 'query', raw = 'raw'}
export type Token = {
  tType: TokenType;
  value: string;
  query: string | null;
}

interface SimpleSearchProps {
  tokenInputRef: React.RefObject<TokenInputRef>,
  tokenValues:Token[],
  setTokenValues:React.Dispatch<React.SetStateAction<Token[]>>,
  setPreventSubmit?:React.Dispatch<React.SetStateAction<boolean>>,
  initString?:string
}

// set up token search object
export function SimpleSearch(props:SimpleSearchProps) {
  const { t } = useTranslation();
  const [queryList, setQueryList] = useState<{[id:string]:string}>(Object.fromEntries(defaultEasyButtons.map(q => [q.name + t(" (default)"), q.query])));
  const [openSuggest, setOpenSuggest] = useState<boolean>(false);
  const [fromDropdown, setFromDropdown] = useState<boolean>(false);
  const [currInput, setCurrInput] = useState<string>('');

  useEffect(() => {
    const getUserQueries = async() => {
      await getUserEasyButtons()
          .then((userQueries) => {
            if (userQueries.data.length == 0) {
              return; // do nothing
            }
            const newQueries = Object.fromEntries(userQueries.data.map((q:easybutton) => [q.name, q.query]));
            // if new values append to queryList
            setQueryList(currQueries => Object.assign({}, currQueries, newQueries));
          })
        .catch(err => console.error('Error fetching data : ', err));
    }
    getUserQueries();
  }, []);

  // takes string input and converts it to a Token type to store in tokens array
  const handleBuildTokenValue = (input:string) => {
    const newToken:Token = {tType:TokenType["raw"], value:input, query:null}
    if (newToken.value.startsWith('~q.') && queryList[newToken.value.replace(/^(~q\.)/, '')] !== undefined) {
      newToken.tType = TokenType['query'];
      newToken.value = newToken.value.replace(/^(~q\.)/, '')
      newToken.query = queryList[newToken.value];
    }
    
    return newToken;
  }

  // display icons next to specific input types
  const handleGetTokenDisplayLabel = (token:Token) => {
    const icon =
      {'query': <SearchCode className="pr-1"></SearchCode>,
        'raw': '',
      }[token.tType] || '';

    return (
      <div className="flex items-center flex-row" style={{margin:0}}>
          {icon}
          {`${token.value}`}
      </div>
    );
  };

  // don't let query types be edited
  const isTokenEditable = (token:Token) => {
    if (token.tType == "query") {
      return false;
    }
    return true;
  }

  // allows for tokens to be edited properly
  const handleTokenEdit = (token:Token) => {
    return token.value;
  }


  const handleTokenValuesChange = (
    (newTokenValues:Token[]) => {
      setOpenSuggest(false);
      props.setTokenValues(newTokenValues);
    }
  );
  
  // eat the enter key input for submit if there is currently untokenized
  const creatorKeyDown = (event: React.KeyboardEvent<HTMLInputElement>) => {
    // 
    if (event.key != "Enter") {
      if (event.key == "ArrowDown") {
        document.getElementById('suggestionCommandList')?.focus();
      }
    }
    // if enter key is pressed and preventsubmit is not set, do nothing
    else if (!props.setPreventSubmit) {
      return;
    }
    // if enter key is pressed and there is a set preventsubmit and input in the box, do not submit
    else if (props.tokenInputRef.current != null && props.tokenInputRef.current.getCreatorValue().length > 0) {
      props.setPreventSubmit(true);
    }
  }

  // populates the suggestions. shows nothing if nothing is matched
  const commandList = () => {
    const currList = Object.keys(queryList).filter((qid) => {
      if (qid.toLowerCase().includes(currInput.toLowerCase())) {
        return true;
      }
      return false;
      }).map((qid) => (
        <CommandItem key={qid} value={"~q." + qid}
            onSelect={(value) => {
              props.tokenInputRef.current?.setCreatorValue('');
              handleTokenValuesChange([...props.tokenValues, handleBuildTokenValue(value)])
              setOpenSuggest(false);
              document.querySelector('.token-input-container')?.querySelector(':scope > .token-input-autosized-wrapper')?.querySelector('input')?.focus();
            }}>
            {qid}
          </CommandItem>))
    
    if (currList.length == 0) {
      return;
    }
    return (
      <Command className="border border-solid rounded-md mt-3 min-w-[150px] focus:border-[hsl(var(--primary))] focus:outline-none" id="suggestionCommandList"
        onKeyDown={(event) => {
          if (event.key != "ArrowUp" && event.key != "ArrowDown") {
            document.querySelector('.token-input-container')?.querySelector(':scope > .token-input-autosized-wrapper')?.querySelector('input')?.focus();
          }
          else if (event.key == "ArrowUp" &&
                   document.getElementById('suggestionCommandList')?.querySelector('div')?.children[0].children[0].getAttribute('data-selected') == 'true' ) {
            document.querySelector('.token-input-container')?.querySelector(':scope > .token-input-autosized-wrapper')?.querySelector('input')?.focus();
          }
        }}>
        <CommandList>
          {currList}
        </CommandList>
      </Command>
    )
  }
  
  return(
    <div className="w-full">
      <TokenInput
        ref={props.tokenInputRef}
        tokenValues={props.tokenValues}
        
        separators={["\n","\r","\r\n", "\t"]}
        className="bg-inherit h-full w-full border-border shadow-sm p-2 transition-colors"
        style={{paddingRight:'2.3em', color:'inherit'}}
        placeholder={props.tokenValues.length==0? t('Search') + "..." : "" }
        disableCreateOnBlur
        
        onCreatorKeyDown={creatorKeyDown}
        onTokenValuesChange={handleTokenValuesChange}
        onBuildTokenValue={handleBuildTokenValue}
        onGetTokenDisplayLabel={handleGetTokenDisplayLabel}
        onGetIsTokenEditable={isTokenEditable}
        onGetTokenEditableValue={handleTokenEdit}
        onCreatorFocus={() => {if (!fromDropdown) {setOpenSuggest(true);} else {setFromDropdown(false)}}}
        onCreatorBlur={() => {{setOpenSuggest(false);}}}
        onInputValueChange={(input) => {setCurrInput(input)}}
        >
      </TokenInput>
      
      <Popper id="suggestionListPop" open={openSuggest}
        anchorEl={document.querySelector('.token-input-autosized-wrapper')}
        placement={'bottom-start'}
        onFocus={() => {setOpenSuggest(true); setFromDropdown(true);}}
        onMouseOver={() => {document.getElementById('suggestionCommandList')?.focus();}}
        onMouseLeave={() => {document.querySelector('.token-input-container')?.querySelector(':scope > .token-input-autosized-wrapper')?.querySelector('input')?.focus()}}
        modifiers={[{ name: 'preventOverflow', enabled: true }]}>
          {commandList()}
      </Popper>
      
    </div>
  )
}


// retokenizes a JSON stringified Token[]
export function retokenize(untokens?: string | null) {
  if (untokens != undefined) {
    const test:Token[] = JSON.parse(untokens);
    return test;
  }
  return [];
}