'use client'
import { useState, useEffect, useRef } from "react";
import { Bar, Area, CartesianGrid, XAxis, YAxis, ComposedChart, ReferenceArea, ResponsiveContainer, Scatter, Line } from "recharts"
import { Button } from "@/components/shadcn/button";
import { CardContent } from "@/components/shadcn/card";
import { RadioGroup, RadioGroupItem } from "@/components/shadcn/radio-group";
import { Input } from "@/components/shadcn/input"
import {
  ChartConfig,
  ChartContainer,
  ChartTooltip,
  ChartLegend,
} from "@/components/shadcn/chart";
import { Checkbox } from "@/components/shadcn/checkbox";
import axios from "axios";
import { PanelConfigData, QueryPanelDisplayData } from "@/lib/types";
import moment from "moment";
import { Range } from '@/lib/types';
import { FormLabel, FormControl, FormField, FormItem } from '@/components/shadcn/form';
import { useForm, SubmitHandler } from 'react-hook-form';
import { useTranslation } from 'react-i18next';
import { useDashboardContext } from '@/dashboard/contexts/DashboardContext';
import { CustomizedLegend } from '@/components/charts/components/CustomizedLegend';
import { CustomizedTooltip } from '@/components/charts/components/CustomizedTooltip';
import { FormLayout } from '@/components/charts/components/FormLayout';
import { subQuery } from "@/dashboard/queries/types";
import { Info } from "lucide-react";
import { Tooltip, TooltipContent, TooltipTrigger } from "@/components/shadcn/tooltip";
 
type HistogramChartType = 'bar' | 'line' | 'point';

const chartConfig = {
  logs: {
    label: "Log Entries",
    color: "hsl(var(--secondary))",
  },
  logsLine: {
    label: "Log Entries (Line)",
    color: "hsl(var(--secondary))",
  },
} satisfies ChartConfig

// Bucket data type
export interface BucketEntry {
  key_as_string: string,
  key: number,
  doc_count: number
}

export type HistogramChartValue = 'count' | 'min' | 'mean' | 'max' | 'total';
type HistogramType = 'histogram' | 'total_and_top_senders' | 'log_count_15_minutes';

interface ChartsToShow  {
  [key:string]: boolean,
}

export interface HistogramPanelData extends PanelConfigData {
  mode: HistogramChartValue,
  time_field: string,
  value_field?: string,
  chartsToShow: ChartsToShow,
  isZoomable: boolean,
  multiSeriesStack: boolean,
  multiSeriesStackMode: string,
  panelType: HistogramType,
  granularity: number
}

export interface HistogramPanelDisplayData extends QueryPanelDisplayData {
  panelConfig: HistogramPanelData,
  range: Range,
  setRange: (range: Range) => void
};

export function BaseHistogram(widgetData: HistogramPanelDisplayData) {
  const { t } = useTranslation();
  const [data, setData] = useState<BucketEntry[]>([]);
  const [names, setNames] = useState<subQuery[]>([]);
  const [totalCounts, setCounts] = useState<{[k:string]:number}>({});
  const [refAreaLeft, setRefAreaLeft] = useState<string | null>(null);
  const [refAreaRight, setRefAreaRight] = useState<string | null>(null);
  const [isSelecting, setIsSelecting] = useState(false);
  const chartRef = useRef<HTMLDivElement>(null);
  const [canReset, setCanReset] = useState(false);
  const [originalRange] = useState<Range>(widgetData.range);
  const animation = useDashboardContext().dashboardMetaInfo.isReport ? false : true;
  const [timezone, setTimezone] = useState('UTC');

  const domain = widgetData.range.endTime.valueOf() - widgetData.range.startTime.valueOf();
  var interval = 1000;
  const granularity = widgetData.panelConfig.granularity? widgetData.panelConfig.granularity : 60;
  if (domain <= 60 * 1000) { // Less than 1 minute
    interval = (1000);
  } else if (domain <= 60 * 60 * 1000) { // Less than 1 hour
    interval = (Math.max(5 * 1000, Math.round(domain / granularity))); // 5s OR 30 bars
  } else if (domain <= 24 * 60 * 60 * 1000) { // Less than 1 day
    interval = (Math.max(5 * 60 * 1000, Math.round(domain / granularity))); // 5m OR 30 bars
  } else { // Greater than 1 day
    interval = (Math.max(10 * 60 * 1000, Math.round(domain / granularity))); // 10 min
  }

  const selectedQueries = widgetData.panelConfig.subQueryChoice === 'selected'
  ? widgetData.queries.filter(query =>
      widgetData.panelConfig.subQueries.some(subQuery => subQuery === query.id)
    )
  : widgetData.queries;

  const reqBody = {
    "@timestamp": {
        "from": widgetData.range.startTime.toDate().getTime(),
        "to": widgetData.range.endTime.toDate().getTime(),
    },
    "queries": selectedQueries,
    "filters": widgetData.filters,
    "field": widgetData.panelConfig.time_field,
    "mode": widgetData.panelConfig.mode,
    "interval": interval,
    "multiSeriesStackMode": widgetData.panelConfig.multiSeriesStackMode
  };
  
  useEffect(() => {
    updateData();
  }, [widgetData]);

  useEffect(() => {
    (widgetData.range.reset &&
    (widgetData.range.startTime !== originalRange.startTime ||
      widgetData.range.endTime !== originalRange.endTime)
    ) && setCanReset(true);
  }, [widgetData.range]);

  const updateData = async() => {
    // TODO: When changig this to use useQuery instead, see MapChart widget for setting the query key.
    if (!widgetData.panelConfig.panelType || widgetData.panelConfig.panelType === 'histogram') {
      axios.post('/nagioslogserver/api/dashboards/histogram', reqBody)
      .then((histogramData: any) => {
        setData(histogramData.data.aggregations.aggs.total.buckets);
        const counts = Object.fromEntries(Object.keys(histogramData.data.aggregations.aggs.total.buckets[0])
                              .filter((k:string) => {if (k.startsWith('doc_count')) {return true;} return false;})
                              .map((k) => [k.replace('doc_count', ''), 0]));
        const ks = Object.keys(counts);
        histogramData.data.aggregations.aggs.total.buckets.forEach((bucket: {[key:string]:number}) => {
          ks.forEach((k:string) => {
            if (bucket.hasOwnProperty('doc_count' + k)){
              counts[k] += bucket['doc_count' + k];
            }
        })})
        setCounts(counts);
        setTimezone(histogramData.data.timezone);
      })
      .catch(err => console.error('Error fetching data:', err));
    } else if (widgetData.panelConfig.panelType === 'total_and_top_senders') {
      axios.post('/nagioslogserver/api/dashboards/total_and_top_senders', reqBody)
      .then((histogramData: any) => {
        const ks : string[] = [];
        const counts = Object.fromEntries(histogramData.data.names.map((name: string) => { ks.push(name); return [name, 0] ; }));
        console.log(histogramData.data.buckets[0])
        histogramData.data.buckets.forEach((bucket: {[key:string]:string|number}) => { 
            ks.forEach((k:string) => {
              if (bucket.hasOwnProperty(k)){
                counts[k] += bucket[k];
              }
            })
        })
        setCounts(counts);
        setData(histogramData.data.buckets);
        setNames(histogramData.data.names.map((name: string, index: number) => {
          return {
            id: index.toString(),
            query: name,
            color: histogramData.data.colors[index],
          }
        })
        );
        setTimezone(histogramData.data.timezone);
      })
      .catch(err => console.error('Error fetching data:', err));
    } else if (widgetData.panelConfig.panelType === 'log_count_15_minutes') {

        const now = Date.now(); 
        const twelveHoursAgo = now - 12 * 60 * 60 * 1000; 

        const reqBody = {
          "@timestamp": {
            "from": twelveHoursAgo,
            "to": now,
          },
          "queries": selectedQueries,
          "filters": widgetData.filters,
          "field": widgetData.panelConfig.time_field,
          "mode": widgetData.panelConfig.mode,
          "interval": 900000,
          "multiSeriesStackMode": widgetData.panelConfig.multiSeriesStackMode
        };

        axios.post('/nagioslogserver/api/dashboards/histogram', reqBody)
        .then((histogramData: any) => {
          setData(histogramData.data.aggregations.aggs.total.buckets);
          const counts = Object.fromEntries(Object.keys(histogramData.data.aggregations.aggs.total.buckets[0])
                                .filter((k:string) => {if (k.startsWith('doc_count')) {return true;} return false;})
                                .map((k) => [k.replace('doc_count', ''), 0]));
          const ks = Object.keys(counts);
          histogramData.data.aggregations.aggs.total.buckets.forEach((bucket: {[key:string]:number}) => {
            ks.forEach((k:string) => {
              if (bucket.hasOwnProperty('doc_count' + k)){
                counts[k] += bucket['doc_count' + k];
              }
          })})
          setCounts(counts);
          setTimezone(histogramData.data.timezone);
      })
      .catch(err => console.error('Error fetching data:', err));
    }
  };

  const enableSelecting = (enable: boolean) => {
    setIsSelecting(enable);
  }

  const handleMouseDown = (e: any) => {
    if (e.activeLabel && widgetData.panelConfig.isZoomable) {
      setRefAreaLeft(e.activeLabel);
      enableSelecting(true);
    }
  };

  const handleMouseMove = (e: any) => {
    if (isSelecting && e.activeLabel) {
      setRefAreaRight(e.activeLabel);
    }
  };

  const handleMouseUp = () => {
    if (refAreaLeft && refAreaRight) {
      const [left, right] = [refAreaLeft, refAreaRight].sort();
      const leftMoment= moment.utc(left);
      const rightMoment = moment.utc(right);
      widgetData.setRange({
        startTime: moment(leftMoment),
        endTime: moment(rightMoment),
        value: "custom",
        label: `${leftMoment.format("MMM DD, HH:mm")} - ${rightMoment.format("MMM DD, HH:mm")}`,
        reset: true
      });
    }
    setRefAreaLeft(null);
    setRefAreaRight(null);
    enableSelecting(false);
  };

  const handleReset = () => {
    widgetData.setRange(originalRange);
    setCanReset(false);
  };

  const handleZoomOut = () => {
    // zooms out on either side by half of the current range (doubling the window)
    const zoomDiff = Math.ceil(widgetData.range.endTime.diff(widgetData.range.startTime) / 2000);
    const start = moment.unix(widgetData.range.startTime.unix() - (zoomDiff != 0 ? zoomDiff : 1));
    const end = moment.unix(widgetData.range.endTime.unix() + (zoomDiff != 0 ? zoomDiff : 1));
    const label = `${start.format("MMM DD, HH:mm")} - ${end.format("MMM DD, HH:mm")}`;
    
    // if both ends go over the original size, just reset it
    if (start.isBefore(originalRange.startTime) && end.isAfter(originalRange.endTime)) {
      handleReset();
      return;
    }
    
    // set zoomed out range
    widgetData.setRange({
      startTime: start.isBefore(originalRange.startTime) ? originalRange.startTime : start,
      endTime: end.isAfter(originalRange.endTime) ? originalRange.endTime : end,
      value: 'custom',
      label: label,
      reset: true
    });
  };

  const formatXAxis = (tickItem: string) => {
    const date = new Date(tickItem);
    if (interval <= 60000) {
      return date.toLocaleTimeString([], { timeZone: timezone, hour: '2-digit', minute: '2-digit', second: '2-digit' });
    } else if (interval <= 3600000 ) {
      return date.toLocaleTimeString([], { timeZone: timezone, hour: '2-digit', minute: '2-digit' });
    } else if (interval <= 10800000) {
      return date.toLocaleString([], { timeZone: timezone, hour: '2-digit', minute: '2-digit', month: 'short', day: 'numeric' });
    } else {
      return date.toLocaleString([], { timeZone: timezone, hour: '2-digit', minute: '2-digit', month: 'short', day: 'numeric' });
    }
  }

  const renderChartData = (type: HistogramChartType) => {
    if (widgetData.panelConfig.panelType === 'total_and_top_senders') {
      return renderTotalAndTopSendersChartData();
    } else {
      return renderHistogramChartData(type);
    }
  }

  const renderTotalAndTopSendersChartData = () => {
    const retVal = names.map((name, index) => {
      index = index;
      return (
        <Line dataKey={name.query} fill={name.color} stroke={name.color} isAnimationActive={animation} />
      );
    });

    return retVal;
  }

  const renderHistogramChartData = (type: HistogramChartType) => {
    const showBars = widgetData.panelConfig.chartsToShow?widgetData.panelConfig.chartsToShow['bar']:false;
    const showLines = widgetData.panelConfig.chartsToShow?widgetData.panelConfig.chartsToShow['line']:false;
    const showPoints = widgetData.panelConfig.chartsToShow?widgetData.panelConfig.chartsToShow['point']:false;
    const barStackProps: any = {};
    const lineStackProps: any = {};

    if (widgetData.panelConfig.multiSeriesStack) {
      barStackProps.stackId = 'a';
      lineStackProps.stackId = 'b';
    }
    const retVal = selectedQueries.map((subQuery, index) => {
      const dataKey = "doc_count" + index;
      const radius = (index === selectedQueries.length - 1) ? [2, 2, 0, 0] : 0;
      const fillColor = (subQuery.color || '#1cd98c');
      switch (type) {
        case 'bar':
          return (
            showBars && <Bar dataKey={dataKey} {...barStackProps} fill={fillColor} radius={radius} isAnimationActive={animation} />
          );
        case 'line':
          return (
            showLines && <Area type="monotone" {...lineStackProps} dataKey={dataKey} stroke={fillColor} fill={fillColor} isAnimationActive={animation} />
          );
        case 'point':
          return (
            showPoints && <Scatter dataKey={dataKey} fill={fillColor} isAnimationActive={animation} />
          );
        }
      });
    return retVal;
  }

  const renderComposedChart = () => {
    if (widgetData.panelConfig.panelType === 'total_and_top_senders') {
      return (
        <ComposedChart
          data={data}
          margin={{
            top: 0,
            left: 0,
            bottom: 0,
            right: 0
          }}
          onMouseDown={handleMouseDown}
          onMouseMove={handleMouseMove}
          onMouseUp={handleMouseUp}
          onMouseLeave={handleMouseUp}
        >
          <CartesianGrid vertical={false} />
          <XAxis
            dataKey="key_as_string"
            tickFormatter={formatXAxis}
            tickLine={false}
            tickMargin={10}
            axisLine={false}
            style={{ fill: 'hsl(var(--foreground))'}}
          />
          <YAxis
            tickLine={false}
            tickMargin={10}
            axisLine={false}
            width={100}
            style={{ fill: 'hsl(var(--foreground))'}}
          />
          <ChartTooltip
            content={
              <CustomizedTooltip queries={names}/>
            }
          />
          <ChartLegend
            content={
              <CustomizedLegend queries={names} counts={totalCounts}/>
            }
          />
          {renderChartData('line')}

        </ComposedChart>
      );
    } else {
      return (
        <ComposedChart
                data={data}
                margin={{
                  top: 0,
                  left: 0,
                  bottom: 0,
                  right: 0
                }}
                onMouseDown={handleMouseDown}
                onMouseMove={handleMouseMove}
                onMouseUp={handleMouseUp}
                onMouseLeave={handleMouseUp}
              >
                <CartesianGrid vertical={false} />
                <XAxis
                  dataKey="key_as_string"
                  tickFormatter={formatXAxis}
                  tickLine={false}
                  tickMargin={10}
                  axisLine={false}
                  style={{ fill: 'hsl(var(--foreground))'}}
                />
                <YAxis
                  tickLine={false}
                  tickMargin={10}
                  axisLine={false}
                  style={{ fill: 'hsl(var(--foreground))'}}
                />

                <ChartLegend
                  content={
                    <CustomizedLegend queries={selectedQueries} counts={totalCounts}/>
                  }
                />
                <ChartTooltip
                  content={
                    <CustomizedTooltip queries={selectedQueries}/>
                  }
                />
                {renderChartData('bar')}
                {renderChartData('line')}
                {renderChartData('point')}
                {refAreaLeft && refAreaRight && (
                  <ReferenceArea
                    x1={refAreaLeft}
                    x2={refAreaRight}
                    strokeOpacity={0.3}
                    fill="hsl(var(--primary))"
                    fillOpacity={0.3}
                  />
                )}
              </ComposedChart>
      );
    }
  }

  return (
    <CardContent className="pt-4 h-[calc(100%-100px)]">
      <ChartContainer
        config={chartConfig}
        className="aspect-auto h-full w-full">
        <div className="h-full" ref={chartRef}>
          <div className="flex justify-end sm:mb-4">
              {canReset && 
                <div>
                  <Button variant="outline" onClick={handleZoomOut} className="text-xs sm:text-sm mr-1">
                    {t("Zoom Out")}
                  </Button>
                  <Button variant="outline" onClick={handleReset}  className="text-xs sm:text-sm">
                    {t("Reset")}
                  </Button>
                </div>
              }
          </div>
          <ResponsiveContainer width="100%" height="100%">
            {renderComposedChart()}
          </ResponsiveContainer>
        </div>
      </ChartContainer>
    </CardContent>
  )
}

export interface HistogramWidgetConfigProps {
  config: HistogramPanelData,
  updateWidgetData: (config: PanelConfigData) => void,
}

export function HistogramConfig({config, updateWidgetData}: HistogramWidgetConfigProps) {
  const { t } = useTranslation();

  const form = useForm<HistogramPanelData>({
    defaultValues: {
        title: config.title || t('New Panel'),
        description: config.description || '',
        subQueryChoice: config.subQueryChoice || 'all',
        subQueries: config.subQueries || [],
        mode: config.mode || 'count',
        time_field: config.time_field || '@timestamp',
        value_field: config.value_field || '',
        chartsToShow: config.chartsToShow || {"bar": true, "line": false, "point": false},
        isZoomable: typeof config.isZoomable !== 'undefined' ? config.isZoomable : true,
        multiSeriesStack: typeof config.multiSeriesStack !== 'undefined' ? config.multiSeriesStack : true,
        multiSeriesStackMode: config.multiSeriesStackMode || 'cumulative',
        granularity: config.granularity ?? 60,
    },
  });
  
  form

  const onSubmit: SubmitHandler<HistogramPanelData> = (data) => {
      updateWidgetData(data);
  };

  return (
    <FormLayout form={form} onSubmit={onSubmit} type='dynamic'>
      <div className="grid gap-2">
        <FormField
          control={form.control}
          name="chartsToShow.bar"
          render={({ field }) => (
            <FormItem className="grid grid-cols-3 items-center gap-4 space-y-0">
              <FormLabel className="text-right">{t('Chart Options')}</FormLabel>
              <FormControl>
                <div className="flex items-center space-x-2">
                  <Checkbox id="bar" 
                      checked={field.value}
                      onCheckedChange={field.onChange}
                      defaultChecked={false}
                  />
                  <label
                    htmlFor="bar"
                    className="text-sm font-medium leading-none peer-disabled:cursor-not-allowed peer-disabled:opacity-70"
                  >
                    {t("Bar")}
                  </label>
                </div>
              </FormControl>
            </FormItem>
          )}
        />
        <FormField
            control={form.control}
            name="chartsToShow.line"
            render={({ field }) => (
              <FormItem className="grid grid-cols-3 items-center gap-4 space-y-0">
                &nbsp;
                <FormControl>
                  <div className="flex items-center space-x-2">
                    <Checkbox id="line" 
                        checked={field.value}
                        onCheckedChange={field.onChange}
                        defaultChecked={false}
                    />
                    <label
                      htmlFor="line"
                      className="text-sm font-medium leading-none peer-disabled:cursor-not-allowed peer-disabled:opacity-70"
                    >
                      {t("Lines")}
                    </label>
                  </div>
                </FormControl>
              </FormItem>
            )}
        />
        <FormField
            control={form.control}
            name="chartsToShow.point"
            render={({ field }) => (
              <FormItem className="grid grid-cols-3 items-center gap-4 space-y-0">
                &nbsp;
                <FormControl>
                  <div className="flex items-center space-x-2">
                    <Checkbox id="point" 
                        checked={field.value}
                        onCheckedChange={field.onChange}
                        defaultChecked={false}
                    />
                    <label
                      htmlFor="point"
                      className="text-sm font-medium leading-none peer-disabled:cursor-not-allowed peer-disabled:opacity-70"
                    >
                      {t("Points")}
                    </label>
                  </div>
                </FormControl>
              </FormItem>
            )}
        />
        <FormField
            control={form.control}
            name="isZoomable"
            render={({ field }) => (
              <FormItem className="grid grid-cols-3 items-center gap-4 space-y-0">
                &nbsp;
                <FormControl>
                  <div className="flex items-center space-x-2">
                    <Checkbox id="zoomable" 
                        checked={field.value}
                        onCheckedChange={field.onChange}
                        defaultChecked={false}
                    />
                    <label
                      htmlFor="zoomable"
                      className="text-sm font-medium leading-none peer-disabled:cursor-not-allowed peer-disabled:opacity-70"
                    >
                      {t("Zoomable")}
                    </label>
                  </div>
                </FormControl>
              </FormItem>
            )}
        />
      </div>
      <div className="grid gap-2">
        <FormField
          control={form.control}
          name="multiSeriesStack"
          render={({ field }) => (
            <FormItem className="grid grid-cols-3 items-center gap-4 space-y-0">
              <FormLabel className="text-right">{t('Multi Series')}</FormLabel>
              <FormControl>
                <div className="flex items-center space-x-2">
                  <Checkbox id="stack" 
                      checked={field.value}
                      onCheckedChange={field.onChange}
                      defaultChecked={false}
                  />
                  <label
                    htmlFor="stack"
                    className="text-sm font-medium leading-none peer-disabled:cursor-not-allowed peer-disabled:opacity-70"
                  >
                    {t("Stack")}
                  </label>
                </div>
              </FormControl>
            </FormItem>
          )}
        />
        <FormField
          control={form.control}
          name="multiSeriesStackMode"
          render={({ field }) => (
            <FormItem className="grid grid-cols-3 items-center gap-4 mt-2">
              &nbsp;
              <FormControl>
                <RadioGroup
                  onValueChange={field.onChange}
                  defaultValue={field.value}
                  className="flex flex-col space-y-1 gap-3"
                >
                  <FormItem className="flex items-center space-x-2 space-y-0">
                    <FormControl>
                      <RadioGroupItem value="cumulative" className="text-foreground border-foreground"/>
                    </FormControl>
                    <FormLabel className="font-normal">
                      {t("Cumulative")}
                    </FormLabel>
                  </FormItem>
                  <FormItem className="flex items-center space-x-2 space-y-0">
                    <FormControl>
                      <RadioGroupItem value="percent" className="text-foreground border-foreground"/>
                    </FormControl>
                    <FormLabel className="font-normal">
                      {t("Percent")}
                    </FormLabel>
                  </FormItem>
                </RadioGroup>
              </FormControl>
            </FormItem>
          )}
        />
      </div>
      <FormField
            control={form.control}
            name="granularity"
            render={({ field }) => (
            <FormItem className="grid grid-cols-3 items-center gap-4 space-y-0">
                <FormLabel className="text-right flex gap-1">
                  {t('Max Bar Count')} 
                  <Tooltip>
                    <TooltipTrigger asChild><Info size={16} /></TooltipTrigger>
                    <TooltipContent className="w-64 text-center">
                      {t("Changes the maximum number of bars possible to display on the chart. Larger max values might not be reached due to minimum granularity of different time scales.")}
                    </TooltipContent>
                  </Tooltip>
                </FormLabel>
                <FormControl>
                    <Input type="number" {...field} min={1} max={1000} className="col-span-2 border-border [appearance:textfield] [&::-webkit-outer-spin-button]:appearance-none [&::-webkit-inner-spin-button]:appearance-none" />
                </FormControl>
            </FormItem>
            )}
        />
    </FormLayout>
  )
}