import { PanelConfigData, PanelDisplayData } from "@/lib/types";
import {
    TableCell,
    TableRow,
    Table,
    TableHead,
} from "@/components/shadcn/table"
import { Checkbox } from "@/components/shadcn/checkbox"
import { CardContent } from "@/components/shadcn/card";
import { FormLabel, FormControl, FormField, FormItem } from '@/components/shadcn/form';
import { useForm, SubmitHandler } from 'react-hook-form';
import { FormLayout } from '@/components/charts/components/FormLayout';
import { useTranslation } from 'react-i18next';
import axios from "axios";
import { useEffect, useState } from "react";
import { Check, PriorityHigh } from "@mui/icons-material";
import { Tooltip, TooltipContent, TooltipTrigger } from "@/components/shadcn/tooltip";
import { Portal as TooltipPortal } from "@radix-ui/react-tooltip";

const views = [
  {name: 'unique_host', label: 'Unique Hosts'},
  {name: 'active_alert', label: 'Active Alerts'},
  {name: 'connected_instance', label: 'Instances'},
  {name: 'update_check', label: 'Update Check'}
];

interface SelectViews {
  [key:string]: boolean,
}

export interface dataBoxData extends PanelConfigData {
  showView: SelectViews,
}

export interface DataBoxDisplayData extends PanelDisplayData {
  panelConfig: dataBoxData
}

export interface DataBoxConfigProps {
  config: dataBoxData,
  updateWidgetData: (config: PanelConfigData) => void,
}

export function DataBox(data: DataBoxDisplayData) {
  const { t } = useTranslation();

  // live data object
  const [rowValue, setData] = useState([{name: "unique_host", value: "-1", link: ()=>{window.parent.location = '/nagioslogserver/admin/unique_hosts'}},
                                        {name: "active_alert", value: "-1", link: ()=>{window.parent.location = '/nagioslogserver/admin'}},
                                        {name: "connected_instance", value: "-1", link: undefined},
                                        {name: "update_check", value: "-1", link: undefined}]);

  const updateRow = (name: string, newVal: any, colname= "value") => {
    setData(prevData => prevData.map(col => 
      col.name === name ? { ...col, [colname]: newVal } : col
    ));
  };

  // updates values
  useEffect(() => {
    (rowValue).map((r) => {
      if (data.panelConfig.showView[r.name] === true) {
        retrieveData(r.name);
      }
      else {
        updateRow(r.name, "-1");
      }
    });
  },[data.panelConfig.showView]);

  // api call function, asynchronous
  const retrieveData = async(v: string) => {
    await axios.post('/nagioslogserver/api/dashboards/' + v)
      .then((got: any) => {
        // if this is for update checks, update value and link appropriately
        if (v === "update_check") {
          if (Number(got.data.update_available) === 1) {
            updateRow(v, got.data.update_version.version);
            updateRow(v, ()=>{window.open("https://www.nagios.com/checkforupdates/?product=nagioslogserver&version=" + got.data.current_version)}, "link");
          }
          else {
            updateRow(v, "updated");
            updateRow(v, undefined, "link");
          }
          return;
        }
        // otherwise, just update value
        updateRow(v, got.data);
      })
      .catch(err => console.error('Error fetching data : ', err));
  };

  // make the cell for update, creates the icons on if update is necessary
  function makeUpdateCell(r : string) {
    if (r != "updated")
      return (<Tooltip>
                <TooltipTrigger asChild>
                  <PriorityHigh color="warning" fontSize="large" />
                </TooltipTrigger>
                <TooltipPortal>
                  <TooltipContent>
                    <p>{t("Version " + r + " is the latest version availble")}</p>
                  </TooltipContent>
                </TooltipPortal>
              </Tooltip>)
    else
      return (<Tooltip>
                <TooltipTrigger>
                  <Check color="success" fontSize="large" />
                </TooltipTrigger>
                <TooltipPortal>
                  <TooltipContent>
                    {t("You're running the latest version of Nagios Log Server")}
                  </TooltipContent>
                </TooltipPortal>
              </Tooltip>)
  }

  // returns a table with each head cell containing 
  // another table that shows label and current value
  return (
    <CardContent className="p-4 pt-4">
      <Table>
        {rowValue.map((r) => 
          (data.panelConfig.showView[r.name] === true) && r.value !== "-1" &&
          <TableHead>
            <Table>
              <TableHead className="text-1xl text-center text-secondary-foreground/80 dark:text-inherit" >
                {t(views.find((v) => v.name === r.name)?.label ?? "")}
              </TableHead>
              <TableRow className="border-b-0">
                <TableCell className="text-3xl text-center text-foreground dark:text-inherit" onClick={r.link} style={r.link ? {cursor:'pointer'} : {}}>
                  {r.name !== 'update_check' ? r.value : makeUpdateCell(r.value)}
                </TableCell>
              </TableRow>
            </Table>
          </TableHead>
        )}
      </Table> 
    </CardContent>
  );
}

export function DataBoxConfig({config, updateWidgetData}: DataBoxConfigProps)  {
  const { t } = useTranslation();
  
  const form = useForm<dataBoxData>({
    defaultValues: {
        title: config.title,
        showView: config.showView || {
          unique_host: true,
          active_alert: true,
          interface: true,
        },
    },
  });

  const onSubmit: SubmitHandler<dataBoxData> = (data) => {
      updateWidgetData(data);
  };

  return (
    <FormLayout form={form} onSubmit={onSubmit} type='static'>
      {(views).map((v) => (
        <FormField
          control={form.control}
          name={`showView.${v.name}`}
          render={({ field }) => (
            <FormItem className="grid grid-cols-3 items-center gap-4 space-y-0">
              {v.name === "unique_host" ? <FormLabel className="text-right">{t('Stats to Show')}</FormLabel> : <div>&nbsp;</div> }
              <FormControl>
                <div className="flex items-center space-x-2 col-span-2">
                  <Checkbox id={v.name}
                      checked={field.value}
                      onCheckedChange={field.onChange}
                      defaultChecked={false}
                  />
                  <label
                    htmlFor={v.name}
                    className="text-sm font-medium leading-none peer-disabled:cursor-not-allowed peer-disabled:opacity-70"
                  >
                    {t(v.label)}
                  </label>
                </div>
              </FormControl>
            </FormItem>
          )}
        />
      ))}
    </FormLayout>
  )
}