"use client"

import { PanelConfigData, QueryPanelDisplayData } from '@/lib/types';
import {
    TableCell,
    TableRow,
    Table,
    TableHead,
    TableHeader,
    TableBody
} from "@/components/shadcn/table"
import { CardContent } from "@/components/shadcn/card";
import { ChartContainer } from "@/components/shadcn/chart";
import { Input } from "@/components/shadcn/input";
import {
  Select, 
  SelectContent, 
  SelectItem, 
  SelectTrigger, 
  SelectValue
} from "@/components/shadcn/select";
import { Checkbox } from "@/components/shadcn/checkbox"
import { FormLabel, FormControl, FormField, FormItem } from '@/components/shadcn/form';
import { useForm, SubmitHandler } from 'react-hook-form';
import { useTranslation } from 'react-i18next';
import { getData } from '@/components/charts/queries';
import { useQuery, keepPreviousData } from '@tanstack/react-query';
import { AxiosError } from 'axios';
import { useRef, useState } from 'react';
import { PanelError } from '@/components/charts/components/PanelError';
import { panelError } from '@/components/charts/types';
import { FormLayout } from '@/components/charts/components/FormLayout';

export type StatisticsTypes =
  'count' | 'min' |'max' | 'avg' | 'sum' |
  'variance' | 'std_deviation' | 'sum_of_squares'

const statistics = [
  {name: 'count', label: 'Count'},
  {name: 'min', label: 'Minimum'},
  {name: 'max', label: 'Maximum'},
  {name: 'avg', label: 'Mean'},
  {name: 'sum', label: 'Total'},
  {name: 'variance', label: 'Variance'},
  {name: 'std_deviation', label: 'Standard Deviation'},
  {name: 'sum_of_squares', label: 'Sum of Squares'}
];

interface StatsToShow  {
  [key:string]: boolean,
}
export interface StatsPanelData extends PanelConfigData {
  featuredStat: StatisticsTypes,
  showStatistic: StatsToShow,
  field: string,
  unit: string,
  queryColumnName: string
}

export interface StatsPanelDisplayData extends QueryPanelDisplayData {
  panelConfig: StatsPanelData,
};

export interface StatsWidgetConfigProps {
  config: StatsPanelData,
  updateWidgetData: (config: PanelConfigData) => void,
}

export function Stats(data: StatsPanelDisplayData) {

  const [error, setError] = useState<panelError>();
  const chartRef = useRef<HTMLDivElement>(null);

  const selectedQueries = data.panelConfig.subQueryChoice === 'selected'
  ? data.queries.filter(query =>
      data.panelConfig.subQueries.some(subQuery => subQuery === query.id)
    )
  : data.queries;

  const chartDataSummaryQuery = useQuery<{key: string, value: number}[] | null, AxiosError>({
    queryKey: [data],
    queryFn: async () => {
        const reqBody = {
          "@timestamp": {
            "from": data.range.startTime.toDate().getTime(),
            "to": data.range.endTime.toDate().getTime(),
          },
          "queries": selectedQueries,
          "filters": data.filters,
          "field": data.panelConfig.field,
        };
        const response = await getData('statistics', reqBody);
        if (response.data.error) {
          setError(response.data.error);
          return null;
        }
        setError(undefined);
        const entries = Object.entries(response.data.aggregations.statistics).map(([key, value]) => {
          if (value !== undefined && typeof value == "number" && data.panelConfig.showStatistic[key]) {
            return {
              key: key,
              value: value
            }
          }
        }).filter((item): item is { key: string, value: number } => item !== undefined);
        return entries;
    },
    placeholderData: keepPreviousData
  });

  const FeaturedStat = () => {

    if (!chartDataSummaryQuery.data) {
      return;
    }
    
    var val;
    if (!(val = chartDataSummaryQuery.data.find(statistic => statistic.key === data.panelConfig.featuredStat)?.value)) {
      val = "0"
    }
    
    const featuredStat = {
      name: statistics.find(statistic => statistic.name === data.panelConfig.featuredStat)?.label,
      value: val
    }

    return (
      featuredStat.value &&
      <TableHeader className="bg-background/50 h-1/5">
        <TableRow>
          <TableHead className="text-foreground dark:text-inherit">{featuredStat.name}</TableHead>
          <TableHead className="text-4xl text-right text-foreground dark:text-inherit">{featuredStat.value}</TableHead>
        </TableRow>
      </TableHeader>
    )
  }

  if (!chartDataSummaryQuery.data) {
    return (
      <div>
        {error && <PanelError error={error} />}
      </div>
    );
  }

  return (
    <CardContent className="p-0">
      <ChartContainer
        config={{}}
        className="aspect-square h-full w-full"
        ref={chartRef}
      >
        <Table>
          <FeaturedStat />
          <TableBody>
          {chartDataSummaryQuery.data.map((stat) => 
            stat.key !== data.panelConfig.featuredStat &&
              <TableRow key={stat.key}>
                <TableCell>{statistics.find(statistic => statistic.name === stat.key)?.label}</TableCell>
                <TableCell className="text-right">{stat.value}</TableCell>
              </TableRow>
          )}
          </TableBody>
        </Table>
      </ChartContainer>
    </CardContent>
  )
}

export const StatsConfig = ({config, updateWidgetData}: StatsWidgetConfigProps) => {

  const { t } = useTranslation();

  const form = useForm<StatsPanelData>({
    defaultValues: {
      title: config.title,
      subQueryChoice: config.subQueryChoice || 'all',
      subQueries: config.subQueries || [],
      featuredStat: config.featuredStat || 'count',
      showStatistic: config.showStatistic || {
        count: true,
        min: true,
        max: true,
        avg: true,
        sum: true,
        variance: true,
        std_deviation: true,
        sum_of_squares: true
      },
      field: config.field || '@timestamp',
      unit: config.unit || 'doc',
      queryColumnName: config.queryColumnName,
    },
  });

  const onSubmit: SubmitHandler<StatsPanelData> = (data) => {
      updateWidgetData(data);
  };

  return (
   <FormLayout form={form} onSubmit={onSubmit} type='dynamic'>
      <FormField
        control={form.control}
        name="featuredStat"
        render={({ field }) => (
          <FormItem className="grid grid-cols-3 items-center gap-4 space-y-0">
              <FormLabel className="text-right">{t('Featured Stat')}</FormLabel>
              <FormControl>
                <Select onValueChange={field.onChange} defaultValue="count" value={field.value}>
                  <SelectTrigger className="border-border col-span-2 w-full">
                    <SelectValue placeholder="Select Chart Value" />
                  </SelectTrigger>
                  <SelectContent>
                    {(statistics).map((stat) => (
                      <SelectItem key={stat.name} value={stat.name}>
                        {stat.label}
                      </SelectItem>
                    ))}
                  </SelectContent>
                </Select>
              </FormControl>
          </FormItem>
        )}
      />
      <FormField
        control={form.control}
        name="field"
        render={({ field }) => (
          <FormItem className="grid grid-cols-3 items-center gap-4 space-y-0">
              <FormLabel className="text-right">{t('Field')}</FormLabel>
              <FormControl>
                  <Input {...field} className="col-span-2 border-border" />
              </FormControl>
          </FormItem>
        )}
      />
      <FormField
        control={form.control}
        name="unit"
        render={({ field }) => (
          <FormItem className="grid grid-cols-3 items-center gap-4 space-y-0">
              <FormLabel className="text-right">{t('Unit')}</FormLabel>
              <FormControl>
                  <Input {...field} className="col-span-2 border-border" />
              </FormControl>
          </FormItem>
        )}
      />
      <div className="grid gap-2">
        {statistics.map((stat) => (
          <FormField
            control={form.control}
            name={`showStatistic.${stat.name}`}
            render={({ field }) => (
              <FormItem className="grid grid-cols-3 items-center gap-4 space-y-0">
                {stat.name === "count" ? <FormLabel className="text-right">{('Columns')}</FormLabel> : <div>&nbsp;</div> }
                <FormControl>
                  <div className="flex items-center space-x-2 col-span-2">
                    <Checkbox id={stat.name}
                        checked={field.value}
                        onCheckedChange={field.onChange}
                        defaultChecked={false}
                    />
                    <label
                      htmlFor={stat.name}
                      className="text-sm font-medium leading-none peer-disabled:cursor-not-allowed peer-disabled:opacity-70"
                    >
                      {stat.label}
                    </label>
                  </div>
                </FormControl>
              </FormItem>
            )}
          />
        ))}
      </div>
    </FormLayout>
  )
}