"use client"

import { Pie, PieChart, XAxis, YAxis, BarChart, Bar, CartesianGrid, Legend, Label as ChartLabel, Line, LineChart, LabelList } from 'recharts';
import { useMemo, useState, Fragment, useEffect, ReactNode, ReactElement, JSXElementConstructor, useRef } from "react"
import { Input } from "@/components/shadcn/input";
import {
  Select, 
  SelectContent, 
  SelectItem, 
  SelectTrigger, 
  SelectValue
} from "@/components/shadcn/select";
import { Label } from "@/components/shadcn/label";
import { CardContent } from "@/components/shadcn/card";
import {
  ChartContainer,
  ChartTooltip,
  ChartTooltipContent,
  ChartLegend,
} from "@/components/shadcn/chart"
import {
  Table,
  TableBody,
  TableCell,
  TableHead,
  TableHeader,
  TableRow,
} from "@/components/shadcn/table"
import {
  ColumnDef,
  flexRender,
  getCoreRowModel,
  useReactTable,
} from "@tanstack/react-table"
import { PanelConfigData, QueryPanelDisplayData } from '../../lib/types';
import axios from 'axios';
import { FormLabel, FormControl, FormField, FormItem } from '@/components/shadcn/form';
import { useForm, SubmitHandler } from 'react-hook-form';
import { useTranslation } from 'react-i18next';
import { PanelError } from '@/components/charts/components/PanelError';
import { panelError } from '@/components/charts/types';
import { useDashboardContext } from '@/dashboard/contexts/DashboardContext';
import { FormLayout } from '@/components/charts/components/FormLayout';
import { Checkbox } from '../shadcn/checkbox';

interface TermData {
  key: string,
  logs: number,
  fill: string
}

export type TermsChartType = 'pie' | 'bar' | 'table' | 'opensearchdiskusage' | 'currentindexdiskusage';

export interface TermPanelData extends PanelConfigData {
  chartType: TermsChartType,
  field: string,
  count: number,
  showValue: boolean
}

interface CurrentIndexDiskUsage {
  timestamp: string,
  used: number
}

export interface TermPanelDisplayData extends QueryPanelDisplayData {
  panelConfig: TermPanelData
 }

interface CountEntry {
  key: string,
  doc_count: number
}

export function Term(termProps: TermPanelDisplayData) {
  const [chartData, setChartData] = useState<any[]>([]);
  const [humanUnits, setHumanUnits] = useState<string>();
  const chartRef = useRef<HTMLDivElement>(null);
  const [error, setError] = useState<panelError>();
  const animation = useDashboardContext().dashboardMetaInfo.isReport ? false : true;
  const { filters, setFilters } = useDashboardContext().dashboardInfo;
  const [selectedColor, setSelectedColor] = useState<string>('');
  const [selectedKey, setSelectedKey] = useState<string | null>(null);
  const [diskUsagePercent, setDiskUsagePercent] = useState<{used:number, free:number}>();
  // Dynamic column generation
  const generateColumns = (data: TermData[]): ColumnDef<TermData>[] => {
    if (data.length === 0) {
      return [];
    }

    return Object.keys(data[0]).filter((key) => key != 'fill').map((key) => ({
      accessorKey: key,
      header: (key == 'key') ?
        termProps.panelConfig.field.charAt(0).toUpperCase() + termProps.panelConfig.field.slice(1) :
        key.charAt(0).toUpperCase() + key.slice(1),
      cell: ({ row }: { row:any }) => {
        const value = row.getValue(key) as string
        return (
          <div className="truncate max-w-[500px]" title={value}>
            {value}
          </div>
        )
      },
    }))
  }

  const columns = useMemo(() => generateColumns(chartData), [chartData]);

  const selectedQueries = termProps.panelConfig.subQueryChoice === 'selected'
  ? termProps.queries.filter(query =>
    termProps.panelConfig.subQueries.some(subQuery => subQuery === query.id)
    )
  : termProps.queries;

  const reqBody = { 
    "@timestamp": { 
        "from": termProps.range.startTime.toDate().getTime(), 
        "to": termProps.range.endTime.toDate().getTime(), 
    },
    "queries": selectedQueries,
    "filters": termProps.filters,
    "field": termProps.panelConfig.field,
    "count": termProps.panelConfig.count,
  };

  useEffect(() => {
      updateData();
  }, [termProps]);


  const table = useReactTable({
    data: chartData,
    columns,
    getCoreRowModel: getCoreRowModel(),
    state: {
    },
  });

  const updateData = async() => {
    if (termProps.panelConfig.chartType === 'opensearchdiskusage') {
      axios.post('/nagioslogserver/api/dashboards/opensearchdiskusage', reqBody)
      .then((termData: any) => {
        if (termData.data.error) {
          setError(termData.data.error as panelError);
          setChartData([]);
        } else {
          setError(undefined);
          setChartData(termData.data);
          const percents = {used: 0, free: 0}
          termData.data.forEach((instance: {[key:string]:number}) => {
            percents.free += instance.free;
            percents.used += instance.used;
          })
          const total = (percents.free + percents.used)
          percents.free = (percents.free / total) * 100;
          percents.used = (percents.used / total) * 100;
          setDiskUsagePercent(percents);
        }
      })
      .catch(err => console.error('Error fetching data:', err));
    } else if (termProps.panelConfig.chartType === 'currentindexdiskusage') {
      axios.post('/nagioslogserver/api/dashboards/currentindexdiskusage', reqBody)
      .then((termData: any) => {
        if (termData.data.error) {
          setError(termData.data.error as panelError);
          setChartData([]);
        } else {
          const newData = termData.data.usage_data.map((entry: CurrentIndexDiskUsage) => {
            return entry;
          });
          setError(undefined);
          setHumanUnits(termData.data.usage_data_human_unit);
          setChartData(newData);
        }
      })
      .catch(err => console.error('Error fetching data:', err));
    } else {
      axios.post('/nagioslogserver/api/dashboards/term', reqBody)
      .then((termData: any) => {
        if (termData.data.error) {
          setError(termData.data.error as panelError);
          setChartData([]);
        } else {
          let colorIndex = 0;
          const newData = termData.data.aggregations.terms.buckets.map((entry: CountEntry) => {
          colorIndex = (colorIndex % 5) + 1;
        
          let fillColor = 'hsl(var(--chart-' + colorIndex + '))';
          if (selectedKey && entry.key === selectedKey && selectedColor) {
           fillColor = selectedColor;
          }
          const newEntry: TermData = { key: entry.key, logs: entry.doc_count, fill: fillColor };
          return newEntry;
        });

          setError(undefined);
          setChartData(newData);
        }
      })
      .catch(err => console.error('Error fetching data:', err));
    }
  };

  const handleClick = (e: { key: string; fill: string}) => {
    const field = termProps.panelConfig.field;
    const query = e.key;
 
  if (filters.some(a => (a.field === field && a.mandate === "must" && a.query === query))) {
    setSelectedKey(null);
    setSelectedColor('');
    setFilters(
      filters.filter(a => (a.field !== field || a.mandate !== "must" || a.query !== query))
    );
  } else {
    setSelectedKey(e.key);
    setSelectedColor(e.fill);
    setFilters([...filters, {id: `id-filter${Date.now()}`, field: field, mandate: "must", query: query, active: true}]);
  }

}

  const formatXAxis = (tickItem: number) => {
    const date = new Date(tickItem);
    return date.getHours() + ":" + date.getMinutes();
  }

  const formatToolTip = (tickItem: number) => {
    const date = new Date(tickItem);
    return date.toLocaleString();
  }

  let chart: ReactNode & ReactElement<any, string | JSXElementConstructor<any>>;

  if (termProps.panelConfig.chartType === 'pie') {
    chart = 
      <PieChart>
        <ChartTooltip
          content={
            <ChartTooltipContent className="w-[200px]" />
          }
        />
        <ChartLegend
          content={
            <div className="flex items-center justify-center pt-3 -translate-y-2 flex-wrap gap-2 [&>*]:basis-1/4 [&>*]:justify-center max-h-[75px] overflow-y-auto">
              {
                chartData.map((entry, index) => (
                  <div className="flex items-center gap-1.5 [&>svg]:h-3 [&>svg]:w-3 [&>svg]:text-muted-foreground" key={`query-${index}`}>
                    <div
                      className="h-2 w-2 shrink-0 rounded-[2px]"
                      style={{
                        backgroundColor: entry.fill,
                      }}
                    />
                    <div className="max-w-24 truncate">{entry.key}</div>
                    {termProps.panelConfig.showValue && <div className="max-w-24 truncate">{'(' + entry.logs + ')'}</div>}
                  </div>
                ))
            }
          </div>}   
        />
        <Pie
          data={chartData} dataKey="logs"
          isAnimationActive={animation}
          onClick={(e) => handleClick(e)}
          className="hover:cursor-pointer"
        />
      </PieChart>
  } else if (termProps.panelConfig.chartType === 'bar') {
    chart = 
      <BarChart
        data={chartData}
        margin={{
          top: 20,
          left: 0,
          bottom: 0,
        }}>
        <CartesianGrid vertical={false} />
        <XAxis
          dataKey="key"
          tickFormatter={(value) => value}
          tickLine={false}
          tickMargin={10}
          axisLine={false}
          style={{ fill: 'hsl(var(--foreground))'}}
        />
        <YAxis
          tickLine={false}
          tickMargin={10}
          axisLine={false}
          style={{ fill: 'hsl(var(--foreground))'}}
        />
        <ChartTooltip
          content={
            <ChartTooltipContent className="w-[200px]" />
          }
        />
        <Bar
          dataKey="logs"
          radius={[4, 4, 4, 4]}
          isAnimationActive={animation}
          onClick={(e) => handleClick(e)}
          className="hover:cursor-pointer"
        >
          {termProps.panelConfig.showValue && <LabelList dataKey="logs" position="top" />}
        </Bar>
      </BarChart>
  } else if (termProps.panelConfig.chartType === 'opensearchdiskusage') {
    chart = 
      <BarChart
        data={chartData}
        layout="vertical"
        margin={{
          top: 20,
          left: 0,
          bottom: 25,
        }}>
        <CartesianGrid vertical={true} />
        <YAxis
          type="category"
          dataKey="name"
          tickFormatter={(value) => value}
          tickLine={false}
          tickMargin={10}
          axisLine={false}
          width={100}
          style={{ fill: 'hsl(var(--foreground))'}}
        />
        <XAxis
          type="number"
          tickFormatter={(value) => (value / (1000000000)).toString()}
          tickLine={false}
          tickMargin={10}
          axisLine={false}
          allowDecimals={false}
          style={{ fill: 'hsl(var(--foreground))'}}
        >
          <ChartLabel value="Disk usage in GB" position="bottom" offset={30} style={{ fill: 'hsl(var(--foreground))'}} />
        </XAxis>
        <ChartTooltip
          content={
            <ChartTooltipContent className="w-[200px]" />
          }
        />
        <Bar
          dataKey="used"
          name={termProps.panelConfig.showValue? (diskUsagePercent? "used (" + diskUsagePercent?.used.toPrecision(3) + "%)" : "used") : "used"}
          radius={[4, 4, 4, 4]}
          isAnimationActive={animation}
          stackId="a"
          fill='hsl(var(--foreground)/0.5)'
        />
        <Bar
          dataKey="free"
          name={termProps.panelConfig.showValue? (diskUsagePercent? "free (" + diskUsagePercent?.free.toPrecision(3) + "%)": "free") : "free"}
          radius={[4, 4, 4, 4]}
          stackId="a"
          isAnimationActive={animation}
          fill='hsl(var(--chart-1))'
        />
        <Legend />
      </BarChart>
  } else if (termProps.panelConfig.chartType === 'currentindexdiskusage') {
    chart =
      <LineChart
        data={chartData}
        layout="horizontal"
        margin={{
          top: 20,
          left: 0,
          bottom: 25,
          right: 10
        }}>
        <CartesianGrid vertical={true} />
        <XAxis
          type="category"
          dataKey="timestamp"
          tickFormatter={(value) => formatXAxis(value)}
          tickLine={false}
          tickMargin={10}
          axisLine={false}
          width={100}
          style={{ fill: 'hsl(var(--foreground))'}}
        >
          <ChartLabel value={"Disk usage in " + humanUnits} position="bottom" offset={30} style={{ fill: 'hsl(var(--foreground))'}} />
        </XAxis>
        <YAxis
          type="number"
          tickLine={false}
          tickMargin={10}
          axisLine={false}
          allowDecimals={false}
          style={{ fill: 'hsl(var(--foreground))'}}
        />
        <ChartTooltip
          formatter={(value) => ['Total Size ' + value + ' ' + humanUnits]}
          labelFormatter={formatToolTip}
        />
        <Line
          dataKey="used"
          isAnimationActive={animation}
          fill='hsl(var(--chart-1))'
        />
        <Legend
          formatter={() => 'Total Size(' + humanUnits + ')'}
        />
      </LineChart>
  } else if (termProps.panelConfig.chartType === 'table') {
    chart = 
      <Table>
        <TableHeader>
          {table.getHeaderGroups().map((headerGroup) => (
            <TableRow key={headerGroup.id}>
              {headerGroup.headers.map((header) => {
                return (
                  <TableHead className="text-secondary-foreground/70 dark:text-inherit" key={header.id}>
                    {header.isPlaceholder
                      ? null
                      : flexRender(
                        header.column.columnDef.header,
                        header.getContext()
                      )}
                    </TableHead>
                  )
                })}
              </TableRow>
            ))}
          </TableHeader>
          <TableBody> {
            table.getRowModel()?.rows?.length ? (
              table.getRowModel().rows.map((row) => (
                <Fragment key={row.id}>
                  <TableRow className="cursor-pointer text-start">
                    {row.getVisibleCells().map((cell) => (
                      <TableCell key={cell.id}>
                        {flexRender(cell.column.columnDef.cell, cell.getContext())}
                      </TableCell>
                    ))}
                  </TableRow>
                </Fragment>
              ))
            ) : (
            <TableRow className="cursor-pointer text-start"/>
          )}
          </TableBody>
      </Table>
  } else {
    chart = <Label/>
  }

  return (
    <CardContent className="sm:px-6 pb-0 h-[calc(100%-80px)]">
      {error ? (
        <PanelError error={error} />
      ) : ( 
          <ChartContainer
            config={{}}
            className="aspect-square h-full w-full"
            ref={chartRef}
          >
            {chart}
          </ChartContainer>
      )}
    </CardContent>
  )
}

export interface TermPanelConfigProps {
  config: TermPanelData,
  updateWidgetData: (config: PanelConfigData) => void,
}

export function TermConfig({config, updateWidgetData}: TermPanelConfigProps) {

  const { t } = useTranslation();

  const form = useForm<TermPanelData>({
    defaultValues: {
        title: config.title,
        subQueryChoice: config.subQueryChoice || 'all',
        subQueries: config.subQueries || [],
        chartType: config.chartType || 'bar',
        field: config.field || 'type',
        count: config.count || 10
    },
  });

  const onSubmit: SubmitHandler<TermPanelData> = (data) => {
      updateWidgetData(data);
  };

  return (
    <FormLayout form={form} onSubmit={onSubmit} type='dynamic'>
      <FormField
        control={form.control}
        name="field"
        render={({ field }) => (
            <FormItem className="grid grid-cols-3 items-center gap-4 space-y-0">
                <FormLabel className="text-right">{t('Field')}</FormLabel>
                <FormControl>
                    <Input {...field} className="col-span-2 border-border" />
                </FormControl>
            </FormItem>
        )}
      />
      <FormField
        control={form.control}
        name="count"
        render={({ field }) => (
            <FormItem className="grid grid-cols-3 items-center gap-4 space-y-0">
                <FormLabel className="text-right">{t('Count')}</FormLabel>
                <FormControl>
                    <Input {...field} className="col-span-2 border-border" />
                </FormControl>
            </FormItem>
        )}
      />
      <FormField
        control={form.control}
        name="chartType"
        render={({ field }) => (
          <FormItem className="grid grid-cols-3 items-center gap-4 space-y-0">
              <FormLabel className="text-right">{t('Chart Type')}</FormLabel>
              <FormControl>
                <Select onValueChange={field.onChange} defaultValue="bar" value={field.value}>
                  <SelectTrigger className="border-border col-span-2 w-full">
                    <SelectValue placeholder="Select Chart Value" />
                  </SelectTrigger>
                  <SelectContent>
                    <SelectItem key="bar" value="bar">Bar</SelectItem>
                    <SelectItem key="pie" value="pie">Pie</SelectItem>
                    <SelectItem key="table" value="table">Table</SelectItem>
                    <SelectItem key="opensearchdiskusage" value="opensearchdiskusage">OpenSearch Disk Usage</SelectItem>
                    <SelectItem key="currentindexdiskusage" value="currentindexdiskusage">Disk Usage - Current Index</SelectItem>
                </SelectContent>
                </Select>
              </FormControl>
          </FormItem>
        )}
      />
      {(['bar', 'pie', "opensearchdiskusage"].includes(form.watch("chartType"))) &&
       <FormField
        control={form.control}
        name="showValue"
        defaultValue={config.showValue}
        render={({ field }) => (
          <FormItem className="grid grid-cols-3 items-center gap-4 space-y-0">
              <FormLabel className="text-right">{t('Show Values')}</FormLabel>
              <FormControl>
                <Checkbox defaultChecked={config.showValue} checked={field.value} onCheckedChange={field.onChange} />
              </FormControl>
          </FormItem>
        )}
      />}
    </FormLayout>
  )
}