import { PanelConfigData, PanelDisplayData } from "@/lib/types";
import {
  Select, 
  SelectContent, 
  SelectItem, 
  SelectTrigger, 
  SelectValue
} from "@/components/shadcn/select";
import { CardContent } from "@/components/shadcn/card";
import { TextArea } from "@/components/shadcn/text-area";
import { FormLabel, FormControl, FormField, FormItem } from '@/components/shadcn/form';
import { useForm, SubmitHandler } from 'react-hook-form';
import { useTranslation } from 'react-i18next';
import DOMPurify from "dompurify";
import ReactMarkdown from 'react-markdown';
import remarkGfm from 'remark-gfm';
import { FormLayout } from '@/components/charts/components/FormLayout';

export type TextMode = 'html' | 'markdown' | 'text';
export type fontSize = '6pt' | '7pt' | '8pt' | '10pt' | '12pt' | '14pt' | '16pt' | '18pt' | '20pt' | '22pt' | '24pt' | '26pt' | '28pt' | '30pt' | '32pt' | '36pt' | '42pt' | '48pt' | '52pt' | '56pt' | '60pt' | '64pt' | '72pt' | '80pt' | '96pt';

export interface TextPanelData extends PanelConfigData {
  mode: TextMode,
  content: string,
  style: {
    'font-size': fontSize
  }
}

export interface TextPanelDisplayData extends PanelDisplayData {
  panelConfig: TextPanelData
}

export interface TextWidgetConfigProps {
  config: TextPanelData,
  updateWidgetData: (config: PanelConfigData) => void,
}

export function Text(data: TextPanelDisplayData) {
  return (
    <CardContent className="pt-6">
        {
          data.panelConfig.mode === 'text' ? (
            <div className="leading-none"style={{ fontSize: data.panelConfig.style['font-size'] }}>{data.panelConfig.content}</div>
          ) : data.panelConfig.mode === 'html' ? (
            <div dangerouslySetInnerHTML={{ __html: DOMPurify.sanitize(data.panelConfig.content)}}></div>
          ) : (
            <ReactMarkdown className="prose dark:prose-invert dark:print:prose" remarkPlugins={[remarkGfm]}>{data.panelConfig.content as string}</ReactMarkdown>
          )
        }
    </CardContent>
  );
}

export function TextConfig({config, updateWidgetData}: TextWidgetConfigProps)  {

  const { t } = useTranslation();
  const fontSizes = ['6pt', '7pt', '8pt', '10pt', '12pt', '14pt', '16pt', '18pt', '20pt', '22pt', '24pt', '26pt', '28pt', '30pt', '32pt', '36pt', '42pt', '48pt', '52pt', '56pt', '60pt', '64pt', '72pt', '80pt', '96pt'];

  const form = useForm<TextPanelData>({
    defaultValues: {
        title: config.title || '',
        mode: config.mode || 'text',
        style: config.style || { 'font-size': '16pt' },
        content: config.content || '',
    },
  });

  const onSubmit: SubmitHandler<TextPanelData> = (data) => {
      updateWidgetData(data);
  };

  return (
    <FormLayout form={form} onSubmit={onSubmit} type='static'>
      <FormField
        control={form.control}
        name="mode"
        render={({ field }) => (
          <FormItem className="grid grid-cols-3 items-center gap-4 space-y-0">
              <FormLabel className="text-right">{t('Content Type')}</FormLabel>
              <FormControl>
                <Select value={field.value} onValueChange={field.onChange} defaultValue="html">
                  <SelectTrigger className="border-border col-span-2 w-full">
                    <SelectValue placeholder="Select Content Type" />
                  </SelectTrigger>
                  <SelectContent>
                    <SelectItem key="html" value="html">HTML</SelectItem>
                    <SelectItem key="markdown" value="markdown">Markdown</SelectItem>
                    <SelectItem key="text" value="text">Text</SelectItem>
                  </SelectContent>
                </Select>
              </FormControl>
          </FormItem>
        )}
      />
      {form.watch('mode') === 'text' && (
        <FormField
          control={form.control}
          name="style.font-size"
          render={({ field }) => (
            <FormItem className="grid grid-cols-3 items-center gap-4 space-y-0">
                <FormLabel className="text-right">{t('Font Size')}</FormLabel>
                <FormControl>
                  <Select value={field.value} onValueChange={field.onChange} defaultValue="16pt">
                    <SelectTrigger className="border-border col-span-2 w-full">
                      <SelectValue placeholder="Select Font Size" />
                    </SelectTrigger>
                    <SelectContent>
                      {fontSizes.map((size, index) => (
                        <SelectItem key={index} value={size}>{size}</SelectItem>
                      ))}
                    </SelectContent>
                  </Select>
                </FormControl>
            </FormItem>
          )}
        />
      )}
      <FormField
        control={form.control}
        name="content"
        render={({ field }) => (
            <FormItem className="grid grid-cols-3 items-start gap-4 space-y-0">
                <FormLabel className="text-right mt-2">{t('Content')}</FormLabel>
                <FormControl>
                    <TextArea {...field} className="col-span-2 border-border h-40" placeholder={t('Type your text content here.')} />
                </FormControl>
            </FormItem>
        )}
      />
    </FormLayout>
  )
}