import { Button } from '@/components/shadcn/button';
import { Form, FormLabel, FormControl, FormField, FormItem } from '@/components/shadcn/form';
import { Input } from '@/components/shadcn/input';
import { Select, SelectTrigger, SelectValue, SelectContent, SelectItem } from '@/components/shadcn/select';
import { SheetFooter } from '@/components/shadcn/sheet';
import { Path, SubmitHandler, UseFormReturn} from 'react-hook-form';
import { useTranslation } from 'react-i18next';
import { PanelConfigData } from '@/lib/types';
import { useDashboardContext } from '@/dashboard/contexts/DashboardContext'
import { Checkbox } from '@/components/shadcn/checkbox';

type FormLayoutProps<T extends PanelConfigData> = {
    type: 'static' | 'dynamic';
    children?: React.ReactNode;
    form: UseFormReturn<T>;
    onSubmit: SubmitHandler<T>;
}

export const FormLayout = <T extends PanelConfigData>({ type, children, form, onSubmit }: FormLayoutProps<T>) => {
  
  const subQueries = useDashboardContext().dashboardInfo.subQueries;
  const { t } = useTranslation();

  return (
    <Form {...form}>
      <form onSubmit={form.handleSubmit(onSubmit)} className="grid gap-6 py-6 pr-6">
        <FormField
          control={form.control}
          name={"title" as Path<T>}
          render={({ field }) => (
              <FormItem className="grid grid-cols-3 items-center gap-4 space-y-0">
                  <FormLabel className="text-right">{t('Title')}</FormLabel>
                  <FormControl>
                      <Input {...field} className="col-span-2 border-border" />
                  </FormControl>
              </FormItem>
          )}
        />
        {type === 'dynamic' &&
          <div>
            <FormField
              control={form.control}
              name={"subQueryChoice" as Path<T>}
              render={({ field }) => (
                <FormItem className="grid grid-cols-3 items-center gap-4 space-y-0">
                    <FormLabel className="text-right">{t('Queries')}</FormLabel>
                    <FormControl>
                      <Select value={field.value} onValueChange={field.onChange} defaultValue="all">
                        <SelectTrigger className="border-border col-span-2 w-full">
                          <SelectValue placeholder="Select queries" />
                        </SelectTrigger>
                        <SelectContent>
                          <SelectItem key="all" value="all">All</SelectItem>
                          <SelectItem key="selected" value="selected">Selected</SelectItem>
                        </SelectContent>
                      </Select>
                    </FormControl>
                </FormItem>
              )}
            />
            {form.watch("subQueryChoice" as Path<T>) === 'selected' &&
              <FormField
                control={form.control}
                name={"subQueries" as Path<T>}
                render={() => (
                  <FormItem className="grid grid-cols-3 items-start gap-4 space-y-0 pt-6">
                    &nbsp;
                    <FormControl>
                      <div className="flex items-center grid gap-4">
                          {subQueries.map((subQuery) => (
                            <FormField
                              key={subQuery.id}
                              control={form.control}
                              name={"subQueries" as Path<T>}
                              render={({ field }) => {
                                return (
                                  <FormItem
                                      key={subQuery.id}
                                      className="flex flex-row items-start space-x-3 ml-0 space-y-0"
                                  >
                                    <FormControl>
                                      <Checkbox
                                        checked={field.value?.some((el: string) => el === subQuery.id)}
                                        onCheckedChange={(checked) => {
                                        return checked
                                          ? field.onChange([...field.value, subQuery.id])
                                          : field.onChange(
                                              field.value?.filter(
                                                  (value: string) => value !== subQuery.id
                                              )
                                          )
                                        }}
                                      />
                                    </FormControl>
                                    <FormLabel className="flex items-center flex-nowrap gap-2 font-normal">
                                      <div className="w-3 h-3 rounded-full shrink-0" style={{backgroundColor: subQuery.color}} />
                                      <div className="truncate max-w-[165px]">{subQuery.query}</div>
                                    </FormLabel>
                                  </FormItem>
                                )
                              }}
                            >
                            </FormField>
                          ))}
                      </div>
                    </FormControl>
                  </FormItem>
                )}
              />
            }
          </div>
        }
        {children}
        <SheetFooter className="pt-2">
          <Button type="submit" className="justify-end">
            {t('Save')}
          </Button>
        </SheetFooter>
      </form>
    </Form>
  );
};