'use client'
import { useRef, useState } from "react"

import { Map, Source, Layer } from 'react-map-gl/maplibre';
import type { MapRef, GeoJSONSource } from 'react-map-gl/maplibre';
import { clusterLayer, clusterCountLayer, unclusteredPointLayer } from './layers';
import { PanelConfigData, QueryPanelDisplayData } from "@/lib/types";

import { Input } from "@/components/shadcn/input";

import { FormLabel, FormControl, FormField, FormItem } from '@/components/shadcn/form';
import { useForm, SubmitHandler } from 'react-hook-form';
import { useTranslation } from 'react-i18next';
import { AxiosError } from "axios";
import { panelError } from "../types";

import { useQuery, keepPreviousData } from '@tanstack/react-query';
import { PanelError } from "../components/PanelError";
import { getData } from "../queries";
import { FormLayout } from '@/components/charts/components/FormLayout';

export interface MapPanelData extends PanelConfigData {
    field: string,
    count: number,
    tile_api_url: string,
    tile_api_key: string
}
  
export interface MapPanelDisplayData extends QueryPanelDisplayData {
    panelConfig: MapPanelData
}

type GeoJSONFeature = {
    type: string,
    geometry: {
        type: string,
        coordinates: []
    },
    properties: {
        id: string
    }
};

type MapData = {
    type: string,
    features: GeoJSONFeature[]
}

export function MapChart(widgetData: MapPanelDisplayData) {
    const mapRef = useRef<MapRef>(null);
    const [error, setError] = useState<panelError>();

    const selectedQueries = widgetData.panelConfig.subQueryChoice === 'selected'
    ? widgetData.queries.filter(query =>
        widgetData.panelConfig.subQueries.some(subQuery => subQuery === query.id)
        )
    : widgetData.queries;

    const onClick = async (event: any) => {
        const feature = event.features[0];
        const clusterId = feature.properties.cluster_id;
  
        const mapboxSource = mapRef?.current?.getSource('geoiplogs') as GeoJSONSource;
  
        const zoom = await mapboxSource.getClusterExpansionZoom(clusterId);
        mapRef?.current?.easeTo({
            center: feature.geometry.coordinates,
            zoom}
        );
    }

    const mapDataQuery = useQuery<MapData | null, AxiosError>({
        queryKey: [widgetData.panelConfig,
            widgetData.range,
            selectedQueries,
            widgetData.filters
        ],
        queryFn: async () => {
            const reqBody = {
                "@timestamp": {
                    "from": widgetData.range.startTime.toDate().getTime(),
                    "to": widgetData.range.endTime.toDate().getTime(),
                },
                "queries": selectedQueries,
                "filters": widgetData.filters,
                "field": widgetData.panelConfig.field,
                "count": widgetData.panelConfig.count,
            };

            const mapData = await getData('map', reqBody);
            if (mapData.data.error) {
              setError(mapData.data.error);
              return null;
            }
            setError(undefined);
            const newMapData: MapData = {
                type: "FeatureCollection",
                features: new Array<GeoJSONFeature>()
            }
            const subfields = widgetData.panelConfig.field.split('.');
            mapData.data.hits.hits.forEach((hit: any) => {
                let hitData: any = hit['_source'];
                subfields.forEach((subfield: string) => {
                    hitData = hitData[subfield];
                });
                let hitGeometry: [] = [];
                let lat=0, lon=0;
                for (const [key, value] of Object.entries(hitData)){
                    switch(key) {
                        case 'lat':
                            lat = value as number;
                            break;
                        case 'lon':
                            lon = value as number;
                            break;
                    }
                };
                hitGeometry.push(lon as never);
                hitGeometry.push(lat as never); // Mapbox wants [longitude, latitude]

                const newFeature: GeoJSONFeature = {
                    type: 'Feature',
                    geometry: {
                        type: 'Point',
                        coordinates: hitGeometry
                    },
                    properties: {
                        id: hit._id
                    }
                };

                newMapData.features.push(newFeature);
            });

            return newMapData;
        },
        placeholderData: keepPreviousData
    });

    const keyAddon = widgetData.panelConfig.tile_api_key ? "?key=" + widgetData.panelConfig.tile_api_key : '';
    const mapStyle = widgetData.panelConfig.tile_api_url + keyAddon;

    if (!mapDataQuery.data) {
        return (
          <div>
            {error && <PanelError error={error} />}
          </div>
        );
    }

    return (
        <Map
            initialViewState={{
                latitude: 45,
                longitude: -93.15,
                zoom: 3
            }}
            mapStyle={mapStyle}
            interactiveLayerIds={[clusterLayer.id || '']}
            onClick={onClick}
            ref={mapRef}
        >
            <Source
                id="geoiplogs"
                type="geojson"
                data={mapDataQuery.data}
                cluster={true}
                clusterMaxZoom={14}
                clusterRadius={50}
            >
                <Layer {...clusterLayer} />
                <Layer {...clusterCountLayer} />
                <Layer {...unclusteredPointLayer} />
            </Source>
        </Map>
      );
}

export interface MapPanelConfigProps {
    config: MapPanelData,
    updateWidgetData: (config: PanelConfigData) => void,
}
  
export function MapChartConfig({config, updateWidgetData}: MapPanelConfigProps) {
    const { t } = useTranslation();
  
    const form = useForm<MapPanelData>({
      defaultValues: {
          title: config.title,
          field: config.field || 'source.geo.location',
          count: config.count || 10000,
          subQueryChoice: config.subQueryChoice || 'all',
          subQueries: config.subQueries || [],
          tile_api_url: config.tile_api_url || 'https://demotiles.maplibre.org/style.json',
          tile_api_key: config.tile_api_key,
      },
    });
  
    const onSubmit: SubmitHandler<MapPanelData> = (data) => {
        updateWidgetData(data);
    };
  
    return (
      <FormLayout form={form} onSubmit={onSubmit} type="dynamic">
          <FormField
            control={form.control}
            name="field"
            render={({ field }) => (
                <FormItem className="grid grid-cols-3 items-center gap-4 space-y-0">
                    <FormLabel className="text-right">{t('Field')}</FormLabel>
                    <FormControl>
                        <Input {...field} className="col-span-2 border-border" />
                    </FormControl>
                </FormItem>
            )}
          />
          <FormField
            control={form.control}
            name="count"
            render={({ field }) => (
                <FormItem className="grid grid-cols-3 items-center gap-4 space-y-0">
                    <FormLabel className="text-right">{t('Count')}</FormLabel>
                    <FormControl>
                        <Input {...field} className="col-span-2 border-border" />
                    </FormControl>
                </FormItem>
            )}
          />
          <FormField
            control={form.control}
            name="tile_api_url"
            render={({ field }) => (
                <FormItem className="grid grid-cols-3 items-center gap-4 space-y-0">
                    <FormLabel className="text-right">{t('Tile API URL')}</FormLabel>
                    <FormControl>
                        <Input {...field} className="col-span-2 border-border" />
                    </FormControl>
                </FormItem>
            )}
          />
          <FormField
            control={form.control}
            name="tile_api_key"
            render={({ field }) => (
                <FormItem className="grid grid-cols-3 items-center gap-4 space-y-0">
                    <FormLabel className="text-right">{t('Tile API Key')}</FormLabel>
                    <FormControl>
                        <Input {...field} className="col-span-2 border-border" />
                    </FormControl>
                </FormItem>
            )}
          />
        </FormLayout>
    )
}