import { useState, useEffect } from 'react'
import { TrendingUp } from "lucide-react"
import { CartesianGrid, Line, LineChart, XAxis, ResponsiveContainer } from "recharts"

import {
  Card,
  CardContent,
  CardDescription,
  CardFooter,
  CardHeader,
  CardTitle,
} from "@/components/shadcn/card";
import {
  ChartConfig,
  ChartContainer,
  ChartTooltip,
  ChartTooltipContent,
} from "@/components/shadcn/chart"


interface LogData {
  key_as_string: string;
  doc_count: number;
}

const chartConfig = {
  log_count: {
    label: "Log Count",
    color: "hsl(var(--chart-2))",
  },
} satisfies ChartConfig

export function LineChartDots() {
  const [data, setData] = useState<{ id: string; data: { x: string; y: number }[] }[]>([]);

  useEffect(() => {
    fetch('/nagioslogserver/api/chart/log_count_15_minutes', { method: 'POST' })
      .then(res => res.json())
      .then((data: LogData[]) => {
        const formattedData = data.map(d => ({
          x: new Date(d.key_as_string).toISOString(), // Use full ISO string for datetime
          y: d.doc_count
        }));
        setData([{ id: 'log_count', data: formattedData }]);
      })
      .catch(err => console.error('Error fetching data:', err));
  }, []);

  return (
    <Card className="w-full h-full">
      <CardHeader>
        <CardTitle>Log Count - Last 15 Minutes</CardTitle>
        <CardDescription>Real-time log entry count</CardDescription>
      </CardHeader>
      <CardContent>
        <ChartContainer config={chartConfig}>
          <ResponsiveContainer width="100%" height={300}>
            <LineChart
              accessibilityLayer
              data={data[0]?.data || []}
              margin={{
                left: 12,
                right: 12,
              }}
            >
              <CartesianGrid vertical={false} />
              <XAxis
                dataKey="x"
                tickLine={false}
                axisLine={false}
                tickMargin={8}
                tickFormatter={(value) => new Date(value).toLocaleTimeString()}
              />
              <ChartTooltip
                cursor={false}
                content={<ChartTooltipContent hideLabel />}
              />
              <Line
                dataKey="y"
                type="natural"
                stroke="var(--color-log_count)"
                strokeWidth={2}
                dot={{
                  fill: "var(--color-log_count)",
                }}
                activeDot={{
                  r: 6,
                }}
              />
            </LineChart>
          </ResponsiveContainer>
        </ChartContainer>
      </CardContent>
      <CardFooter className="flex-col items-start gap-2 text-sm">
        <div className="flex gap-2 font-medium leading-none">
          Real-time log count trend <TrendingUp className="h-4 w-4" />
        </div>
        <div className="leading-none text-muted-foreground">
          Showing log entry count for the last 15 minutes
        </div>
      </CardFooter>
    </Card>
  )
}
