"use client"

import { useState, useEffect, useRef, ReactNode, ReactElement, JSXElementConstructor } from 'react';
import { Bar, BarChart, CartesianGrid, XAxis, YAxis } from "recharts";
import { PanelConfigData, PanelDisplayData } from '../../../lib/types';
import { useForm, SubmitHandler } from 'react-hook-form';
import { FormLayout } from '@/components/charts/components/FormLayout';
import { FormLabel, FormControl, FormField, FormItem } from '@/components/shadcn/form';
import { Checkbox } from '@/components/shadcn/checkbox';
import {
  ChartConfig,
  ChartContainer,
  ChartTooltip,
  ChartTooltipContent,
  ChartLegend
} from "@/components/shadcn/chart";
import { CardContent } from "@/components/shadcn/card";
import { useTranslation } from 'react-i18next';


interface LogData {
  twData: Record<string, any>;
  pwData: Record<string, any>;
  weekString: string[];
}

type TwoWeekChartType = 'bar';

export interface TwoWeekPanelData extends PanelConfigData {
  chartType: TwoWeekChartType;
  showValue: boolean;
}
export interface TwoWeekPanelDisplayData extends PanelDisplayData {
  panelConfig: TwoWeekPanelData,
 }

const chartConfig = {
  thisWeek: {
    label: "This Week",
    color: "hsl(var(--chart-2))",
  },
  lastWeek: {
    label: "Last Week",
    color: "hsl(var(--chart-3))",
  },
} satisfies ChartConfig

export function TotalLogEntries2Weeks(panelData: TwoWeekPanelDisplayData) {
  const [data, setData] = useState<{ day: string; thisWeek: number; lastWeek: number }[]>([]);
  const [weekSum, setWeekSum] = useState<{thisWeek: Number; lastWeek: Number}>();
  const chartRef = useRef<HTMLDivElement>(null);

  useEffect(() => {
    fetch('/nagioslogserver/api/chart/total_logs_last_two_weeks', { method: 'POST' })
      .then(res => res.json())
      .then((logData: LogData) => {
        const wsums = {thisWeek: 0, lastWeek: 0};
        const formattedData = logData.weekString.map((day, index) => {
          let key = Object.keys(logData.twData)[index];
          let pwKey = Object.keys(logData.pwData)[index];
          let thisWeek = 0;
          let lastWeek = 0;
          if (key && logData.twData[key]){
            thisWeek = logData.twData[key].doc_count;
            wsums.thisWeek += logData.twData[key].doc_count;
          }
          if (pwKey && logData.pwData[pwKey]){
            lastWeek = logData.pwData[pwKey].doc_count;
            wsums.lastWeek += logData.pwData[pwKey].doc_count;
          }

          let retVal = 
          {
            day: day.replace('<br/>', ' '), // Remove <br/> and replace with space
            thisWeek: thisWeek,
            lastWeek: lastWeek,
          };
          return retVal;
        });
        setWeekSum(wsums);
        setData(formattedData);
      })
      .catch(err => console.error('Error fetching data:', err));
  }, []);

  const handleClick = () => {
    window.parent.location.href = '/nagioslogserver/admin/index_status';
  }

  let chart: ReactNode & ReactElement<any, string | JSXElementConstructor<any>>;

  chart =
    <BarChart
      data={data}
      margin={{
        left: 12,
        right: 12,
      }}>
      <CartesianGrid vertical={false} />
      <XAxis
        dataKey="day"
        tickFormatter={(value) => value}
        tickLine={false}
        tickMargin={10}
        axisLine={false}
        height={60}
        style={{ fill: 'hsl(var(--foreground))'}}
      />
      <YAxis
        tickLine={false}
        tickMargin={10}
        axisLine={false}
        style={{ fill: 'hsl(var(--foreground))'}}
      />
      <ChartTooltip
        content={
          <ChartTooltipContent className="w-[200px]" />
        }
      />
      <Bar
        dataKey="lastWeek"
        radius={[4, 4, 4, 4]}
        fill="hsl(var(--foreground)/0.5)"
        onClick={() => handleClick()}
        className="hover:cursor-pointer"
      />
      <Bar
        dataKey="thisWeek"
        radius={[4, 4, 4, 4]}
        fill="hsl(var(--chart-1))"
        onClick={() => handleClick()}
        className="hover:cursor-pointer"
      />
      { (panelData.panelConfig.showValue && weekSum) &&
        <ChartLegend formatter={(cat:'lastWeek' | 'thisWeek') => cat=='lastWeek' ?
                                  "Last Week (" + weekSum.lastWeek + ")" :
                                  "This Week (" + weekSum.thisWeek + ")"}/>}
    </BarChart>

  return (
    <CardContent className="sm:px-6 pb-0 h-[calc(100%-80px)]">
      <ChartContainer
        config={chartConfig}
        className="aspect-square h-full w-full"
        ref={chartRef}
      >
        {chart}
      </ChartContainer>
    </CardContent>
  )
}

export interface TwoWeekPanelConfigProps {
  config: TwoWeekPanelData,
  updateWidgetData: (config: PanelConfigData) => void,
}

export function TwoWeekConfig({config, updateWidgetData}: TwoWeekPanelConfigProps) {

  const { t } = useTranslation();

  const form = useForm<TwoWeekPanelData>({
    defaultValues: {
        title: t('Total Log Entries Within The Last Two Weeks'),
        subQueryChoice: config.subQueryChoice || 'all',
        subQueries: config.subQueries || [],
        chartType: 'bar',
        showValue: config.showValue
    },
  });

  const onSubmit: SubmitHandler<TwoWeekPanelData> = (data) => {
      updateWidgetData(data);
  };

  return (
    <FormLayout form={form} onSubmit={onSubmit} type='dynamic'>
      <FormField
        control={form.control}
        name="showValue"
        defaultValue={config.showValue}
        render={({ field }) => (
          <FormItem className="grid grid-cols-3 items-center gap-4 space-y-0">
              <FormLabel className="text-right">{t('Show Values')}</FormLabel>
              <FormControl>
                <Checkbox defaultChecked={config.showValue} checked={field.value} onCheckedChange={field.onChange} />
              </FormControl>
          </FormItem>
        )}
      />
    </FormLayout>
  )
}