"use client"

import { TrendingUp } from "lucide-react"
import { LabelList, Pie, PieChart } from "recharts"
import { useState, useEffect } from "react"
import {
  Select, 
  SelectContent, 
  SelectItem, 
  SelectTrigger, 
  SelectValue
} from "@/components/shadcn/select";

import {
  Card,
  CardContent,
  CardDescription,
  CardFooter,
  CardHeader,
  CardTitle,
} from "@/components/shadcn/card";
import {
  ChartConfig,
  ChartContainer,
  ChartTooltip,
  ChartTooltipContent,
} from "@/components/shadcn/chart"

const chartConfig = {
  count: {
    label: "Log Entries",
  },
  syslog: {
    label: "Syslog",
    color: "hsl(var(--chart-1))",
  },
  ssh: {
    label: "SSH",
    color: "hsl(var(--chart-2))",
  },
  apache: {
    label: "Apache",
    color: "hsl(var(--chart-3))",
  },
  mysql: {
    label: "MySQL",
    color: "hsl(var(--chart-4))",
  },
  other: {
    label: "Other",
    color: "hsl(var(--chart-5))",
  },
} satisfies ChartConfig

const intervalOptions = [
  { value: '1s', label: '1 Second' },
  { value: '10m', label: '10 Minutes (Auto)' },
  { value: '5m', label: '5 Minutes' },
  { value: '30m', label: '30 Minutes' },
  { value: '1h', label: '1 Hour' },
  { value: '3h', label: '3 Hours' },
]

export function PieChartLabel() {
  const [data, setData] = useState<any[]>([]);
  const [interval, setInterval] = useState('10m');
  const [startTime, setStartTime] = useState<string | undefined>(undefined);
  const [endTime, setEndTime] = useState<string | undefined>(undefined);

  const fetchData = (start?: string, end?: string) => {
    const params = new URLSearchParams({
      interval: interval,
      ...(start && { start_time: start }),
      ...(end && { end_time: end }),
    });

    fetch('/nagioslogserver/api/chart/terms_data', {
      method: 'POST',
      headers: {
        'Content-Type': 'application/x-www-form-urlencoded',
      },
      body: params.toString(),
    })
      .then(response => response.json())
      .then((termData: { logType: string, count: number }[]) => {
        setData(termData);
//        setOriginalData(formattedData);
        setStartTime(start);
        setEndTime(end);
      })
      .catch(err => {
        console.error('Error fetching data:', err);
      });
  };

  useEffect(() => {
    fetchData(startTime, endTime);
  }, [interval]);

  const handleIntervalChange = (value: string) => {
    setInterval(value);
  };

  return (
    <Card className="flex flex-col w-full h-full">
      <CardHeader className="items-center pb-0">
        <CardTitle>Log Type Distribution</CardTitle>
        <CardDescription>Last 24 Hours</CardDescription>
      </CardHeader>
      <div className="flex items-center px-6 py-4 sm:px-8">
          <Select value={interval} onValueChange={handleIntervalChange}>
            <SelectTrigger className="w-[180px]">
              <SelectValue placeholder="Select interval" />
            </SelectTrigger>
            <SelectContent>
              {intervalOptions.map((option) => (
                <SelectItem key={option.value} value={option.value}>
                  {option.label}
                </SelectItem>
              ))}
            </SelectContent>
          </Select>
        </div>
      <CardContent className="flex-1 pb-0">
        <ChartContainer
          config={chartConfig}
          className="mx-auto aspect-square max-h-[250px]"
        >
          <PieChart>
            <ChartTooltip
              content={<ChartTooltipContent nameKey="count" hideLabel />}
            />
            <Pie data={data} dataKey="count">
              <LabelList
                dataKey="logType"
                className="fill-background"
                stroke="none"
                fontSize={12}
                formatter={(value: keyof typeof chartConfig) =>
                  chartConfig[value]?.label
                }
              />
            </Pie>
          </PieChart>
        </ChartContainer>
      </CardContent>
      <CardFooter className="flex-col gap-2 text-sm">
        <div className="flex items-center gap-2 font-medium leading-none">
          SSH logs up by 5.2% this hour <TrendingUp className="h-4 w-4" />
        </div>
        <div className="leading-none text-muted-foreground">
          Showing log type distribution for the last 24 hours
        </div>
      </CardFooter>
    </Card>
  )
}
