// @ts-nocheck
'use client'
import { useState, useEffect, useMemo, useRef } from "react"
import { Bar, Area, CartesianGrid, XAxis, YAxis, ComposedChart, ReferenceArea, ResponsiveContainer } from "recharts"
import { Button } from "@/components/shadcn/button"
import {
  Card,
  CardContent,
  CardDescription,
  CardHeader,
} from "@/components/shadcn/card";
import {
  ChartConfig,
  ChartContainer,
  ChartTooltip,
  ChartTooltipContent,
  ChartLegend,
  ChartLegendContent,
} from "@/components/shadcn/chart"

import {
  Switch,
} from "@/components/shadcn/switch"
import {
  Select, 
  SelectContent, 
  SelectItem, 
  SelectTrigger, 
  SelectValue
} from "@/components/shadcn/select";
import { Label } from "@/components/shadcn/label"
import { Input } from "@/components/shadcn/input"

export const description = "A histogram of log entries"

const chartConfig = {
  logs: {
    label: "Log Entries",
    color: "hsl(var(--chart-2))",
  },
  logsLine: {
    label: "Log Entries (Line)",
    color: "hsl(var(--chart-2))",
  },
} satisfies ChartConfig

const intervalOptions = [
  { value: '1s', label: '1 Second' },
  { value: '10m', label: '10 Minutes (Auto)' },
  { value: '5m', label: '5 Minutes' },
  { value: '30m', label: '30 Minutes' },
  { value: '1h', label: '1 Hour' },
  { value: '3h', label: '3 Hours' },
]

export function SearchAGram() {
  const [data, setData] = useState<any[]>([]);
  const [interval, setInterval] = useState('10m');
  const [showBars, setShowBars] = useState(true);
  const [showLines, setShowLines] = useState(false);
  const [refAreaLeft, setRefAreaLeft] = useState<string | null>(null);
  const [refAreaRight, setRefAreaRight] = useState<string | null>(null);
  const [startTime, setStartTime] = useState<string | null>(null);
  const [endTime, setEndTime] = useState<string | null>(null);
  const [originalData, setOriginalData] = useState<any[]>([]);
  const [isSelecting, setIsSelecting] = useState(false);
  const chartRef = useRef<HTMLDivElement>(null);

  const fetchData = (start?: string, end?: string) => {
    const params = new URLSearchParams({
      interval: interval,
      ...(start && { start_time: start }),
      ...(end && { end_time: end }),
    });

    fetch('/nagioslogserver/api/chart/dashboard_data', {
      method: 'POST',
      headers: {
        'Content-Type': 'application/x-www-form-urlencoded',
      },
      body: params.toString(),
    })
      .then(response => response.json())
      .then((histogramData: { time: number, count: number }[]) => {
        const formattedData = histogramData.map(entry => ({
          date: new Date(entry.time).toISOString(),
          logs: entry.count
        }));
        setData(formattedData);
        setOriginalData(formattedData);
        setStartTime(formattedData[0]?.date || null);
        setEndTime(formattedData[formattedData.length - 1]?.date || null);
      })
      .catch(err => {
        console.error('Error fetching data:', err);
      });
  };

  useEffect(() => {
    fetchData(startTime, endTime);
  }, [interval]);

  const zoomedData = useMemo(() => {
    if (!startTime || !endTime) {
      return data;
    }

    const dataPointsInRange = originalData.filter(
      (dataPoint) => dataPoint.date >= startTime && dataPoint.date <= endTime
    );

    // If there's only one data point or less, return at least two points
    if (dataPointsInRange.length <= 1) {
      const index = originalData.findIndex(d => d.date === dataPointsInRange[0]?.date);
      if (index !== -1 && index < originalData.length - 1) {
        return [originalData[index], originalData[index + 1]];
      } else if (index === originalData.length - 1 && index > 0) {
        return [originalData[index - 1], originalData[index]];
      }
    }

    return dataPointsInRange;
  }, [startTime, endTime, originalData, data]);

  const total = useMemo(
    () => zoomedData.reduce((acc, curr) => acc + curr.logs, 0),
    [zoomedData]
  )

  const handleIntervalChange = (value: string) => {
    setInterval(value);
  };

  const handleMouseDown = (e: any) => {
    if (e.activeLabel) {
      setRefAreaLeft(e.activeLabel);
      setIsSelecting(true);
    }
  };

  const handleMouseMove = (e: any) => {
    if (isSelecting && e.activeLabel) {
      setRefAreaRight(e.activeLabel);
    }
  };

  const handleMouseUp = () => {
    if (refAreaLeft && refAreaRight) {
      const [left, right] = [refAreaLeft, refAreaRight].sort();
      setStartTime(left);
      setEndTime(right);
      fetchData(left, right);
    }
    setRefAreaLeft(null);
    setRefAreaRight(null);
    setIsSelecting(false);
  };

  const handleReset = () => {
    setStartTime(originalData[0]?.date || null);
    setEndTime(originalData[originalData.length - 1]?.date || null);
    fetchData();
  };

  const formatXAxis = (tickItem: string) => {
    const date = new Date(tickItem);
    switch(interval) {
      case '1s':
        return date.toLocaleTimeString([], { hour: '2-digit', minute: '2-digit', second: '2-digit' });
      case '5m':
      case '10m':
        return date.toLocaleTimeString([], { hour: '2-digit', minute: '2-digit' });
      case '30m':
      case '1h':
        return date.toLocaleTimeString([], { hour: '2-digit', minute: '2-digit' });
      case '3h':
        return date.toLocaleString([], { hour: '2-digit', minute: '2-digit', month: 'short', day: 'numeric' });
      default:
        return date.toLocaleTimeString();
    }
  };

  return (
    <Card className="w-full h-full">
      <CardHeader className="flex flex-col items-stretch space-y-0 border-b p-0 sm:flex-row">
        <div className="flex items-center justify-between w-full px-6 py-4">
          <div className="w-1/2">
            <CardDescription>
              <Input
                placeholder="Enter search query"
                className="w-full"
              />
            </CardDescription>
          </div>
          <div className="flex items-center space-x-4">
            <Select value={interval} onValueChange={handleIntervalChange}>
              <SelectTrigger className="w-[180px]">
                <SelectValue placeholder="Select interval" />
              </SelectTrigger>
              <SelectContent>
                {intervalOptions.map((option) => (
                  <SelectItem key={option.value} value={option.value}>
                    {option.label}
                  </SelectItem>
                ))}
              </SelectContent>
            </Select>
            <div className="flex items-center space-x-4">
              <div className="flex items-center space-x-2">
                <Switch
                  id="show-bars"
                  checked={showBars}
                  onCheckedChange={setShowBars}
                />
                <Label htmlFor="show-bars">Bars</Label>
              </div>
              <div className="flex items-center space-x-2">
                <Switch
                  id="show-lines"
                  checked={showLines}
                  onCheckedChange={setShowLines}
                />
                <Label htmlFor="show-lines">Lines</Label>
              </div>
            </div>
          </div>
        </div>
      </CardHeader>
      <CardContent className="px-2 sm:p-6">
        <ChartContainer
          config={chartConfig}
          className="aspect-auto h-[250px] w-full"
        >
          <div className="h-full" ref={chartRef}>
            <div className="flex justify-end my-2 sm:mb-4">
              <Button variant="outline" onClick={handleReset} disabled={!startTime && !endTime} className="text-xs sm:text-sm">
                Reset
              </Button>
            </div>
            <ResponsiveContainer width="100%" height="100%">
              <ComposedChart
                data={zoomedData}
                margin={{
                  top: 20,
                  right: 30,
                  left: 20,
                  bottom: 5,
                }}
                onMouseDown={handleMouseDown}
                onMouseMove={handleMouseMove}
                onMouseUp={handleMouseUp}
                onMouseLeave={handleMouseUp}
              >
                <CartesianGrid strokeDasharray="3 3" />
                <XAxis
                  dataKey="date"
                  tickFormatter={formatXAxis}
                  style={{ fontSize: '10px', userSelect: 'none' }}
                />
                <YAxis style={{ fontSize: '10px', userSelect: 'none' }} />
                <ChartTooltip
                  content={
                    <ChartTooltipContent
                      className="w-[200px]"
                      nameKey="logs"
                      labelFormatter={(value) => new Date(value).toLocaleString()}
                    />
                  }
                />
                <ChartLegend content={<ChartLegendContent />} />
                {showBars && <Bar dataKey="logs" fill={chartConfig.logs.color} />}
                {showLines && <Area type="monotone" dataKey="logs" stroke={chartConfig.logsLine.color} fill={chartConfig.logsLine.color} />}
                {refAreaLeft && refAreaRight && (
                  <ReferenceArea
                    x1={refAreaLeft}
                    x2={refAreaRight}
                    strokeOpacity={0.3}
                    fill="hsl(var(--primary))"
                    fillOpacity={0.3}
                  />
                )}
              </ComposedChart>
            </ResponsiveContainer>
          </div>
        </ChartContainer>
      </CardContent>
    </Card>
  )
}