"use client"

import * as React from "react"
import { Bar, BarChart, CartesianGrid, XAxis, YAxis, Tooltip, Legend } from "recharts"

import {
  Card,
  CardContent,
  CardDescription,
  CardHeader,
  CardTitle,
} from "@/components/shadcn/card";
import {
  ChartConfig,
  ChartContainer,
  ChartTooltipContent,
} from "@/components/shadcn/chart"


export const description = "A histogram of log entries"

async function* generateMockLogData() {
  const startDate = new Date("2024-01-01")
  const totalLogs = 10000

  for (let i = 0; i < 96; i++) { // 96 intervals of 15 minutes in a day
    const date = new Date(startDate.getTime() + i * 15 * 60 * 1000)
    
    // Generate random log counts for each interval
    const intervalLogs = Math.floor(Math.random() * (totalLogs / 48)) // Up to 2x average
    const error = Math.floor(Math.random() * (intervalLogs * 0.2)) // Up to 20% errors
    const warning = Math.floor(Math.random() * (intervalLogs * 0.4)) // Up to 40% warnings
    const info = intervalLogs - error - warning // Remaining logs are info

    yield {
      date: date.toISOString().split('T')[1].substring(0, 5), // Format as HH:MM
      error,
      warning,
      info,
    }

    // Simulate delay between data points
    await new Promise(resolve => setTimeout(resolve, 10))
  }
}

const chartConfig = {
  logs: {
    label: "Log Entries",
    color: "hsl(var(--foreground))",
  },
  error: {
    label: "Error",
    color: "hsl(var(--destructive))",
  },
  warning: {
    label: "Warning",
    color: "hsl(var(--chart-3))",
  },
  info: {
    label: "Info",
    color: "hsl(var(--chart-2))",
  },
} satisfies ChartConfig

export function StreamHistogram() {
  const [activeChart, setActiveChart] =
    React.useState<keyof typeof chartConfig | "all">("all")
  const [chartData, setChartData] = React.useState<any[]>([])
  const [total, setTotal] = React.useState({
    error: 0,
    warning: 0,
    info: 0,
  })

  React.useEffect(() => {
    let isMounted = true
    const fetchData = async () => {
      for await (const dataPoint of generateMockLogData()) {
        if (!isMounted) break
        setChartData(prevData => [...prevData, dataPoint])
        setTotal(prevTotal => ({
          error: prevTotal.error + dataPoint.error,
          warning: prevTotal.warning + dataPoint.warning,
          info: prevTotal.info + dataPoint.info,
        }))
      }
    }
    fetchData()
    return () => { isMounted = false }
  }, [])

  return (
    <Card>
      <CardHeader className="flex flex-col items-stretch space-y-0 border-b p-0 sm:flex-row">
        <div className="flex flex-1 flex-col justify-center gap-1 px-6 py-5 sm:py-6">
          <CardTitle>Log Histogram</CardTitle>
          <CardDescription>
            Showing log entries for the last 24 hours
          </CardDescription>
        </div>
        <div className="flex">
          {["all", "error", "warning", "info"].map((key) => {
            const chart = key as keyof typeof chartConfig | "all"
            return (
              <button
                key={chart}
                data-active={activeChart === chart}
                className="relative z-30 flex flex-1 flex-col justify-center gap-1 border-t px-6 py-4 text-left even:border-l data-[active=true]:bg-muted/50 sm:border-l sm:border-t-0 sm:px-8 sm:py-6"
                onClick={() => setActiveChart(chart)}
              >
                <span className="text-xs text-muted-foreground">
                  {chart === "all" ? "All" : chartConfig[chart as keyof typeof chartConfig].label}
                </span>
                <span className="text-lg font-bold leading-none sm:text-3xl">
                  {chart === "all" 
                    ? Object.values(total).reduce((acc, curr) => acc + curr, 0).toLocaleString()
                    : total[chart as keyof typeof total].toLocaleString()}
                </span>
              </button>
            )
          })}
        </div>
      </CardHeader>
      <CardContent className="px-2 sm:p-6">
        <ChartContainer
          config={chartConfig}
          className="aspect-auto h-[250px] w-full"
        >
          <BarChart
            data={chartData}
            margin={{
              top: 20,
              right: 30,
              left: 20,
              bottom: 5,
            }}
          >
            <CartesianGrid strokeDasharray="3 3" />
            <XAxis
              dataKey="date"
              tickFormatter={(value) => new Date(value).toLocaleDateString()}
              style={{ fill: 'hsl(var(--foreground))'}}
            />
            <YAxis style={{ fill: 'hsl(var(--foreground))'}} />
            <Tooltip
              content={
                <ChartTooltipContent
                  className="w-[200px]"
                  nameKey="logs"
                  labelFormatter={(value) => new Date(value).toLocaleDateString()}
                />
              }
            />
            <Legend />
            {activeChart === "all" ? (
              <>
                <Bar dataKey="error" stackId="a" fill={chartConfig.error.color} />
                <Bar dataKey="warning" stackId="a" fill={chartConfig.warning.color} />
                <Bar dataKey="info" stackId="a" fill={chartConfig.info.color} />
              </>
            ) : (
              <Bar dataKey={activeChart} fill={chartConfig[activeChart as keyof typeof chartConfig].color} />
            )}
          </BarChart>
        </ChartContainer>
      </CardContent>
    </Card>
  )
}
