import { useState, useEffect } from "react"
import { Table } from "@tanstack/react-table"
import { ToggleGroup, ToggleGroupItem } from "@/components/shadcn/toggle-group"

interface ColumnToggleProps<TData> {
    table: Table<TData>
    defaultVisibleColumns: string[]
}

export function ColumnToggle<TData>({ table, defaultVisibleColumns }: ColumnToggleProps<TData>) {
    const [visibleColumns, setVisibleColumns] = useState<string[]>(defaultVisibleColumns)

    useEffect(() => {
        table.getAllColumns().forEach((column) => {
            if (column.getCanHide()) {
                column.toggleVisibility(visibleColumns.includes(column.id))
            }
        })
    }, [table, visibleColumns])

    const toggleColumn = (columnId: string) => {
        setVisibleColumns((prev) => 
            prev.includes(columnId)
                ? prev.filter((id) => id !== columnId)
                : [...prev, columnId]
        )
    }

    return (
        <ToggleGroup type="multiple" className="ml-auto flex-wrap">
            {table
                .getAllColumns()
                .filter((column) => column.getCanHide())
                .map((column) => {
                    return (
                        <ToggleGroupItem
                            key={column.id}
                            value={column.id}
                            aria-label={`Toggle ${column.id}`}
                            data-state={visibleColumns.includes(column.id) ? "on" : "off"}
                            onClick={() => toggleColumn(column.id)}
                        >
                            {column.id}
                        </ToggleGroupItem>
                    )
                })}
        </ToggleGroup>
    )
}