import { useEffect, useRef, useState } from "react"
import {
    Sheet,
    SheetContent,
    SheetHeader,
    SheetTitle,
} from "@/components/shadcn/sheet"
import {
    Tabs,
    TabsContent,
    TabsList,
    TabsTrigger
} from "@/components/shadcn/tabs"
import {
    Table,
    TableBody,
    TableCell,
    TableHead,
    TableHeader,
    TableRow,
} from "@/components/shadcn/table"
import { Toggle } from "@/components/shadcn/toggle"
import {
    TooltipProvider,
    Tooltip,
    TooltipContent,
    TooltipTrigger,
} from "@/components/shadcn/tooltip"
import {
    Accordion,
    AccordionContent,
    AccordionItem,
    AccordionTrigger
} from "@/components/shadcn/accordion"
import {
    Card,
    CardContent,
    CardHeader
} from "@/components/shadcn/card"
import { ChevronDown } from "lucide-react";
import { ScrollArea } from "@/components/shadcn/scroll-area"
import { Input } from "@/components/shadcn/input";
import FilterAltIcon from '@mui/icons-material/FilterAlt';
import FilterAltOffIcon from '@mui/icons-material/FilterAltOff';
import { useDashboardContext } from '@/dashboard/contexts/DashboardContext';
import { filter } from '@/dashboard/filters/types';
import { cn } from "@/lib/utils";
import { useTranslation } from 'react-i18next';
import { t } from "i18next"
import { VisibilityState } from "@tanstack/react-table"
import { Checkbox } from "@/components/shadcn/checkbox"
import { tableGetLocaleTimestamp, highlightValue } from "./log-table";

interface LogEntry {
    [key: string]: any
}

interface LogEntrySheetProps {
    isOpen: boolean
    onOpenChange: (open: boolean) => void
    logEntry: LogEntry | null
    columnVisibility: VisibilityState
    setVisibility: React.Dispatch<React.SetStateAction<VisibilityState>>
    termRegex?:RegExp
}


const FilterToggle = ({ filter } : { filter: filter }) => {
    
    const { t } = useTranslation();
    const { filters, setFilters } = useDashboardContext().dashboardInfo;
    const onToggle = (filter: filter) => {
        if (filters.some(a => (a.field === filter.field && a.mandate === filter.mandate && a.query === filter.query))) {
            setFilters(
                filters.filter(a => (a.field !== filter.field || a.mandate !== filter.mandate || a.query !== filter.query))
            );
        } else {
            setFilters([...filters, filter]);
        }
    }
    
    return (
        <TooltipProvider delayDuration={10} disableHoverableContent>
            <Tooltip>
                <TooltipTrigger asChild>
                    <Toggle
                        pressed={filters.some(a => (a.field === filter.field && a.mandate === filter.mandate && a.query === filter.query))}
                        size="sm"
                        onPressedChange={() => onToggle(filter)}
                        key={filter.id + filter.mandate}
                        className={cn("opacity-0 transition-opacity group-hover:opacity-100", filters.some(a => (a.field === filter.field && a.mandate === filter.mandate && a.query === filter.query)) ? "bg-accent opacity-100" : "bg-transparent")}
                    >
                        {filter.mandate === "must" ? <FilterAltIcon fontSize="small"/> : <FilterAltOffIcon fontSize="small"/>}
                    </Toggle>
                </TooltipTrigger>
                <TooltipContent>
                    <p>{filter.mandate === "must" ? t("Toggle filter to match this value") : t("Toggle filter to NOT match this value")}</p>
                </TooltipContent>
            </Tooltip>
        </TooltipProvider>
    )
}

export function LogEntrySheet({ isOpen, onOpenChange, logEntry, columnVisibility, setVisibility, termRegex }: LogEntrySheetProps) {
    const [jsonSearch, setJsonSearch] = useState("");

    const filterJsonElements = (jsonObj: any, searchTerm: string) => {
        return Object.entries(jsonObj).filter(([key, value]) => 
            key.toLowerCase().includes(searchTerm.toLowerCase()) || 
            JSON.stringify(value).toLowerCase().includes(searchTerm.toLowerCase())
        );
    };

    const EllipsisTooltip = ({ value } : { value: string }) => {
        const [overflowing, setOverflowing] = useState(false);
        const elementRef = useRef<HTMLDivElement>(null);
        useEffect(() => {
            elementRef.current && setOverflowing(elementRef.current.scrollWidth > elementRef.current.clientWidth);
        }, []);
        return (
            <div className={cn("w-fit", !overflowing ? "cursor-text" : "cursor-default")}>
                <Tooltip>
                    <TooltipTrigger asChild className={cn(!overflowing && "pointer-events-none")}>
                        <div ref={elementRef} className="max-w-[460px] overflow-hidden text-ellipsis">
                            {highlightValue(value, termRegex)}
                        </div>
                    </TooltipTrigger>
                    <TooltipContent className="cursor-text">
                        {value}
                    </TooltipContent>
                </Tooltip>
            </div>
        )
    }

    return (
        <Sheet open={isOpen} onOpenChange={onOpenChange}>
            <SheetContent className="sm:max-w-[800px] p-0">
                <div className="p-4 h-full">
                    <SheetHeader className="pb-4">
                        <SheetTitle>Log Entry Details</SheetTitle>
                    </SheetHeader>
                    <ScrollArea className="h-[calc(100%-60px)]">
                        {logEntry && <Card>
                            <CardHeader>
                                <div className="flex flex-row justify-between">
                                    <div >{highlightValue(tableGetLocaleTimestamp(logEntry["@timestamp"]) + ' ' + (new Date().toLocaleTimeString(undefined,{timeZoneName:'short', hour12:false}).split(' ')[1]??''), termRegex)}</div>
                                    <div >{highlightValue(t("type: ") + (logEntry["type"]? logEntry["type"] : ""),termRegex)}</div>
                                </div>
                            </CardHeader>
                            <CardContent>{highlightValue(logEntry["message"] != undefined ? logEntry["message"].toString() : "", termRegex)}</CardContent>
                        </Card>}
                        <Accordion type="single" className="w-full pt-4" collapsible>
                            <AccordionItem value="details">
                                <AccordionTrigger>
                                    {t("More Details")}
                                    <ChevronDown className="h-5 w-5 shrink-0 transition-transform duration-200 hover:cursor-pointer" />
                                </AccordionTrigger>
                                <AccordionContent>
                                    <div className="mt-2 h-full">
                                        <Tabs defaultValue="entry" className="h-full">
                                            <div className="flex items-center space-x-2 mb-2">
                                                <Input
                                                    type="text"
                                                    placeholder={t("Search elements...")}
                                                    className="flex-grow border-border"
                                                    value={jsonSearch}
                                                    onChange={(e) => setJsonSearch(e.target.value)}
                                                />
                                                <TabsList>
                                                    <TabsTrigger value="entry">{t("Entry")}</TabsTrigger>
                                                    <TabsTrigger value="json">JSON</TabsTrigger>
                                                </TabsList>
                                            </div>
                                            <TabsContent value="entry">
                                                <Table>
                                                    <TableHeader>
                                                        <TableRow>
                                                            <TableHead>{t("Key")}</TableHead>
                                                            <TableHead>{t("Value")}</TableHead>
                                                            <TableHead></TableHead>
                                                            <TableHead></TableHead>
                                                        </TableRow>
                                                    </TableHeader>
                                                    <TableBody>
                                                        {logEntry && filterJsonElements(logEntry, jsonSearch).sort((a,b) => { return a[0].localeCompare(b[0]); }) //sort alphabetically
                                                            .map(([key, value]) => (
                                                            <TableRow key={key} className="group">
                                                                <TableCell className="flex items-center gap-2">
                                                                    <Checkbox checked={(columnVisibility[key]!==undefined? columnVisibility[key] : true)}  
                                                                            onClick={() =>{
                                                                                setVisibility({...columnVisibility, [key]:!(columnVisibility[key]!==undefined? columnVisibility[key] : true)})}} 
                                                                    />
                                                                    {key}
                                                                </TableCell>
                                                                <TableCell>
                                                                    <EllipsisTooltip value={JSON.stringify(value)} />
                                                                </TableCell>
                                                                <TableCell className="py-0 pl-0 pr-1">
                                                                    <FilterToggle filter={{id: `id-filter${Date.now()}-must`, field: key, mandate: "must", query: value as string, active: true}} />      
                                                                </TableCell>
                                                                <TableCell className="py-0 pl-0 pr-4">
                                                                    <FilterToggle filter={{id: `id-filter${Date.now()}-must_not`, field: key, mandate: "must_not", query: value as string, active: true}} />
                                                                </TableCell>
                                                            </TableRow>
                                                        ))}
                                                    </TableBody>
                                                </Table>
                                            </TabsContent>
                                            <TabsContent value="json">
                                                <pre className="whitespace-pre-wrap text-sm text-left pl-4 py-2 max-h-[calc(100vh-200px)] ">
                                                    {logEntry && JSON.stringify(filterJsonElements(logEntry, jsonSearch).reduce((acc, [key, value]) => {
                                                        (acc as Record<string, any>)[key] = value;
                                                        return acc;
                                                    }, {}), null, 2)}
                                                </pre>
                                            </TabsContent>
                                        </Tabs>
                                    </div>
                                </AccordionContent>
                            </AccordionItem>
                        </Accordion>
                    </ScrollArea>
                </div>
            </SheetContent>
        </Sheet>
    )
}