"use client"

import {
    useState,
    useMemo,
    Fragment,
    useEffect,
    DragEvent
} from "react";
import {
    DropdownMenu,
    DropdownMenuCheckboxItem,
    DropdownMenuContent,
    DropdownMenuTrigger,
} from "@/components/shadcn/dropdown-menu";
import {
    Column,
    ColumnDef,
    ColumnFiltersState,
    ColumnSizingState,
    SortingState,
    VisibilityState,
    flexRender,
    getCoreRowModel,
    getFilteredRowModel,
    getPaginationRowModel,
    getSortedRowModel,
    useReactTable
} from "@tanstack/react-table";
import axios from 'axios';
import {
    CardContent,
    CardHeader,
} from "@/components/shadcn/card";
import {
    Table,
    TableBody,
    TableCell,
    TableHead,
    TableHeader,
    TableRow,
} from "@/components/shadcn/table"
import { FormLabel, FormControl, FormField, FormItem } from '@/components/shadcn/form'
import { Button } from "@/components/shadcn/button"
import { Input } from "@/components/shadcn/input"
import { Checkbox } from "@/components/shadcn/checkbox"
import { LogEntrySheet } from "./log-entry-sheet"
import {ColumnEntry, PanelConfigData, QueryPanelDisplayData} from "@/lib/types"
import { useForm, SubmitHandler } from 'react-hook-form';
import { ArrowUpDown, ChevronDown } from "lucide-react"
import { PanelError } from '@/components/charts/components/PanelError';
import { panelError } from '@/components/charts/types';
import { ScrollArea, ScrollBar } from "@/components/shadcn/scroll-area"
import { FormLayout } from '@/components/charts/components/FormLayout';
import { useTranslation } from 'react-i18next';
import lucene from 'lucene';

interface LogEntry {
    [key: string]: any 
}

interface LogTableProps {
    data: LogEntry[],
}

export interface LogTablePanelData extends PanelConfigData {
    pages: number,
    size: number,
    columns: {[key: string]: boolean},
    highlight: boolean
}

export interface LogTablePanelDisplayData extends QueryPanelDisplayData {
    panelConfig: LogTablePanelData,
    setPanelConfig: (panelConfig: LogTablePanelData) => void
}

export interface LogTableWidgetConfigProps {
    config: LogTablePanelData,
    updateWidgetData: (config: PanelConfigData) => void,
}

// Flatten a LogStash Object
const flattenLogStashObject = ((data: LogEntry, base = ""): LogEntry => {
    const retVal: LogEntry = {};

    if (base != "") {
       base = base + ".";
    }

    Object.keys(data).forEach((key) => {
        if (data[key] !== null && data[key] !== undefined && typeof data[key] === 'object') {
            // flatten it and add its entries into retVal
            const child = flattenLogStashObject(data[key], key);
            Object.keys(child).forEach((childKey) => {
                retVal[base + childKey] = child[childKey];
            })
        } else {
            retVal[base + key] = data[key];
        }
    })

    return retVal;
})

const defaultVisibleColumns: {[key: string]: boolean} = {};
const browserTimezone = Intl.DateTimeFormat().resolvedOptions().timeZone;
const timeFormat = new Intl.DateTimeFormat(navigator.languages, { timeZone: browserTimezone, year:'numeric', month:'2-digit', day:'2-digit', hour: '2-digit', minute: '2-digit', second: '2-digit'});
const COL_MIN_SIZE = 75;

export const tableGetLocaleTimestamp = (value?:string) => {
    const date = value? new Date(value) : new Date();
    return timeFormat.format(date);
};

export const highlightValue = (value:string, termRegex?:RegExp) => {
    let highlightPos = 1;
    let vals = [value];
    if (termRegex) {
        vals = value.split(termRegex).filter((v) => {if (v != undefined) { return true; }})
        // to get around having to regex match every term, just assume that they alternate. find if it starts with a match and go from there
        if (vals[0].match(termRegex)) {
            highlightPos = 0;
        }
    }
    return (
        <span>
        {termRegex? vals.map((term, index) => {
            if (index == highlightPos) {
                    highlightPos += 2;
                    return <mark>{term}</mark>
                }
                return term
            }) : vals} 
        </span>
    )
};

// Dynamic column generation
const generateColumnsForLogEntry = (data: LogEntry, stateFunc: (arg0:boolean) => void, sizes:ColumnSizingState, layout:string, termRegex?:RegExp ) => {
    const desiredColumns = [
        '@timestamp',
        'ip',
        'type',
        'message'
    ];
    const retVal = Object.keys(data).map((key) => {
        if (!desiredColumns.includes(key)) {
            defaultVisibleColumns[key] = false;
        }
        return {
            id: key,
            accessorKey: key,
            accessorFn: (d: {[key: string]: string}) => d[key],
            header: ({ column }: { column: Column<LogEntry, unknown>}) => {
                return (
                <div className="flex w-full items-center pr-3">
                <Button
                    variant="ghost"
                    className="p-2 m-2 truncate"
                    onClick={() => column.toggleSorting(column.getIsSorted() === "asc")}
                    style={{justifyContent: 'left' }}
                >
                    {key}
                    <ArrowUpDown className="ml-2 h-4 w-4" />
                </Button>
                <div className="cursor-grab material-symbols-outlined text-lg"
                    onMouseDown={() => stateFunc(true)} 
                    onMouseOut={() => stateFunc(false)} >
                    drag_handle
                </div>
                </div>
                )
            },
            // @ts-ignore 
            cell: ({ row }) => {
                var value = row.getValue(key) as string //className="truncate max-w-[500px]"
                
                if (key == '@timestamp') {
                    const date = new Date(value);
                    value = timeFormat.format(date);
                }
                
                return (
                    <div className={layout=='auto'? "truncate max-w-[500px]" : "truncate"} title={value}>
                        {highlightValue(value, termRegex)}
                    </div>
                )
            },
            size: sizes[key] || document.getElementById(key)?.getBoundingClientRect().width,
        }
    });

    return retVal;
}

const doesColumnArrayContainColumn = (colArray: ColumnEntry[], column: ColumnEntry):boolean => {
    return colArray.some((entry) => {
        return entry.accessorKey == column.accessorKey;
    });
}

const generateColumns = (data: LogEntry[], stateFunc: (arg0:boolean) => void, sizeState:ColumnSizingState, layout:string, termRegex?: RegExp ): ColumnDef<LogEntry>[] => {
    if (data.length === 0) return [];

    const retVal: ColumnEntry[] = [];

    data.forEach((logEntry) => {
        const columns = generateColumnsForLogEntry(logEntry, stateFunc, sizeState, layout, termRegex);
        columns.forEach((column) => {
            if (!doesColumnArrayContainColumn(retVal, column)) {
                retVal.push(column);
            }
        });
    });

    return retVal;
}

export function LogTable (data: LogTablePanelDisplayData) {
    const { t } = useTranslation();
    const [sorting, setSorting] = useState<SortingState>([])
    const [columnFilters, setColumnFilters] = useState<ColumnFiltersState>([])
    const [columnVisibility, setColumnVisibility] = useState<VisibilityState>(data.panelConfig.columns || defaultVisibleColumns)
    const [rowSelection, setRowSelection] = useState({})
    const [columnOrder, setColumnOrder] = useState<string[]>([]);
    const [selectedRow, setSelectedRow] = useState<LogEntry | null>(null)
    const [isSheetOpen, setIsSheetOpen] = useState(false)
    const [logData, setLogData] = useState<LogTableProps>({data: []});
    const [error, setError] = useState<panelError>();
    const [dragActivated, activateDrag] = useState(false);
    const [layout, setLayout] = useState<'auto' | 'fixed'>('auto');
    const [pagination, setPagination] = useState({pageIndex:0, pageSize: data.panelConfig.size || 50})
    const [termRegex, setTermRegex] = useState<RegExp>();

    const selectedQueries = data.panelConfig.subQueryChoice === 'selected'
    ? data.queries.filter(query =>
        data.panelConfig.subQueries.some(subQuery => subQuery === query.id)
        )
    : data.queries;
    
    const count = (data.panelConfig.pages || 5) * (data.panelConfig.size || 50);

    const reqBody = {
        "@timestamp": {
            "from": data.range.startTime.toDate().getTime(),
            "to": data.range.endTime.toDate().getTime(),
        },
        "queries": selectedQueries,
        "filters": data.filters,
        "count": count
    };

    useEffect(() => {
        updateData();
        const queryTerms:string[] = [];
        const unNestTerms = (terms:string[], lucParse:lucene.AST, depth=0) => {
            if (depth > 100) {return terms;} // lets not let this get too deep...
            if ( (lucParse as lucene.BinaryAST).right != undefined &&
                 !((lucParse as lucene.BinaryAST).operator?.includes("NOT"))) { // if its binary AST and no NOT operator
                if (((lucParse as lucene.BinaryAST).right as lucene.NodeTerm).term != undefined) { // and has a term
                    terms.push(((lucParse as lucene.BinaryAST).right as lucene.NodeTerm).term) // add term to list
                }
                else if (((lucParse as lucene.BinaryAST).right as lucene.AST).left != undefined){ // if nested
                    terms.push.apply(unNestTerms(terms, ((lucParse as lucene.BinaryAST).right as lucene.AST), depth+1)) // recurse
                }
            }
            if ( !((lucParse as lucene.LeftOnlyAST).start?.includes("NOT")) ){
                if (lucParse.left.hasOwnProperty('term')) {
                    terms.push((lucParse.left as lucene.NodeTerm).term);
                }
                else if ((lucParse.left as lucene.AST).left != undefined) {
                    terms.push.apply(unNestTerms(terms, (lucParse.left as lucene.AST), depth+1))
                }
            }
            return terms;
        }
        selectedQueries.forEach((q) => {
            if (q.query != '' && q.query != '*') {
                const terms = unNestTerms([], lucene.parse(q.query));
                queryTerms.push.apply(queryTerms, terms);
            }
        });
        if (queryTerms.length > 0) {
            try {
                const regexString = queryTerms.reduce(
                    (exp, term) => exp + "|(" + term + ")",
                    '(' + queryTerms[0] + ")"
                );
                setTermRegex(new RegExp(regexString, "gi"));
            } catch (err) {
                console.warn("Failed to build term regex for highlighting:", err);
                setTermRegex(undefined);
            }
        } else {
            setTermRegex(undefined);
        }
    }, [data]);

    const updateData = async() => {
        axios.post('/nagioslogserver/api/dashboards/table', reqBody)
        .then((logData) => {
            if (logData.data.error) {
                setError(logData.data.error);
                setLogData({data: []});

            } else {
                const newData = logData.data.hits.hits.map((entry: LogEntry) => {
                    const newEntry: LogEntry = {};
                    newEntry._index = entry._index;
                    newEntry._id = entry._id;
                    newEntry._score = entry._score;  // adding score for table sorting
                    Object.assign(newEntry, entry._source);
                    return flattenLogStashObject(newEntry);
                });
                setError(undefined);
                setLogData({data: newData});
                // set initial column order if none is defined, initial column order is alphabetical
                if (columnOrder.length === 0 && newData.length > 0){
                    setColumnOrder(Object.keys(newData[0]).sort((a,b) => { return a.localeCompare(b); }));
                }
            }
        })
        .catch(err => console.error('Error fetching data:', err));
    };
    const [columnSizing, setColSizing] = useState<ColumnSizingState>({});
    const columns = useMemo(() => generateColumns(logData.data, activateDrag, columnSizing, layout, data.panelConfig.highlight? termRegex: undefined), [logData, columnSizing, layout])

    useEffect(() => {
        data.setPanelConfig({...data.panelConfig, columns: columnVisibility});
    }, [columnVisibility]);

    const table = useReactTable({
        data: logData.data,
        columns,
        columnResizeMode: 'onChange',
        onColumnSizingChange: (updater) => {
            // get auto sizes and swap to fixed mode when first resizing ( or after column visibility change )
            if (layout == 'auto') {
                const autoSizes = {...columnSizing};
                table.getVisibleLeafColumns().forEach((col) => {
                    autoSizes[col.id] = Math.round(document.getElementById(col.id)?.getBoundingClientRect().width || 0);
                    col.columnDef.size = autoSizes[col.id];
                    columnSizing[col.id] = autoSizes[col.id];
                });
                setColSizing(autoSizes);
                setLayout('fixed');
                return;
            }
            const newSizes = typeof updater === 'function' ? updater(columnSizing) : updater;
            
            // get delta and change sizes of selected and adjacent column
            const cols = Object.keys(newSizes);
            const updated = {...columnSizing};
            cols.forEach(id => {
                if (newSizes[id] !== columnSizing[id]){
                    const newSize = newSizes[id] || Math.round(document.getElementById(id)?.getBoundingClientRect().width || 0);
                    if (newSize < COL_MIN_SIZE) {
                        return;
                    }
                    const delta = (newSize - (columnSizing[id] || Math.round(document.getElementById(id)?.getBoundingClientRect().width || 0)));
                    const colIndex = table.getVisibleLeafColumns().findIndex(col => col.id == id);
                    const adjCol = table.getVisibleLeafColumns()[colIndex + 1];
                    if (adjCol) {
                        updated[adjCol.id] = Math.max(adjCol.getSize() - delta, COL_MIN_SIZE); 
                    }
                    updated[id] = newSize;
                }
            });
            setColSizing(updated);
            return updated;
        },
        onSortingChange: setSorting,
        onColumnFiltersChange: setColumnFilters,
        getCoreRowModel: getCoreRowModel(),
        getPaginationRowModel: getPaginationRowModel(),
        onPaginationChange:setPagination,
        getSortedRowModel: getSortedRowModel(),
        getFilteredRowModel: getFilteredRowModel(),
        onColumnVisibilityChange: (visiChange) => {
            // set layout to auto to allow for proper size, set new column visibility
            setLayout('auto');
            const newVisibles = typeof visiChange === 'function' ? visiChange(columnVisibility) : visiChange;
            setColumnVisibility(newVisibles);
	    },
        onRowSelectionChange: setRowSelection,
        state: {
            sorting,
            columnFilters,
            columnVisibility,
            columnSizing,
            rowSelection,
            columnOrder,
            pagination
        }
    });

    useEffect(() => {
        table.setPageSize(data.panelConfig.size || 50);
    }, [data.panelConfig.size]);

    const exportAsCSV = () => {
        axios.post('/nagioslogserver/api/dashboards/export_table',
            { ...reqBody, "export_fields": table.getVisibleFlatColumns().map(column => column.id)},
            { responseType: 'blob' }
        )
        .then((exportData) => { 
            const blob = exportData.data;
            const url = URL.createObjectURL(blob);
            const link = document.createElement('a');
            link.href = url;
            link.setAttribute('download', 'export.csv');
            document.body.appendChild(link);
            link.click();
            document.body.removeChild(link);
        })
        .catch(err => console.error('Error downloading CSV data:', err));
    }

    // drag handling functions
    let prevDropTarget:HTMLTableCellElement | null;
    function dropHandler(event:DragEvent<HTMLTableCellElement>, id:string){
        event.preventDefault()
        //formatting and checks
        prevDropTarget = null;
        event.currentTarget.classList.remove("table-drag-over");
        const dragId = event.dataTransfer.getData("text");
        if (id == dragId) {
            return;
        }
        // actually reorder it
        const newOrder = [...columnOrder!];
        const oldIndex = newOrder.indexOf(dragId);
        let newIndex = newOrder.indexOf(id);
        newOrder.splice(newIndex, 0, newOrder.splice(oldIndex, 1)[0]);
        event.currentTarget.style.cursor = "default";
        
        setColumnOrder(newOrder);
    }

    // if the drop is cancelled still remove the select formatting
    function dragEndHandler(){
        if (prevDropTarget != null){
            prevDropTarget.classList.remove("table-drag-over");
        } 
    }

    // swap formatting for selection when hovering over different headers
    function dragoverHandler(event:DragEvent<HTMLTableCellElement>){
        event.preventDefault();
        event.currentTarget.classList.add("table-drag-over");
        if (prevDropTarget != null && prevDropTarget != event.currentTarget) {
            prevDropTarget.classList.remove("table-drag-over");
        }
        prevDropTarget = event.currentTarget;
    }

    // store column name when dragging
    function dragStartHandler(event:DragEvent<HTMLTableCellElement>, id:string){
        event.dataTransfer.setData("text",id);
    }

    return (
        <div className="h-[calc(100%-49px)]">
            {error ? (
                <PanelError error={error} />
            ) : (
                <div className="w-full h-full">
                    <CardHeader className="flex flex-row gap-2 justify-between items-center py-4 space-y-0 px-2 hover:bg-secondary/30">
                        <Input
                            placeholder={t("Filter messages...")}
                            value={(table.getColumn("message")?.getFilterValue() as string) ?? ""}
                            onChange={(event) =>
                                table.getColumn("message")?.setFilterValue(event.target.value)
                            }
                            className="max-w-sm border-border bg-background"
                        />
                        <div className="flex items-center justify-center gap-2">
                            <Button onClick={exportAsCSV}>
                                {t("Export as CSV")}
                            </Button>
                            <DropdownMenu>
                                <DropdownMenuTrigger asChild>
                                    <Button variant="outline" className="ml-auto gap-2 bg-secondary">
                                        {t("Columns")}<ChevronDown className="h-4 w-4"/>
                                    </Button>
                                </DropdownMenuTrigger>
                                <DropdownMenuContent align="end" className="max-h-96 overflow-y-auto">
                                    {table
                                        .getAllColumns()
                                        .filter(
                                            (column) => column.getCanHide()
                                        )
                                        .sort((a,b) => { return a.id.localeCompare(b.id); }) //sort alphabetically
                                        .map((column) => {
                                            return (
                                            <DropdownMenuCheckboxItem
                                                key={column.id}
                                                checked={column.getIsVisible()}
                                                onCheckedChange={(value) =>
                                                    column.toggleVisibility(!!value)
                                                }
                                                onSelect={(event)=>{event.preventDefault();}}
                                            >
                                                {column.id}
                                            </DropdownMenuCheckboxItem>
                                        )
                                    })}
                                </DropdownMenuContent>
                            </DropdownMenu>
                        </div>
                    </CardHeader>
                    <ScrollArea className="h-[calc(100%-132.2px)]">
                        <CardContent className="p-0 border-t border-b border-border bg-background/70">
                            <Table style={{tableLayout: layout }}> 
                                <TableHeader>
                                    {table.getHeaderGroups().map((headerGroup) => (
                                        <TableRow key={headerGroup.id}>
                                            {headerGroup.headers.map((header) => {
                                                return (
                                                    <TableHead key={header.id} id={header.id} className="relative px-0 text-secondary-foreground/70" 
                                                        style={{ width: `${table.getColumn(header.id)?.columnDef.size !== 0 ? `${table.getColumn(header.id)?.columnDef.size}px` : 'auto'}` }}
                                                        onDrop={(event) => {dropHandler(event, header.id)}} draggable={dragActivated}
                                                        onDragOver={(event) => {dragoverHandler(event)}} onDragEnd={dragEndHandler}
                                                        onDragStart={(event) => {dragStartHandler(event, header.id)}}>
                                                        {header.isPlaceholder
                                                            ? null
                                                            : flexRender(
                                                                header.column.columnDef.header,
                                                                header.getContext()
                                                            )}
                                                        <div {...{
                                                            onMouseDown: header.getResizeHandler(),
                                                            className: `absolute top-0 right-0 cursor-col-resize w-2 h-full hover:bg-gray-700 hover:w-2 
                                                                        resizer ${header.column.getIsResizing() ? 'isResizing' : ''}`,
                                                            style: { userSelect: "none", touchAction: "none", },
                                                        }} />
                                                    </TableHead>
                                                )
                                            })}
                                        </TableRow>
                                    ))}
                                </TableHeader>
                                <TableBody>
                                    {table.getRowModel().rows?.length ? (
                                        table.getRowModel().rows.map((row) => (
                                            <Fragment key={row.id}>
                                                <TableRow
                                                    data-state={row.getIsSelected() && "selected"}
                                                    onClick={() => {
                                                        setSelectedRow(row.original)
                                                        setIsSheetOpen(true)
                                                    }}
                                                    className="cursor-pointer text-start"
                                                >
                                                    {row.getVisibleCells().map((cell) => (
                                                        <TableCell key={cell.id}>
                                                            {flexRender(cell.column.columnDef.cell, cell.getContext())}
                                                        </TableCell>
                                                    ))}
                                                </TableRow>
                                            </Fragment>
                                        ))
                                    ) : (
                                        <TableRow>
                                            <TableCell colSpan={columns.length} className="h-24 text-center">
                                                {t("No results.")}
                                            </TableCell>
                                        </TableRow>
                                    )}
                                </TableBody>
                            </Table>
                        </CardContent>
                        <ScrollBar orientation="horizontal" />
                    </ScrollArea>
                    <div className="flex items-center justify-end space-x-2 pb-4 pt-[0.9rem] px-2 border-t border-border hover:bg-secondary/30">
                        <div className="mx-2">
                            <b>{logData.data.length > 0? (pagination.pageIndex * pagination.pageSize) + 1 : 0}</b>
                            {t(" to ")}
                            <b>{Math.min(logData.data.length, ((pagination.pageIndex + 1) * pagination.pageSize))}</b>
                            <small>{t(" of " + logData.data.length)}</small>
                        </div>
                        <Button
                            variant="outline"
                            size="sm"
                            onClick={() => table.previousPage()}
                            disabled={!table.getCanPreviousPage()}
                        >
                            {t("Previous")}
                        </Button>
                        <Button
                            variant="outline"
                            size="sm"
                            onClick={() => table.nextPage()}
                            disabled={!table.getCanNextPage()}
                        >
                            {t("Next")}
                        </Button>
                    </div>
                    <LogEntrySheet
                        isOpen={isSheetOpen}
                        onOpenChange={setIsSheetOpen}
                        logEntry={selectedRow}
                        columnVisibility={columnVisibility}
                        setVisibility={setColumnVisibility}
                        termRegex={data.panelConfig.highlight ? termRegex : undefined}
                    />
                </div>
            )}
        </div>
    )
}

export const LogTableConfig = ({config, updateWidgetData}: LogTableWidgetConfigProps) => {
  
    const { t } = useTranslation();

    const form = useForm<LogTablePanelData>({
      defaultValues: {
          title: config.title,
          subQueryChoice: config.subQueryChoice || 'all',
          subQueries: config.subQueries || [],
          pages: config.pages || 5,
          size: config.size || 50,
          highlight: config.highlight || false,
      },
    });
  
    const onSubmit: SubmitHandler<LogTablePanelData> = (data) => {
        updateWidgetData(data);
    };
  
    return (
      <FormLayout form={form} onSubmit={onSubmit} type='dynamic'>
        <FormField
            control={form.control}
            name="pages"
            render={({ field }) => (
            <FormItem className="grid grid-cols-3 items-center gap-4 space-y-0">
                <FormLabel className="text-right">{t('Pages')}</FormLabel>
                <FormControl>
                    <Input {...field} className="col-span-2 border-border" />
                </FormControl>
            </FormItem>
            )}
        />
        <FormField
            control={form.control}
            name="size"
            render={({ field }) => (
            <FormItem className="grid grid-cols-3 items-center gap-4 space-y-0">
                <FormLabel className="text-right">{t('Rows Per Page')}</FormLabel>
                <FormControl>
                    <Input {...field} className="col-span-2 border-border" />
                </FormControl>
            </FormItem>
            )}
        />
        <FormField
          control={form.control}
          name="highlight"
          render={({ field }) => (
            <FormItem className="grid grid-cols-3 items-center gap-4 space-y-0">
              <FormLabel className="text-right">{t('Highlight Queries')}</FormLabel>
              <FormControl>
                <Checkbox 
                    checked={field.value}
                    onCheckedChange={field.onChange}
                />
              </FormControl>
            </FormItem>
          )}
        />
      </FormLayout>
    )
  }
