import axios, { AxiosError } from 'axios';
import { ReactNode, useEffect } from 'react';
import '@/i18n';
import { I18nextProvider, useTranslation } from 'react-i18next';
import { useQuery } from '@tanstack/react-query';

interface LanguageResponse {
    language: string;
}

export const LanguageProvider = ({ children }: { children: ReactNode }) => {
    const { i18n } = useTranslation();

    const languageQuery = useQuery<LanguageResponse, AxiosError>({
        queryKey: ['language'],
        queryFn: async () => {
            const { data } = await axios.get<LanguageResponse>('/nagioslogserver/api/user/get_language');
            return data;
        },
        retry: false,
    });

    const languageResourceQuery = useQuery({
        enabled: languageQuery.isSuccess,
        queryKey: ['languageResource', languageQuery.data?.language],
        queryFn: async () => {
            const { data } = await axios.get(`/nagioslogserver/media/language/json/${languageQuery.data?.language}.json`);
            return data;
        },
    });

    // Have to tie these queries together because addResource doesn't cause rerender.
    // Have to use a useQuery instead of i18next backend to prevent multiple loads of different file
    //  (gets en_US.json because it's the default, then gets es_ES when you update language, this method only gets it once)
    useEffect(() => {
        if (languageResourceQuery.isSuccess && languageQuery.data?.language) {
            i18n.addResourceBundle(languageQuery.data.language, 'translations', languageResourceQuery.data);
            i18n.changeLanguage(languageQuery.data.language);
        }
    }, [i18n, languageResourceQuery.isSuccess, languageResourceQuery.status, languageQuery.data?.language]);

    // useQuery errors are thrown (specified in queryClient) and handled by Error Boundary

    // If someone has an old value in their database, the language can be en instead of en_US, don't stop the application from loading
    if (languageQuery.isLoading || (languageResourceQuery.isLoading && languageResourceQuery.failureCount == 0)) {
        return;
    }

    return (
        <I18nextProvider i18n={i18n} defaultNS={'translations'}>
            {children}
        </I18nextProvider>
    );
};
