import { useQuery } from '@tanstack/react-query';
import axios from 'axios';
import { createContext, useContext, useEffect } from 'react';

type Theme = 'default' | 'dark' | 'light';

type ThemeProviderProps = {
    children: React.ReactNode;
    defaultTheme?: Theme;
    storageKey?: string;
};

const initialState: Theme = localStorage.getItem('theme') as Theme || 'light';

const ThemeProviderContext = createContext<Theme>(initialState);

export function ThemeProvider({ children, defaultTheme = 'dark', storageKey = 'vite-ui-theme', ...props }: ThemeProviderProps) {
    const themeQuery = useQuery({
        queryKey: ['theme'],
        queryFn: async () => {
            const response = await axios.get('/nagioslogserver/api/user/get_theme');
            if (response.data.theme == 'light') {
                localStorage.setItem('theme', 'light');
                return 'light';
            } else if (response.data.theme == 'dark') {
                localStorage.setItem('theme', 'dark');
                return 'dark';
            } else {
                throw new Error('not a valid theme.');
            }
        },
        initialData: initialState,
    });

    useEffect(() => {
        const root = window.document.documentElement;

        root.classList.remove('light', 'dark');

        root.classList.add(themeQuery.data);
    }, [themeQuery.data]);

    // keepalive will handle 401 error
    // if (themeQuery.isError && themeQuery.error && themeQuery.error instanceof AxiosError && themeQuery.error.response?.status != 401) {
    //     return <ShowNetworkError />;
    // }

    return (
        <ThemeProviderContext.Provider {...props} value={themeQuery.data}>
            {children}
        </ThemeProviderContext.Provider>
    );
}

export const useTheme = () => {
    const context = useContext(ThemeProviderContext);

    if (context === undefined) throw new Error('useTheme must be used within a ThemeProvider');

    return context;
};
