import { DialogContent, DialogFooter, DialogHeader, DialogTitle } from '@/components/shadcn/dialog';
import { Input } from '@/components/shadcn/input';
import { Button } from '@/components/shadcn/button';
import { useForm, SubmitHandler } from 'react-hook-form';
import { Form, FormLabel, FormControl, FormField, FormItem } from '@/components/shadcn/form';
import { useTranslation } from 'react-i18next';
import { toast } from '@/components/shadcn/use-toast';
import { AxiosError } from 'axios';
import { useMutation, useQueryClient } from '@tanstack/react-query';
import { addDashboard } from '@/dashboard/queries';
import { Checkbox } from "@/components/shadcn/checkbox";
import { useNavigate } from "react-router-dom";
import { useDashboardContext } from '@/dashboard/contexts/DashboardContext';

/**
 * The add dashboard dialog.
 */
export const AddDashboardDialogContent = () => {
    const { t } = useTranslation();
    const queryClient = useQueryClient();
    const navigate = useNavigate();
    const { range } = useDashboardContext().dashboardInfo;
    
    const addDashboardMutation = useMutation({
        mutationFn: addDashboard,
        onSuccess: (response) => {
            queryClient.invalidateQueries({ queryKey: ['dashboards'] });
            navigate(`?dashboardId=${response.data.message.id}`);
        },
        onError: (error: AxiosError<{ message: string }>) => {
            toast({
                title: t('Failed to Add Dashboard.'),
                variant: 'error',
                description: error.response?.data.message || t('An unknown error occured.'),
            });
        },
    });

    const form = useForm<{name: string, global: boolean}>({
        defaultValues: {
            name: t("My New Dashboard")
        },
    });

    const onSubmit: SubmitHandler<{name: string, global: boolean}> = (data) => {
        addDashboardMutation.mutate({title: data.name, data: JSON.stringify({title: data.name, rows: [], subQueries: [], filters: [], range: range}), global: data.global});
    };

    return (
        
        <DialogContent className="sm:max-w-[425px]">
            <DialogHeader>
                <DialogTitle>{t('Create a New Dashboard')}</DialogTitle>
            </DialogHeader>
            <Form {...form}>
                <form onSubmit={form.handleSubmit(onSubmit)} className="grid gap-4 py-2">
                    <FormField
                        control={form.control}
                        name="name"
                        render={({ field }) => (
                            <FormItem className="grid grid-cols-4 items-center gap-4 space-y-0">
                                <FormLabel className="text-right">{t('Name')}</FormLabel>
                                <FormControl>
                                    <Input {...field} className="col-span-3 border-border" />
                                </FormControl>
                            </FormItem>
                        )}
                    />
                    <FormField
                        control={form.control}
                        name="global"
                        render={({ field }) => (
                            <FormItem className="grid grid-cols-4 items-center gap-4 space-y-0">
                                <FormLabel className="text-right">{t('Global')}</FormLabel>
                                <FormControl>
                                    <Checkbox id="global" 
                                        checked={field.value}
                                        onCheckedChange={field.onChange}
                                        defaultChecked={false}
                                    />
                                </FormControl>
                            </FormItem>
                        )}
                    />
                    <DialogFooter className="pt-2">
                        <Button type="submit" className="justify-end">
                            {t('Create Dashboard')}
                        </Button>
                    </DialogFooter>
                </form>
            </Form>
        </DialogContent>
    );
};
