import { useEffect, useState, useMemo } from 'react';
import { CircleChevronDown } from "lucide-react";
import { ChevronDown } from "lucide-react";
import { Button } from "@/components/shadcn/button";
import DragIndicatorIcon from '@mui/icons-material/DragIndicator';
import {
  Accordion,
  AccordionContent,
  AccordionItem,
  AccordionTrigger,
} from "@/components/shadcn/accordion";
import { Label } from '@/components/shadcn/label';
import { SidebarProvider } from "@/components/shadcn/sidebar";
import { DragDropContext, Droppable, Draggable, DropResult } from '@hello-pangea/dnd';
import moment from 'moment';
import { create } from 'zustand';
import { AxiosError } from 'axios';
import { row, DashboardDefinition, DashboardState} from '@/lib/types';
import { DashboardToolbar } from '@/dashboard/toolbar/DashboardToolbar';
import { QueryToolbar } from '@/dashboard/queries/QueryToolbar';
import { NLQToolbar } from '@/dashboard/queries/NLQToolbar';
import { useDashboardContext } from '@/dashboard/contexts/DashboardContext';
import { useQuery } from '@tanstack/react-query';
import { getDashboardById } from '@/dashboard/queries';
import { getReportById } from '@/dashboard/reports/queries';
import { ReportToolbar } from '@/dashboard/reports/ReportToolbar';
import DeleteOutlineOutlinedIcon from '@mui/icons-material/DeleteOutlineOutlined';
import AddIcon from '@mui/icons-material/Add';
import { useTranslation } from 'react-i18next';
import { subQuery } from '@/dashboard/queries/types';
import { AddPanel } from '@/dashboard/panels/AddEditPanel';
import { DashboardSidebar } from "@/dashboard/dashboard-sidebar";
import { getNaturalLanguageQueryStatus } from '@/dashboard/queries/queries';
import { EditableLabel } from '@/components/EditableLabel';
import { cn } from '@/lib/utils';
import { DashboardRow } from '@/dashboard/DashboardRow';

export const dashboardStore = create<DashboardState>((set) => ({
  definition: {name: '', user: '', rows: []},
  setDefinition: (newRows: DashboardDefinition) => set(() => ({definition: newRows})),
}));

const Dashboard = ({ dashboardId, report, exporting, searchQuery, simpleTokens }: { dashboardId: string, report: boolean, exporting: boolean, searchQuery: subQuery | null, simpleTokens: string | null }) => {
  const { definition, setDefinition } = dashboardStore();
  const { dashboardInfo, dashboardMetaInfo } = useDashboardContext();
  const { setFilters, setSubQueries, range, setRange, refreshRate, setRefreshRate } = dashboardInfo;
  const { setIsReport, setIsExporting } = dashboardMetaInfo;
  useEffect(() => {
    setIsReport(report);
    setIsExporting(exporting);
  }, []);
  const { t } = useTranslation();
  const [accordionValues, setAccordionValues] = useState<string[]>([]);
  const [originalAccordionValues, setOriginalAccordionValues] = useState<string[]>([]);
  const [isDragging, setIsDragging] = useState(false);
  const [showAdvanced, setShowAdvanced] = useState(false);
  const [prevRowCount, setPrevRowCount] = useState(definition.rows.length);

  const intervalOptions: {[key: string]: { label: string; offset: number }} = {
    '5m': { label: t('Last 5m'), offset: 5 * 60 },
    '15m': { label: t('Last 15m'), offset: 15 * 60 },
    '1h': { label: t('Last 1h'), offset: 60 * 60 },
    '2h': { label: t('Last 2h'), offset: 2 * 60 * 60 },
    '6h': { label: t('Last 6h'), offset: 6 * 60 * 60 },
    '12h': { label: t('Last 12h'), offset: 12 * 60 * 60 },
    '24h': { label: t('Last 24h'), offset: 24 * 60 * 60 },
    '2d': { label: t('Last 2d'), offset: 2 * 24 * 60 * 60 },
    '7d': { label: t('Last 7d'), offset: 7 * 24 * 60 * 60 },
    '30d': { label: t('Last 30d'), offset: 30 * 24 * 60 * 60 }
  };

  const dashboardSummaryQuery = useQuery<DashboardDefinition, AxiosError>({
    queryKey: [dashboardId],
    queryFn: async () => {
        const response = report ? await getReportById(dashboardId) : await getDashboardById(dashboardId);
        const data = {
          name: response.data.title,
          rows: response.data.rows,
          subQueries: response.data.subQueries,
          filters: response.data.filters,
//          uneditable: response.data.uneditable || false,
//          uneditable: false,
          uneditable: (response.data.id === "home" || response.data.id === "simple"),
          simple: (response.data.id === "simple"? true : false),
          range: response.data.range ? 
            response.data.range.value === 'custom' ?
              {
                startTime: moment(response.data.range.startTime),
                endTime: moment(response.data.range.endTime),
                value: "custom",
                label: response.data.range.label
              } :
              {
                startTime: response.data.range.value ? moment().subtract(intervalOptions[response.data.range.value].offset, 'seconds') : moment().subtract(1, 'day'),
                endTime: moment(),
                value: response.data.range.value || '24h',
                label: response.data.range.label || t('Last 24h')
              } :
            {
              startTime: moment().subtract(1, 'day'),
              endTime:  moment(),
              value: '24h',
              label: t('Last 24h')
            },
            user: response.data.user,
            refreshRate: response.data.refreshRate
        };

        if (dashboardId && !report && !exporting && (response.data.savetolocalstorage !== false && response.data.id !== "simple")) {
          localStorage.setItem('dashboardId', dashboardId);
        }

        searchQuery ? setSubQueries([searchQuery]) : setSubQueries(data.subQueries);
        data.refreshRate && setRefreshRate(data.refreshRate);
        setFilters(data.filters);
        setRange(data.range);
        setDefinition(data);
        setAccordionValues(data.rows.map((row: row) => {return row.id;}));
        return data;
    },
    retry: false,
    refetchOnWindowFocus: false
  });

  const naturalLanguageQueryStatusSummary = useQuery<number, AxiosError>({
    queryKey: ['nlq_status'],
    queryFn: async () => {
      const response = await getNaturalLanguageQueryStatus();
      return response.data.value;
    }
  });

  useEffect(() => {
    if (range.value !== "custom" && refreshRate !== 0) {
      const interval = setInterval(() => {
        const end = moment();
        const start = moment(end).subtract(intervalOptions[range.value].offset, 'seconds');
        setRange(
          {
            ...range,
            startTime: start,
            endTime: end,
            reset: false
          }
        );
      }, refreshRate);
      return () => clearInterval(interval);
    }
  }, [range, refreshRate]);

  useEffect(() => {
   if (definition.rows.length > prevRowCount) {
     // If a new row was added, add the id for the new row to accordianValues. This opens the row automatically at row creation time. 
     const newRow = definition.rows[definition.rows.length - 1];
     setAccordionValues((prev) => [...prev, newRow.id]);
   }
   setPrevRowCount(definition.rows.length);
   }, [definition.rows.length]);

  const onBeforeCapture = () => {
    setOriginalAccordionValues(accordionValues);
    setAccordionValues([]);
    setIsDragging(true);
  }

  const onDragEnd = (result: DropResult) => {
    if (!result.destination) {
      return;
    }
    const newRows = Array.from(definition.rows);
    const [reorderedItem] = newRows.splice(result.source.index, 1);
    newRows.splice(result.destination.index, 0, reorderedItem);
    setAccordionValues(originalAccordionValues);
    setIsDragging(false);
    setDefinition({...definition, rows: newRows});
  };

  const getRowById = (rowId: string): number => {
    for(var i = 0; i < definition.rows.length; i++) {
      if (definition.rows[i].id == rowId) {
        return i;
      }
    }

    return -1;
  }

  const updateRowDefinition = (newRow: row) => {
    const newRows = definition.rows.map(row => {
      if (row.id === newRow.id) {
        return newRow;
      }
      return row;
    });
    setDefinition({...definition, rows: newRows});
  }

  const MemoizedRows = useMemo(() => definition.rows.map((row, index) => {
    if (row.title === "Nagios Logserver Simple Search") {
      row.uneditable = true;
    }
    const onClickDeleteRow = () => {
      const rowIndex = getRowById(row.id);

      if (rowIndex == -1) {
        return;
      }

      const newRows = definition.rows.filter(
        (testRow) => row.id != testRow.id
      )

      setDefinition({...definition, rows: newRows});
    }

    const onRowNameChange = (value: string) => {
      row.title = value;
      updateRowDefinition(row);
    }

    const toggleAccordion = (rowId: string) => {
      setAccordionValues((prevValues) => {
        if (prevValues.includes(rowId)) {
          return prevValues.filter((value) => value !== rowId);
        } else {
          return [...prevValues, rowId];
        }
      })
    }

    return (
      <div className="page-section">
        <Draggable key={row.id} draggableId={row.id} index={index}>
        {(provided) => (
          <AccordionItem
            value={row.id}
            ref={provided.innerRef}
            {...provided.draggableProps}
          >
            {!row.uneditable &&
            <AccordionTrigger className="p-0 hover:no-underline hover:cursor-default mb-2">
              <CircleChevronDown
                className="h-5 w-5 shrink-0 transition-transform duration-200 hover:cursor-pointer mr-2 ml-[0.1rem]"
                onClick={() => toggleAccordion(row.id)}
              />
              <div className="flex items-center w-full">
                <div className="flex items-center justify-between w-full py-1">
                  <div className="flex gap-2 w-1/2 items-center pl-1">
                    {!report &&
                      <EditableLabel
                        value={row.title}
                        placeholder={row.title}
                        onSave={onRowNameChange}
                        className='flex items-center gap-2'
                        labelClassName='flex py-2 px-2 text-lg'
                        inputClassName='flex text-lg my-1 px-2 h-9'
                      />
                    }
                    {report &&
                      <Label className="flex py-2 text-lg">{row.title}</Label>
                    }
                  </div>
                  {!report &&
                    <div className="flex items-center pr-1">
                      <AddPanel rowId={row.id} >
                        <Button
                          size="icon"
                          className="min-w-9"
                          variant="ghost"
                        >
                          <AddIcon fontSize="small" />
                        </Button>
                      </AddPanel>
                      <Button
                        size="icon"
                        className="min-w-9"
                        variant="ghost"
                        onClick={onClickDeleteRow}
                      >
                        <DeleteOutlineOutlinedIcon fontSize="small" />
                      </Button>
                      <div {...provided.dragHandleProps} className="flex items-center">
                        <Button
                          size="icon"
                          className="min-w-9 hover:cursor-grab"
                          variant="ghost"
                        >
                          <DragIndicatorIcon fontSize="small" />
                        </Button>
                      </div>
                    </div>
                  }
                </div>
              </div>
            </AccordionTrigger> }
            <AccordionContent className={(isDragging && !accordionValues.includes(row.id)) ? "max-h-0" : "max-h-full"}>
              <DashboardRow
                rowId={row.id}
                panels={row.panels}
                uneditable={row.uneditable}
              />
            </AccordionContent>
          </AccordionItem>
        )}
      </Draggable>
    </div>
  )}), [definition]);

  if (!dashboardSummaryQuery.data) {
    return;
  }
  return (
    <SidebarProvider>
      {!report && !definition.uneditable && <DashboardSidebar dashboardId={dashboardId} dashboardName={dashboardSummaryQuery.data.name} user={dashboardSummaryQuery.data.user}/> }
      <div className="h-full flex flex-col relative w-full">
        {!report && !definition.uneditable &&
          <div>
            <DashboardToolbar dashboardId={dashboardId} dashboardName={dashboardSummaryQuery.data.name} />
            <div className={cn("overflow-hidden h-auto transition-[max-height_0.15s_ease-out]", naturalLanguageQueryStatusSummary.data != 1 || showAdvanced ? "max-h-96" : "max-h-14")}>
              {naturalLanguageQueryStatusSummary.data == 1 &&
                <div className="flex items-center w-full mt-3 relative">
                  <NLQToolbar />
                  <button
                    className="absolute w-[9.5rem] right-7 flex items-center gap-2 p-1"
                    onClick={() => setShowAdvanced(!showAdvanced)}
                  >
                    <ChevronDown className={cn("h-4 w-4 shrink-0 transition-transform duration-200", showAdvanced && "rotate-180")} />
                    <span className="flex items-center text-sm whitespace-nowrap truncate">Advanced Search</span>
                  </button>
                </div>
              }
              <QueryToolbar/>
            </div>
          </div>
        }
        {!report && definition.simple &&
          <QueryToolbar isSimple={true} simpleTokens={simpleTokens} />
        } 
        {report &&
          <ReportToolbar dashboardId={dashboardId} dashboardName={dashboardSummaryQuery.data.name} />
        }
        <div className="flex-grow px-4">
          <DragDropContext
            onDragEnd={onDragEnd}
            onBeforeCapture={onBeforeCapture}
          >
            <Droppable droppableId="rows">
              {(provided) => (
                <Accordion
                  type="multiple"
                  className="w-full pb-12"
                  ref={provided.innerRef}
                  {...provided.droppableProps}
                  value={accordionValues}
                  defaultValue={definition.rows.map((row) => {return row.id;})}
                >
                  {MemoizedRows}
                  {provided.placeholder}
                </Accordion>
              )}
            </Droppable>
          </DragDropContext>
        </div>
      </div>
    </SidebarProvider>
  );
};

export default Dashboard;
