import { deleteDashboard } from '@/dashboard/queries';
import {
    AlertDialog,
    AlertDialogAction,
    AlertDialogCancel,
    AlertDialogContent,
    AlertDialogDescription,
    AlertDialogFooter,
    AlertDialogHeader,
    AlertDialogTitle,
    AlertDialogTrigger,
} from '@/components/shadcn/alert-dialog';
import { buttonVariants } from '@/components/shadcn/button';
import { ReactNode } from 'react';
import { useTranslation } from 'react-i18next';
import { useMutation, useQueryClient } from '@tanstack/react-query';
import { toast } from '@/components/shadcn/use-toast';
import { AxiosError } from 'axios';
import { useNavigate } from 'react-router-dom';
import { useDashboardContext } from './contexts/DashboardContext';

export const DeleteDashboardDialog = ({ dashboardId, children }: { dashboardId: string; children: ReactNode }) => {
    const navigate = useNavigate();
    const queryClient = useQueryClient();
    const { defaultDashboardId } = useDashboardContext().dashboardMetaInfo;
    
    const deleteDashboardMutation = useMutation({
        mutationFn: deleteDashboard,
        onSuccess: (response) => {
            queryClient.invalidateQueries({ queryKey: ['dashboards'] });
            if (response.data['_id'] === dashboardId) {
              navigate(`?dashboardId=${defaultDashboardId}`);
            }
            toast({
                title: t('Dashboard deleted.'),
                variant: 'success',
            });
        },
        onError: (error: AxiosError<{ message: string }>) => {
            toast({
                title: t('Failed to delete dashboard.'),
                variant: 'error',
                description: error.response?.data.message,
            });
        },
      });
    
    const { t } = useTranslation();

    return (
        <AlertDialog>
            <AlertDialogTrigger>{children}</AlertDialogTrigger>
            <AlertDialogContent>
                <AlertDialogHeader>
                    <AlertDialogTitle>{t('Confirm Dashboard Deletion')}</AlertDialogTitle>
                    <AlertDialogDescription>{t('Are you sure you want to delete this dashboard?')}</AlertDialogDescription>
                </AlertDialogHeader>
                <AlertDialogFooter>
                    <AlertDialogCancel>{t('Cancel')}</AlertDialogCancel>
                    <AlertDialogAction onClick={() => deleteDashboardMutation.mutate(dashboardId)} className={buttonVariants({ variant: 'destructive' })}>
                        {t('Delete')}
                    </AlertDialogAction>
                </AlertDialogFooter>
            </AlertDialogContent>
        </AlertDialog>
    );
};
