import { Button } from '@/components/shadcn/button';
import { Dialog, DialogContent, DialogDescription, DialogFooter, DialogHeader, DialogTitle, DialogTrigger } from '@/components/shadcn/dialog';
import { Input } from '@/components/shadcn/input';
import { toast } from '@/components/shadcn/use-toast';
import { AxiosError, AxiosResponse } from 'axios';
import { ReactNode, useState } from 'react';
import { useTranslation } from 'react-i18next';
import { useMutation } from '@tanstack/react-query';
import { DashboardDefinition } from '@/lib/types';
import { getDashboardById } from './queries';

export const ExportDashboardDialog = ({ id, dashboard, children }: { id: string, dashboard: DashboardDefinition; children: ReactNode }) => {
    const { t } = useTranslation();
    const [exportTitle, setExportTitle] = useState(dashboard.name);
    const [open, setOpen] = useState(false);

    const exportDashboardMutation = useMutation({
        mutationFn: getDashboardById,
        onSuccess: (response: AxiosResponse<{ dashboard: string }, AxiosError>) => {
            const msg = JSON.stringify(response.data, null, 4);
            const link = document.createElement('a');
            const file = new Blob([msg], { type: 'application/json' });
            link.href = URL.createObjectURL(file);
            link.download = exportTitle + '.json';
            link.click();
            URL.revokeObjectURL(link.href);
            setOpen(false);
            toast({
                title: t('Dashboard exported.'),
                variant: 'success',
                description: t('Dashboard successfully exported.'),
            });
        },
        onError: () => {
            toast({
                title: t('Failed to export dashboard.'),
                variant: 'error',
                description: t('Please try again.'),
            });
        },
    });

    const submitButtonDisabled = () => {
        return (exportTitle.trim().length == 0);
    }

    return (
        <Dialog open={open} onOpenChange={setOpen}>
            <DialogTrigger>{children}</DialogTrigger>
            <DialogContent>
                <DialogHeader>
                    <DialogTitle>{t('Export Dashboard')}</DialogTitle>
                    <DialogDescription>{t('Use this to export an exact clone of the current dashboard.')}</DialogDescription>
                </DialogHeader>
                <Input value={exportTitle} onChange={(value) => setExportTitle(value.target.value)} />
                <DialogFooter>
                    <Button type="submit" onClick={() => exportDashboardMutation.mutate(id)} disabled={submitButtonDisabled()} >
                        {t('Export')}
                    </Button>
                </DialogFooter>
            </DialogContent>
        </Dialog>
    );
};
