import { Dialog, DialogContent, DialogFooter, DialogHeader, DialogTrigger } from '@/components/shadcn/dialog';
import { ReactNode, useState } from 'react';
import { useForm } from 'react-hook-form';
import { useTranslation } from 'react-i18next';
import { z } from 'zod';
import { zodResolver } from '@hookform/resolvers/zod';
import { addDashboard } from '@/dashboard/queries';
import { Form, FormControl, FormField, FormItem, FormLabel, FormMessage } from '@/components/shadcn/form';
import { Input } from '@/components/shadcn/input';
import { Button } from '@/components/shadcn/button';
import { useMutation, useQueryClient } from '@tanstack/react-query';
import { toast } from '@/components/shadcn/use-toast';
import { AxiosError } from 'axios';

export const ImportDashboardDialog = ({ children }: { children: ReactNode }) => {
    const { t } = useTranslation();
    const queryClient = useQueryClient();
    const [open, setOpen] = useState(false);
    const addDashboardMutation = useMutation({
        mutationFn: addDashboard,
        onSuccess: () => {
            queryClient.invalidateQueries({ queryKey: ['dashboards'] });
            setOpen(false);
            toast({
                title: t('Saved new Dashboard.'),
                variant: 'success',
                description: t('Select your new dashboard in the dashboard dropdown menu.'),
            });
        },
        onError: (error: AxiosError<{ message: string }>) => {
            toast({
                title: t('Failed to Add Dashboard.'),
                variant: 'error',
                description: error.response?.data.message || t('An unknown error occured.'),
            });
        },
    });
    const importFormSchema = z.object({
        import: z.instanceof(File).refine((file) => {
            return file.type == 'application/json';
        }, t('File Is Not a Valid Dashboard File!')),
    });

    const importForm = useForm<z.infer<typeof importFormSchema>>({
        resolver: zodResolver(importFormSchema),
        mode: 'onChange',
        defaultValues: {
            import: undefined,
        },
    });

    const onImportSubmit = async (data: z.infer<typeof importFormSchema>) => {
        const dashboard = JSON.parse(await data.import.text());
        delete dashboard.id;
        addDashboardMutation.mutate({
            title: dashboard.title,
            data: JSON.stringify(dashboard),
            global: false
        });
    };

    return (
        <Dialog open={open} onOpenChange={setOpen}>
            <DialogTrigger>{children}</DialogTrigger>
            <DialogContent>
                <DialogHeader>{t('Import a Dashboard')}</DialogHeader>
                <Form {...importForm}>
                    <form className="flex flex-col gap-2" onSubmit={importForm.handleSubmit(onImportSubmit)}>
                        <FormField
                            control={importForm.control}
                            name="import"
                            render={({ field: { value, onChange, ...fieldProps } }) => (
                                <FormItem>
                                    <FormLabel>{t('Dashboard File')}</FormLabel>
                                    <FormControl>
                                        {/* Why this looks strange https://github.com/shadcn-ui/ui/discussions/2137 */}
                                        <Input {...fieldProps} type="file" onChange={(event) => onChange(event.target.files && event.target.files[0])} />
                                    </FormControl>
                                    <FormMessage />
                                </FormItem>
                            )}
                        />
                        <DialogFooter>
                            <Button type="submit">{t('Import')}</Button>
                        </DialogFooter>
                    </form>
                </Form>
            </DialogContent>
        </Dialog>
    );
};
