import { Dialog, DialogContent, DialogFooter, DialogHeader, DialogTitle, DialogDescription, DialogTrigger } from '@/components/shadcn/dialog';
import { Input } from '@/components/shadcn/input';
import { Button } from '@/components/shadcn/button';
import {
    Select,
    SelectContent,
    SelectItem,
    SelectTrigger,
    SelectValue,
} from "@/components/shadcn/select";
import { Checkbox } from "@/components/shadcn/checkbox";
import { ReactNode, useState } from 'react';
import { useForm, SubmitHandler } from 'react-hook-form';
import { Form, FormLabel, FormControl, FormField, FormItem } from '@/components/shadcn/form';
import { useTranslation } from 'react-i18next';
import { toast } from '@/components/shadcn/use-toast';
import { alert } from '@/dashboard/alerts/types';
import { AxiosError } from 'axios';
import { useMutation, useQuery, useQueryClient } from '@tanstack/react-query';
import { addAlert, getNrdp, getSnmp, getUsers, getTemplates } from '@/dashboard/alerts/queries';
import { MultiSelect } from '@/components/shadcn/multi-select';
import { useDashboardQueryContext } from '@/dashboard/contexts/DashboardQueryContext';

/**
 * The alert dialog.
 */
export const AddAlertDialog = ({ dashboardName, dashboardId, children }: { dashboardName: string | undefined, dashboardId: string, children: ReactNode }) => {
    const [open, setOpen] = useState(false);
    const { t } = useTranslation();
    const queryClient = useQueryClient();
    const dashboardQuery = useDashboardQueryContext();

    const addAlertMutation = useMutation({
        mutationFn: addAlert,
        onSuccess: () => {
            queryClient.invalidateQueries({ queryKey: ['alerts'] });
            setOpen(false);
            toast({
                title: t('Alert added.'),
                variant: 'success',
                description: t('Access your new alert in the Alerting section.'),
            });
        },
        onError: (error: AxiosError<{ message: string }>) => {
            toast({
                title: t('Failed to add Alert.'),
                variant: 'error',
                description: error.response?.data.message || t('An unknown error occured.'),
            });
        },
    });

    const nrdpSummaryQuery = useQuery<[{id: string, name: string}], AxiosError>({
        queryKey: ['nrdp'],
        queryFn: async () => {
            const response = await getNrdp();
            return response.data;
        },
    });

    const snmpSummaryQuery = useQuery<[{id: string, name: string}], AxiosError>({
        queryKey: ['snmp'],
        queryFn: async () => {
            const response = await getSnmp();
            return response.data;
        },
    });

    const usersSummaryQuery = useQuery<[{value: string, label: string}], AxiosError>({
        queryKey: ['users'],
        queryFn: async () => {
            const response = await getUsers();
            return response.data.map((user: {id: string, name: string, username: string}) => {
                return {
                    value: user.id, label: user.name + ' (' + user.username + ')'
                };
            });
        },
    });

    const templatesSummaryQuery = useQuery<[{id: string, name: string}], AxiosError>({
        queryKey: ['templates'],
        queryFn: async () => {
            const response = await getTemplates();
            return response.data;
        },
    });

    const form = useForm<alert>({
        defaultValues: {
            name: dashboardName + t(' Alert'),
            check_interval: '',
            lookback_period: '',
            dash_query: JSON.stringify(dashboardQuery),
            query: JSON.stringify(dashboardQuery),
            dash_id: dashboardId
        },
    });

    const onSubmit: SubmitHandler<alert> = (data) => {
        data.dash_query = JSON.stringify(dashboardQuery);
        data.query = JSON.stringify(dashboardQuery);
        addAlertMutation.mutate({ data });
    };

    if (!nrdpSummaryQuery.isSuccess) {
        return;
    }

    if (!snmpSummaryQuery.isSuccess) {
        return;
    }

    if (!usersSummaryQuery.isSuccess) {
        return;
    }

    if (!templatesSummaryQuery.isSuccess) {
        return;
    }

    return (
        <Dialog open={open} onOpenChange={setOpen}>
            <DialogTrigger asChild>
                {children}
            </DialogTrigger>
            <DialogContent className="sm:max-w-[425px]">
                <DialogHeader>
                    <DialogTitle>{t('Create an Alert')}</DialogTitle>
                    <DialogDescription>{t("Create an alert based on the current dashboard's queries and filters.")}</DialogDescription>
                </DialogHeader>
                <Form {...form}>
                    <form onSubmit={form.handleSubmit(onSubmit)} className="flex flex-col gap-2">
                        <FormField
                            control={form.control}
                            name="name"
                            render={({ field }) => (
                                <FormItem>
                                    <FormLabel className="">{t('Alert Name')}</FormLabel>
                                    <FormControl>
                                        <Input {...field} className="border-border" />
                                    </FormControl>
                                </FormItem>
                            )}
                        />
                        <div className="flex gap-2">
                            <FormField
                                control={form.control}
                                name="check_interval"
                                render={({ field }) => (
                                    <FormItem className="">
                                        <FormLabel className="">{t('Check Interval')}</FormLabel>
                                        <FormControl>
                                            <Input {...field} className="border-border" />
                                        </FormControl>
                                    </FormItem>
                                )}
                            />
                            <FormField
                                control={form.control}
                                name="lookback_period"
                                render={({ field }) => (
                                    <FormItem className="">
                                        <FormLabel className="">{t('Lookback Period')}</FormLabel>
                                        <FormControl>
                                            <Input {...field} className="border-border" />
                                        </FormControl>
                                    </FormItem>
                                )}
                            />
                        </div>
                        <div className="flex gap-2">
                            <FormField
                                control={form.control}
                                name="warning"
                                render={({ field }) => (
                                    <FormItem className="">
                                        <FormLabel>{t("Thresholds")}</FormLabel>
                                        <FormControl>
                                            <Input {...field} className="border-border" placeholder={t("Warning")} />
                                        </FormControl>
                                    </FormItem>
                                )}
                            />
                            <FormField
                                control={form.control}
                                name="critical"
                                render={({ field }) => (
                                    <FormItem className="self-end">
                                        <FormControl>
                                            <Input {...field} className="border-border" placeholder={t("Critical")} />
                                        </FormControl>
                                    </FormItem>
                                )}
                            />
                        </div>
                        <FormField
                            control={form.control}
                            name="method.type"
                            render={({ field }) => (
                                <FormItem className="">
                                    <FormLabel>{t("Notification Method")}</FormLabel>
                                    <Select value={field.value} onValueChange={field.onChange} defaultValue="none">
                                        <FormControl>
                                            <SelectTrigger className="border-border">
                                                <SelectValue defaultValue="none"/>
                                            </SelectTrigger>
                                        </FormControl>
                                        <SelectContent>
                                            <SelectItem value="none">{t("None")}</SelectItem>
                                            <SelectItem value="nrdp">{t("Nagios (send using NRDP)")}</SelectItem>
                                            <SelectItem value="exec">{t("Execute script")}</SelectItem>
                                            <SelectItem value="snmp">{t("Send SNMP trap")}</SelectItem>
                                            <SelectItem value="email">{t("Email users")}</SelectItem>
                                        </SelectContent>
                                    </Select>
                                </FormItem>
                            )}
                        />
                        {(form.watch("method.type") === 'nrdp') &&
                            <div className="flex flex-col gap-2">
                                <FormField
                                    control={form.control}
                                    name="method.server_id"
                                    render={({ field }) => (
                                        <FormItem className="">
                                            <FormLabel>{t("NRDP Server")}</FormLabel>
                                            <FormControl>
                                                <Select value={field.value} onValueChange={field.onChange} defaultValue={field.value}>
                                                    <FormControl>
                                                        <SelectTrigger className="border-border">
                                                            <SelectValue placeholder={t("Please select an NRDP server")} />
                                                        </SelectTrigger>
                                                    </FormControl>
                                                    <SelectContent>
                                                        {nrdpSummaryQuery.data.map((server) => {
                                                            return (
                                                                <SelectItem value={server.id}>{server.name}</SelectItem>
                                                            );
                                                        })}
                                                    </SelectContent>
                                                </Select>
                                            </FormControl>
                                        </FormItem>
                                    )}
                                />
                                <div className="flex gap-2">
                                    <FormField
                                        control={form.control}
                                        name="method.hostname"
                                        render={({ field }) => (
                                            <FormItem>
                                                <FormLabel>{t("Hostname")}</FormLabel>
                                                <FormControl>
                                                    <Input {...field} className="border-border" />
                                                </FormControl>
                                            </FormItem>
                                        )}
                                    />
                                    <FormField
                                        control={form.control}
                                        name="method.servicename"
                                        render={({ field }) => (
                                            <FormItem className="self-end">
                                                <FormLabel>{t("Servicename")}</FormLabel>
                                                <FormControl>
                                                    <Input {...field} className="border-border" />
                                                </FormControl>
                                            </FormItem>
                                        )}
                                    />
                                </div>
                            </div>
                        }
                        {(form.watch("method.type") === 'exec') &&
                            <div className="flex flex-col gap-2">
                                <FormField
                                    control={form.control}
                                    name="method.path"
                                    render={({ field }) => (
                                        <FormItem>
                                            <FormLabel>{t("Script")}</FormLabel>
                                            <FormControl>
                                                <Input {...field} className="border-border" placeholder="/usr/local/nagioslogserver/scripts/myscript.sh"/>
                                            </FormControl>
                                        </FormItem>
                                    )}
                                />
                                <FormField
                                    control={form.control}
                                    name="method.args"
                                    render={({ field }) => (
                                        <FormItem>
                                            <FormLabel>{t("Arguments")}</FormLabel>
                                            <FormControl>
                                                <div>
                                                    <Input {...field} className="border-border" placeholder="-H 192.168.0.1 -U test -p hello"/>
                                                    <div className="text-sm text-secondary-foreground/70 mt-2">
                                                        <div>{t("Alerts will automatically replace these placeholders")}:</div>
                                                        <div><strong>%count%</strong> - {t("The total # of events")}</div>
                                                        <div><strong>%status%</strong> -{t("The status (ok, warning, critical)")}</div>
                                                        <div><strong>%output%</strong> - {t("The output from the alert")}</div>
                                                        <div><strong>%lastrun%</strong> - {t("The timestamp of the last run")}</div>
                                                    </div>
                                                </div>
                                            </FormControl>
                                        </FormItem>
                                    )}
                                />
                            </div>
                        }
                        {(form.watch("method.type") === 'snmp') &&
                            <FormField
                                control={form.control}
                                name="method.snmp_receiver"
                                render={({ field }) => (
                                    <FormItem>
                                        <FormLabel>{t("Trap Receiver")}</FormLabel>
                                        <FormControl>
                                            <Select value={field.value} onValueChange={field.onChange} defaultValue={field.value}>
                                                <FormControl>
                                                    <SelectTrigger className="border-border">
                                                        <SelectValue placeholder={t("Please select an SNMP trap receiver")} />
                                                    </SelectTrigger>
                                                </FormControl>
                                                <SelectContent>
                                                    {snmpSummaryQuery.data.map((server) => {
                                                        return (
                                                            <SelectItem value={server.id}>{server.name}</SelectItem>
                                                        );
                                                    })}
                                                </SelectContent>
                                            </Select>
                                        </FormControl>
                                    </FormItem>
                                )}
                            />
                        }
                        {(form.watch("method.type") === 'email') &&
                            <div className="flex flex-col gap-2">
                                <FormField
                                    control={form.control}
                                    name="method.tpl_id"
                                    render={({ field }) => (
                                        <FormItem>
                                            <FormLabel>{t("Email Template")}</FormLabel>
                                            <FormControl>
                                                <Select value={field.value} onValueChange={field.onChange} defaultValue={field.value}>
                                                    <FormControl>
                                                        <SelectTrigger className="border-border">
                                                            <SelectValue placeholder={t("Select an email template")} />
                                                        </SelectTrigger>
                                                    </FormControl>
                                                    <SelectContent>
                                                        <SelectItem value="system">{t("System Default")}</SelectItem>
                                                        {templatesSummaryQuery.data.map((template) => {
                                                            return (
                                                                <SelectItem value={template.id}>{template.name}</SelectItem>
                                                            );
                                                        })}
                                                    </SelectContent>
                                                </Select>
                                            </FormControl>
                                        </FormItem>
                                    )}
                                />
                                <FormField
                                    control={form.control}
                                    name="method.user_ids"
                                    render={({ field }) => (
                                        <FormItem>
                                            <FormLabel>{t("Select Users")}</FormLabel>
                                            <FormControl>
                                                <MultiSelect
                                                    value={field.value}
                                                    options={usersSummaryQuery.data}
                                                    onValueChange={field.onChange}
                                                    defaultValue={field.value}
                                                    placeholder={t("Select users")}
                                                    animation={2}
                                                    maxCount={3}
                                                />
                                            </FormControl>
                                        </FormItem>
                                    )}
                                />
                            </div>
                        }
                        <FormField
                            control={form.control}
                            name="alert_crit_only"
                            render={({ field }) => (
                                <FormItem>
                                    <FormControl>
                                    <div className="flex items-center space-x-2 mt-2">
                                        <Checkbox id="alert_crit_only" 
                                            checked={field.value}
                                            onCheckedChange={field.onChange}
                                            defaultChecked={false}
                                        />
                                        <label
                                            htmlFor="alert_crit_only"
                                            className="text-sm font-medium leading-none peer-disabled:cursor-not-allowed peer-disabled:opacity-70"
                                        >
                                            {t("Only alert when Warning or Critical threshold is met.")}
                                        </label>
                                    </div>
                                    </FormControl>
                                </FormItem>
                            )}
                        />
                        <DialogFooter className="pt-2">
                            <Button type="submit" className="justify-end">
                                {t('Create Alert')}
                            </Button>
                        </DialogFooter>
                    </form>
                </Form>
            </DialogContent>
        </Dialog>
    );
};
