import { ReactNode, createContext, useContext, useState } from "react";
import moment from 'moment';
import { useTranslation } from 'react-i18next';
import { useQuery } from "@tanstack/react-query";
import { AxiosError } from "axios";
import { filter, field } from "@/dashboard/filters/types";
import { subQuery } from "@/dashboard/queries/types";
import { Range } from '@/lib/types';
import { getDefaultDashboard } from '@/dashboard/queries';

const now = moment();
const aDayAgo = moment(now).subtract(1, 'day');

type dashboardContextType = {
    dashboardInfo: {
        filters: filter[],
        setFilters: React.Dispatch<React.SetStateAction<filter[]>>,
        subQueries: subQuery[],
        setSubQueries: React.Dispatch<React.SetStateAction<subQuery[]>>,
        range: Range,
        setRange: React.Dispatch<React.SetStateAction<Range>>,
        fields: field[],
        setFields: React.Dispatch<React.SetStateAction<field[]>>,
        refreshRate: number,
        setRefreshRate: React.Dispatch<React.SetStateAction<number>>,
    },
    dashboardMetaInfo: {
        defaultDashboardId: string,
        setDefaultDashboardId: React.Dispatch<React.SetStateAction<string>>,
        isReport: boolean,
        setIsReport: React.Dispatch<React.SetStateAction<boolean>>,
        isExporting: boolean,
        setIsExporting: React.Dispatch<React.SetStateAction<boolean>>
    }
}

const browserTimezone = Intl.DateTimeFormat().resolvedOptions().timeZone;

// allows for customization of the locale time format while still keeping a standard display throughout. returns a function one can use to format Dates
// precision : defines the returned formats precision as a range (['month', 'minute']), singular value (['hour']), or left undefined (full date time)
// ops : overwrites or adds additional options to the formatting, such as overwriting month to display as 'short' instead of '2-digit'
export const getLocaleTimeFormat = (precision?: [string, string] | string, ops?: Intl.DateTimeFormatOptions) => {
    const options : Intl.DateTimeFormatOptions = Object.assign({}, { timeZone: browserTimezone, year:'numeric', month:'2-digit', day:'2-digit', hour: '2-digit', minute: '2-digit', second: '2-digit' }, ops);
    const preciseKeys : string[] = ['timeZone'];
    if (precision?.length == 2) {
        const keys = Object.keys(options);
        keys.slice(keys.indexOf(precision[0]) != -1 ? keys.indexOf(precision[0]) : 0, keys.indexOf(precision[1]) != -1 ? keys.indexOf(precision[1]) + 1 : -1)
            .forEach(k =>  preciseKeys.push(k) )
    }
    else if (precision?.length == 1) {
        preciseKeys.push(precision);
    }
    const preciseOptions = Object.fromEntries(preciseKeys.map((k) => { return [k, options[k as keyof Intl.DateTimeFormatOptions]] }))
    const timeFormat = new Intl.DateTimeFormat(navigator.languages, precision? preciseOptions : options);
    return timeFormat;
};

const DashboardContext = createContext<dashboardContextType | undefined>(undefined);

export const useDashboardContext = () => {
    const context = useContext(DashboardContext);
    if (context === undefined) {
        throw new Error('Dashboard context not found');
    }
    return context;
}

export const DashboardProvider = ( { children }: { children: ReactNode}) => {

    const [filters, setFilters] = useState<filter[]>([]);
    const [fields, setFields] = useState<field[]>([]);

    const defaultSubQuery: subQuery = { id: 'id0', query: '*', color: '#1cd98c' };
    const [subQueries, setSubQueries] = useState<subQuery[]>([defaultSubQuery]);

    const { t } = useTranslation();
    const initialRange: Range = {
        startTime: aDayAgo,
        endTime: now,
        value: "24h",
        label: t("Last 24h"),
        reset: false
    }
    const [range, setRange] = useState(initialRange);

    const [defaultDashboardId, setDefaultDashboardId] = useState('');
    useQuery<string, AxiosError>({
        queryKey: ['defaultDashboardId'],
        queryFn: async () => {
            const response = await getDefaultDashboard();
            setDefaultDashboardId(response.data.dashboard || '');
            return response.data;
        }
    });

    const [isReport, setIsReport] = useState(false);
    const [isExporting, setIsExporting] = useState(false);
    const [refreshRate, setRefreshRate] = useState(0);

    return (
        <DashboardContext.Provider value=
            {
                {
                    dashboardInfo: {
                        filters, setFilters,
                        subQueries, setSubQueries,
                        range, setRange,
                        fields, setFields,
                        refreshRate, setRefreshRate
                    },
                    dashboardMetaInfo: {
                        defaultDashboardId, setDefaultDashboardId,
                        isReport, setIsReport,
                        isExporting, setIsExporting
                    }
                }
            }
        >
            {children}
        </DashboardContext.Provider>
    )
}