import { ReactNode, createContext, useContext, useEffect, useState } from "react";
import { useDashboardContext } from "@/dashboard/contexts/DashboardContext";

type subQueryData = {
    query_string: {
        query: string;
    };
}

type filterMatch = {
    match: {
        field: string;
    }
};

type filterRange = {
    range: {
        '@timestamp': {
            from: number,
            to: number
        }
    }
}

type dashboardQuery = {
    query: {
        bool: {
            must: [
                {
                    bool: {
                        should: subQueryData[]
                    }
                },
                {
                    bool: {
                        [key: string]: (filterMatch | filterRange)[]
                    }
                }
            ]
        }
    }
}

const emptyDashboardQuery: dashboardQuery = {
    query: {
        bool: {
            must: [
                {
                    bool: {
                        should: []
                    }
                },
                {
                    bool: {
                        must: []
                    }
                }
            ]
        }
    }
}

const DashboardQueryContext = createContext<dashboardQuery>(emptyDashboardQuery);

export const useDashboardQueryContext = () => {
    return useContext(DashboardQueryContext);
}

export const DashboardQueryProvider = ( { children }: { children: ReactNode}) => {
    const [dashboardQuery, setDashboardQuery] = useState(initialDashboardQuery);
    const { filters, subQueries, range } = useDashboardContext().dashboardInfo;
    useEffect(() => {
    
        const queryRange: filterRange = {
            range: {
                '@timestamp': {
                    from: range.startTime.toDate().getTime(),
                    to: range.endTime.toDate().getTime()
                }
            }
        }
    
        const newDashboardQuery: dashboardQuery = {
            query: {
                bool: {
                    must: [
                        {
                            bool: {
                                should: []
                            }
                        },
                        {
                            bool: {
                                must: [
                                    queryRange
                                ]
                            }
                        }
                    ]
                }
            }
        }
    
        for (const subQuery of subQueries) {
            const newSubQuery: subQueryData = {
                query_string:
                    {
                        query: subQuery.query
                    }
            };
            newDashboardQuery.query.bool.must[0].bool.should.push(newSubQuery);
        }
        
        for (const filter of filters) {
            const newFilter: filterMatch = {
                match: {
                    field: filter.query
                }
            }
            if (!(filter.mandate in newDashboardQuery.query.bool.must[1].bool)) {
                newDashboardQuery.query.bool.must[1].bool[filter.mandate] = [];
            }
            newDashboardQuery.query.bool.must[1].bool[filter.mandate].push(newFilter);
        }
        setDashboardQuery(newDashboardQuery);
    }, [filters, subQueries, range]);

    return (
        <DashboardQueryContext.Provider value={dashboardQuery}>
            {children}
        </DashboardQueryContext.Provider>
    )
}

const initialDashboardQuery = emptyDashboardQuery;