import { UserEasyButtonsToolbar, DefaultEasyButtonsToolbar } from '@/dashboard/easybuttons/EasyButtonsToolbar';
import { FilterToolbar } from '@/dashboard/filters/FilterToolbar';
import { useTranslation } from 'react-i18next';
import {
    Accordion,
} from "@/components/shadcn/accordion";
import {
  Sidebar,
  SidebarContent,
  SidebarGroup,
  SidebarGroupContent,
  SidebarGroupLabel,
  SidebarMenu,
} from "@/components/shadcn/sidebar";
import {
    Tooltip,
    TooltipContent,
    TooltipTrigger,
} from '@/components/shadcn/tooltip';
import { AxiosError } from 'axios';
import { toast } from '@/components/shadcn/use-toast';
import {
    DropdownMenu,
    DropdownMenuContent,
    DropdownMenuItem,
    DropdownMenuSeparator,
    DropdownMenuTrigger,
} from '@/components/shadcn/dropdown-menu';
import { AddDashboardDialogContent } from '@/dashboard/AddDashboardDialogContent';
import { Link } from 'react-router-dom';
import { DashboardResult } from '@/lib/types';
import { getDashboards, setDefaultDashboard } from '@/dashboard/queries';
import { useMutation, useQuery } from '@tanstack/react-query';
import CloseIcon from '@mui/icons-material/Close';
import PublicIcon from '@mui/icons-material/Public'
import React, { useState } from 'react';
import { useDashboardContext } from '@/dashboard/contexts/DashboardContext';
import { Button } from '@/components/shadcn/button';
import AddIcon from '@mui/icons-material/Add';
import SpaceDashboardIcon from '@mui/icons-material/SpaceDashboard';
import UnfoldMoreOutlinedIcon from '@mui/icons-material/UnfoldMoreOutlined';
import { Dialog, DialogTrigger } from '@/components/shadcn/dialog';
import { ScrollArea } from '@/components/shadcn/scroll-area';
import { DeleteDashboardDialog } from './DeleteDashboardDialog';

export function DashboardSidebar({ dashboardId, dashboardName, user }: { dashboardId: string; dashboardName: string; user: string | undefined }) {
  const { t } = useTranslation();
  const [open, setOpen] = useState(false);
  const { defaultDashboardId, setDefaultDashboardId } = useDashboardContext().dashboardMetaInfo;
  const subtitle = (defaultDashboardId === dashboardId) ? t("Default") : user;
  
  const dashboardsSummaryQuery = useQuery<{id: string, name: string, user: string, can_user_delete: boolean}[], AxiosError>({
    queryKey: ['dashboards'],
    queryFn: async () => {
        const response = await getDashboards();
        return response.data.hits.hits.filter((dashboardHit: DashboardResult) => {
          return typeof(dashboardHit._source.userselectable) === 'undefined' || dashboardHit._source.userselectable !== false;
        })
        .map((dashboardHit: DashboardResult) => {
          return {
              id: dashboardHit._id,
              name: dashboardHit._source.title,
              user: dashboardHit._source.user,
              can_user_delete: dashboardHit._source.can_user_delete && dashboardHit._id !== 'blank' || false,
          };
        })
    },
  });

  const setDefaultDashboardMutation = useMutation({
    mutationFn: setDefaultDashboard,
    onSuccess: (response) => {
        if (response.data.status === 'success') {
          setDefaultDashboardId(dashboardId);
        }
        toast({
            title: t('Set default dashboard.'),
            variant: 'success',
        });
    },
    onError: (error: AxiosError<{ message: string }>) => {
        toast({
            title: t('Failed to set default dashboard.'),
            variant: 'error',
            description: error.response?.data.message,
        });
    },
  });

  const onClickSetDefaultDashboard = (e: React.MouseEvent) => {
    e.preventDefault();
    setDefaultDashboardMutation.mutate(dashboardId);
  }

  if (!dashboardsSummaryQuery.data) {
    return;
  }

  return (
    <Sidebar className="bg-secondary">
      <SidebarContent>
        <SidebarGroup>
          <SidebarGroupLabel className="px-0 mt-1">
            <Dialog>
              <DropdownMenu open={open} onOpenChange={setOpen} modal={false}>
                <DropdownMenuTrigger asChild>
                  <Button className="flex justify-between items-center w-full p-2 h-11" variant="ghost">
                    <div className="flex gap-2 items-center">
                      {!(dashboardId === defaultDashboardId) &&
                        <Tooltip>
                          <TooltipTrigger asChild>
                            <Button
                              onPointerDown={(e) => onClickSetDefaultDashboard(e)}
                              className="flex aspect-square size-8 items-center justify-center rounded-lg bg-background border border-primary text-primary hover:text-background hover:bg-primary transition-colors duration-300"
                            >
                              <SpaceDashboardIcon fontSize="small" />
                            </Button>
                          </TooltipTrigger>
                        <TooltipContent>
                          {t("Set as default dashboard")}
                        </TooltipContent>
                      </Tooltip>
                      }
                      {dashboardId === defaultDashboardId &&
                        <Tooltip>
                          <TooltipTrigger asChild>
                            <Button className="flex aspect-square size-8 items-center justify-center rounded-lg bg-primary text-background border border-primary">
                              <SpaceDashboardIcon fontSize="small" />
                            </Button>
                          </TooltipTrigger>
                          <TooltipContent>
                            {t("Your default dashboard!")}
                          </TooltipContent>
                        </Tooltip>
                      }
                      <div className="grid flex-1 text-left">
                        <span className="truncate text-semibold">{dashboardName}</span>
                        <span className="truncate text-xs text-foreground/70">{subtitle}</span>
                      </div>
                    </div>
                    <UnfoldMoreOutlinedIcon fontSize="small"/>
                  </Button>
                </DropdownMenuTrigger>
                <DropdownMenuContent side="right" className="mt-2">
                  <ScrollArea className="max-h-[75vh] overflow-y-auto">
                    <DialogTrigger asChild>
                      <DropdownMenuItem asChild className="flex gap-2">
                        <Button variant="ghost" className="justify-start w-full h-8 px-2 font-normal rounded gap-2">
                          <AddIcon fontSize="small" />
                          {t('New Dashboard')}
                        </Button>
                      </DropdownMenuItem>
                    </DialogTrigger>
                    <DropdownMenuSeparator />
                      {dashboardsSummaryQuery.data.map((dashboard: {id: string, name: string, user: string, can_user_delete :boolean}) => (
                        (dashboard.id != 'simple' && dashboard.id != 'home') &&
                        <div className="group flex justify-between items-center gap-2 rounded hover:bg-accent">
                          <Link to={`?dashboardId=${dashboard.id}`} className="grow">
                            <DropdownMenuItem onClick={() => setOpen(false)} className="flex items-center gap-2 hover:bg-transparent focus:bg-transparent">
                              {dashboard.id === defaultDashboardId && <SpaceDashboardIcon fontSize="small" className="text-primary" />}
                              {dashboard.user === "global" && <PublicIcon fontSize="small" />}
                              {dashboard.name}
                            </DropdownMenuItem>
                          </Link>
                          {dashboard.can_user_delete && <DeleteDashboardDialog dashboardId={dashboard.id}>
                            <Tooltip>
                              <TooltipTrigger>
                                <div className="rounded-md p-1 text-sm hover:cursor-pointer hover:bg-muted">
                                  <CloseIcon fontSize="small" />
                                </div>
                              </TooltipTrigger>
                              <TooltipContent>{t('Delete')}</TooltipContent>
                            </Tooltip>
                          </DeleteDashboardDialog>}
                        </div>
                      ))}
                    </ScrollArea>
                  </DropdownMenuContent>
                </DropdownMenu>
              <AddDashboardDialogContent />
            </Dialog>
          </SidebarGroupLabel>
          <SidebarGroupContent>
            <SidebarMenu>
              <Accordion type="multiple" className="w-full mt-2" defaultValue={["default_easybuttons", "user_easybuttons", "filters"]}>
                <DefaultEasyButtonsToolbar />
                <UserEasyButtonsToolbar />
                <FilterToolbar/>
              </Accordion>
            </SidebarMenu>
          </SidebarGroupContent>
        </SidebarGroup>
      </SidebarContent>
    </Sidebar>
  )
}
