import { Dialog, DialogContent, DialogDescription, DialogFooter, DialogHeader, DialogTitle, DialogTrigger } from '@/components/shadcn/dialog';
import { Input } from '@/components/shadcn/input';
import { Button } from '@/components/shadcn/button';
import { Textarea } from '@/components/shadcn/textarea';
import { Checkbox } from "@/components/shadcn/checkbox";
import { ReactNode, useState } from 'react';
import { useForm, SubmitHandler } from 'react-hook-form';
import { Form, FormLabel, FormControl, FormField, FormItem } from '@/components/shadcn/form';
import { useTranslation } from 'react-i18next';
import { toast } from '@/components/shadcn/use-toast';
import { easybutton } from '@/dashboard/easybuttons/types';
import { AxiosError } from 'axios';
import { addUserEasyButton, editUserEasyButton } from '@/dashboard/easybuttons/queries';
import { useMutation, useQueryClient } from '@tanstack/react-query';

/**
 * The easybutton dialog. Passing in an easybutton makes it an edit easybutton dialog. Otherwise it's an add easybutton dialog.
 */
export const AddEditUserEasyButtonDialog = ({ easybutton, children }: { easybutton?:easybutton; children: ReactNode }) => {
    const [open, setOpen] = useState(false);
    const { t } = useTranslation();
    const queryClient = useQueryClient();

    const addUserEasyButtonMutation = useMutation({
        mutationFn: addUserEasyButton,
        onSuccess: () => {
            queryClient.invalidateQueries({ queryKey: ['easybutton'] });
            setOpen(false);
            toast({
                title: t('Query Template added.'),
                variant: 'success',
                description: t('Click to toggle and right click to configure.'),
            });
        },
        onError: (error: AxiosError<{ message: string }>) => {
            toast({
                title: t('Failed to add  Query Template.'),
                variant: 'error',
                description: error.response?.data.message || t('An unknown error occured.'),
            });
        },
    });

    const editUserEasyButtonMutation = useMutation({
        mutationFn: editUserEasyButton,
        onSuccess: () => {
            queryClient.invalidateQueries({ queryKey: ['easybutton'] });
            setOpen(false);
            toast({
                title: t(' Query Template edited.'),
                variant: 'success',
                description: t('Click to toggle and right click to configure.'),
            });
        },
        onError: (error: AxiosError<{ message: string }>) => {
            toast({
                title: t('Failed to edit  Query Template.'),
                variant: 'error',
                description: error.response?.data.message || t('An unknown error occured.'),
            });
        },
    });

    const form = useForm<easybutton>({
        defaultValues: {
            name: easybutton ? easybutton.name : '',
            query: easybutton ? easybutton.query : '',
            global: easybutton ? easybutton.global : false
        },
    });

    const onSubmit: SubmitHandler<easybutton> = (data) => {
        if (easybutton) {
            editUserEasyButtonMutation.mutate({ id: easybutton.id, name: data.name, query: data.query, global: data.global });
        } else {
            addUserEasyButtonMutation.mutate({ name: data.name, query: data.query, global: data.global });
        }
    };

    return (
        <Dialog open={open} onOpenChange={setOpen}>
            <DialogTrigger asChild>
                {children}
            </DialogTrigger>
            <DialogContent className="sm:max-w-[425px]">
                <DialogHeader>
                    <DialogTitle>{easybutton ? t('Edit Query Template') : t('Add  Query Template')}</DialogTitle>
                    {!easybutton && <DialogDescription>{t('Use this to add a new query template!')}</DialogDescription>}
                </DialogHeader>
                <Form {...form}>
                    <form onSubmit={form.handleSubmit(onSubmit)} className="grid gap-4 py-2">
                        <FormField
                            control={form.control}
                            name="name"
                            render={({ field }) => (
                                <FormItem className="grid grid-cols-4 items-center gap-4 space-y-0">
                                    <FormLabel className="text-right">{t('Name')}</FormLabel>
                                    <FormControl>
                                        <Input {...field} className="col-span-3 border-border" />
                                    </FormControl>
                                </FormItem>
                            )}
                        />
                        <FormField
                            control={form.control}
                            name="query"
                            render={({ field }) => (
                                <FormItem className="grid grid-cols-4 items-start gap-4 space-y-0">
                                    <FormLabel className="text-right mt-2">{t('Query')}</FormLabel>
                                    <FormControl>
                                        <Textarea
                                            placeholder={t('Write your query here.')}
                                            className="resize-none col-span-3 border-border placeholder:text-foreground/30"
                                            {...field}
                                        />
                                    </FormControl>
                                </FormItem>
                            )}
                        />
                        <FormField
                            control={form.control}
                            name="global"
                            render={({ field }) => (
                                <FormItem className="grid grid-cols-4 items-center gap-4 space-y-0">
                                    <FormLabel className="text-right">{t('Global')}</FormLabel>
                                    <FormControl>
                                        <Checkbox id="global" 
                                            checked={field.value}
                                            onCheckedChange={field.onChange}
                                            defaultChecked={false}
                                        />
                                    </FormControl>
                                </FormItem>
                            )}
                        />
                        <DialogFooter>
                            <Button type="submit" className="justify-end">
                                {t('Submit')}
                            </Button>
                        </DialogFooter>
                    </form>
                </Form>
            </DialogContent>
        </Dialog>
    );
};
