import { Toggle } from '@/components/shadcn/toggle';
import {
    ContextMenu,
    ContextMenuContent,
    ContextMenuItem,
    ContextMenuTrigger,
} from "@/components/shadcn/context-menu";
import {
    AlertDialog,
    AlertDialogContent,
    AlertDialogDescription,
    AlertDialogFooter,
    AlertDialogHeader,
    AlertDialogTrigger,
    AlertDialogAction,
    AlertDialogCancel,
    AlertDialogTitle
} from '@/components/shadcn/alert-dialog';
import { ChevronDown } from "lucide-react";
import {
  AccordionContent,
  AccordionItem,
  AccordionTrigger,
} from "@/components/shadcn/accordion";
import { Button } from '@/components/shadcn/button';
import { AddEditUserEasyButtonDialog } from '@/dashboard/easybuttons/AddEditUserEasyButtonDialog';
import { DeleteUserEasyButtonDialog } from '@/dashboard/easybuttons/DeleteUserEasyButtonButton';
import { useState } from 'react';
import { easybutton } from '@/dashboard/easybuttons/types';
import { AxiosError } from 'axios';
import { easyButtonStore } from '@/dashboard/easybuttons/store';
import { getUserEasyButtons } from '@/dashboard/easybuttons/queries';
import { useQuery } from '@tanstack/react-query';
import { useTranslation } from 'react-i18next';
import AddIcon from '@mui/icons-material/Add';
import { useDashboardContext } from '@/dashboard/contexts/DashboardContext';
import { colors } from '@/dashboard/queries/store';
import { defaultEasyButtons } from "./queries";

const EasyButtonToggle = ({ id, name, query, icon }: { id: string, name: string, query: string, icon?: string }) => {

    const { subQueries, setSubQueries } = useDashboardContext().dashboardInfo;
    const active = subQueries.some(subQuery => subQuery.id === id);
    
    const onToggle = () => {
        if (active) {
            setSubQueries(
                subQueries.filter(q =>
                  q.id !== id
                )
            );
        } else {
            setSubQueries([
                ...subQueries,
                { id: id, query: query, color: colors[subQueries.length % 6]}]
            );
        }
    }
    return (
        <Toggle pressed={active}
            className="gap-2 rounded bg-border/30 data-[state=on]:bg-border text-sm py-1 px-2 h-7 font-normal hover:text-foreground/90 dark:hover:bg-border"
            onPressedChange={onToggle}>
            {(icon && icon === "public") ? 
                <i className="material-symbols-outlined text-sm">{icon}</i>
            : 
                undefined}
            {name}
        </Toggle>
    )
}

export const DefaultEasyButtonsToolbar = () => {

    const { t } = useTranslation();
    const {display} = easyButtonStore();

    return (
        <AccordionItem value="default_easybuttons">
            <AccordionTrigger className="p-2 mt-2 mb-4 text-sm flex justify-start gap-2 hover:no-underline">
                <ChevronDown className="h-5 w-5 shrink-0 transition-transform duration-200 hover:cursor-pointer" />
                {t("Default Query Templates")}
            </AccordionTrigger>
            <AccordionContent>
                <div className="flex flex-col gap-2">
                    <div className="h-full flex gap-1 flex-wrap">
                        {display.show_default_buttons && defaultEasyButtons.map((button) => (
                            <EasyButtonToggle id={button.id} name={button.name} query={button.query} icon={button.icon} />
                        ))}
                    </div>
                </div>
            </AccordionContent>
        </AccordionItem>
    );
}

export const UserEasyButtonsToolbar = () => {
    const { t } = useTranslation();
    const [alertOpen, setAlertOpen] = useState(false);
    
    const userEasyButtonsSummaryQuery = useQuery<easybutton[], AxiosError>({
        queryKey: ['easybutton'],
        queryFn: async () => {
            const response = await getUserEasyButtons();
            return response.data;
        },
    });

    return (
        <AccordionItem value="user_easybuttons">
            <AccordionTrigger className="p-2 mt-2 mb-4 text-sm flex gap-2 hover:no-underline justify-between">
                <ChevronDown className="h-5 w-5 shrink-0 transition-transform duration-200 hover:cursor-pointer" />
                <p className="">{t("Custom Query Templates")}</p>
                <div className="ml-auto" onClick={(e) => e.stopPropagation()}>
                    <AddEditUserEasyButtonDialog>
                        <Button size="icon" variant="ghost" className="h-7 w-7">
                            <AddIcon fontSize="small" />
                        </Button>
                    </AddEditUserEasyButtonDialog>
                </div>
            </AccordionTrigger>
            <AccordionContent>
                <div className="flex flex-col">
                    <div className="h-full flex gap-1 flex-wrap">
                        {userEasyButtonsSummaryQuery.data ? (
                            userEasyButtonsSummaryQuery.data.map((button) => (
                                <AlertDialog open={alertOpen} onOpenChange={setAlertOpen}>
                                    <ContextMenu key={button.id} modal={false}>
                                        <ContextMenuTrigger>
                                            <EasyButtonToggle id={button.id} name={button.name} query={button.query} icon={button.icon} />
                                        </ContextMenuTrigger>
                                        <ContextMenuContent className="p-1 flex flex-col w-fit items-start">
                                            <ContextMenuItem asChild>
                                                <AddEditUserEasyButtonDialog easybutton={button}>
                                                    <Button variant="ghost" className="justify-start w-full h-8 px-2 font-normal rounded">{t('Edit')}</Button>
                                                </AddEditUserEasyButtonDialog>
                                            </ContextMenuItem>
                                            <AlertDialogTrigger asChild>
                                                <ContextMenuItem asChild>
                                                    <Button variant="ghost" className="justify-start w-full h-8 px-2 font-normal rounded">{t('Delete')}</Button>
                                                </ContextMenuItem>
                                            </AlertDialogTrigger>
                                        </ContextMenuContent>
                                    </ContextMenu>
                                    <AlertDialogContent>
                                        <AlertDialogHeader>
                                            <AlertDialogTitle>{t('Confirm Query Template Deletion')}</AlertDialogTitle>
                                            <AlertDialogDescription>{t('Are you sure you want to delete this query template?')}</AlertDialogDescription>
                                        </AlertDialogHeader>
                                        <AlertDialogFooter>
                                            <AlertDialogCancel asChild>
                                                <Button variant="outline">{t('Cancel')}</Button>
                                            </AlertDialogCancel>
                                            <AlertDialogAction asChild>
                                                <DeleteUserEasyButtonDialog id={button.id} setOpen={setAlertOpen}/>
                                            </AlertDialogAction>
                                        </AlertDialogFooter>
                                    </AlertDialogContent>
                                </AlertDialog>
                            ))
                        ) : (
                            <div className="bg-accent rounded-md h-7 w-20"></div>
                        )}
                    </div>
                </div>
            </AccordionContent>
        </AccordionItem>
    )
}