import { Dialog, DialogContent, DialogDescription, DialogFooter, DialogHeader, DialogTitle, DialogTrigger } from '@/components/shadcn/dialog';
import { Button } from '@/components/shadcn/button';
import {
    Select,
    SelectContent,
    SelectGroup,
    SelectItem,
    SelectTrigger,
    SelectValue,
} from "@/components/shadcn/select";
import {
    Command,
    CommandEmpty,
    CommandGroup,
    CommandInput,
    CommandItem,
    CommandList,
} from "@/components/shadcn/command";
import {
    Popover,
    PopoverContent,
    PopoverTrigger,
} from "@/components/shadcn/popover";
import { ReactNode, useState } from 'react';
import { useForm, SubmitHandler } from 'react-hook-form';
import { Form, FormControl, FormField, FormItem } from '@/components/shadcn/form';
import { useTranslation } from 'react-i18next';
import { toast } from '@/components/shadcn/use-toast';
import { filter, field } from '@/dashboard/filters/types';
import { Input } from '@/components/shadcn/input';
import { Check, ChevronsUpDown } from "lucide-react";
import { cn } from "@/lib/utils";
import { useSuspenseQuery } from '@tanstack/react-query';
import { AxiosError } from 'axios';
import { getFields } from '@/dashboard/filters/queries';
import { useDashboardContext } from '@/dashboard/contexts/DashboardContext';

type addEditFilterDialogTypes = {
    filter?: filter,
    filters: filter[],
    setFilters: (filters: filter[]) => void,
    children?: ReactNode,
    dropdown?: ReactNode 
};

/**
 * The filter dialog. Passing in a filter makes it an edit filter dialog. Otherwise it's an add filter dialog.
 */
export default function AddEditFilterDialog({ filter, filters, setFilters, children, dropdown }: addEditFilterDialogTypes ) {

    const [open, setOpen] = useState(false);
    const [popoverOpen, setPopoverOpen] = useState(false);
    const { t } = useTranslation();
    const { setFields, range } = useDashboardContext().dashboardInfo;

    const getNestedFieldNames = (obj: Object, fields: Set<string>, parent_name: string) => {
        if (!obj) {
          return;
        }
        Object.keys(obj).forEach(key => {
          const name = parent_name ? parent_name + '.' + key : key;
          fields.add(name);
    
          // If the field is an object, recursively call getFieldNames
          if (key !== null && obj[key as keyof Object] !== null && obj[key as keyof Object] !== undefined && typeof obj[key as keyof Object] === 'object') {
            getNestedFieldNames(obj[key as keyof Object], fields, name);  // Recurse into nested object or array
          }
        });
      }

    const fieldsSummaryQuery = useSuspenseQuery<field[], AxiosError>({
        queryKey: ['fields'],
        queryFn: async () => {
          const reqBody = {
            "@timestamp": {
                "from": range.startTime.toDate().getTime(), 
                "to": range.endTime.toDate().getTime(), 
            },
            "queries": [],
            "filters": []
          };
          const response = await getFields(reqBody);
          const fields = new Set<string>(['_index', '_id']);
          if (response && response.data) {
            response.data.hits.hits.forEach((hit: {_source: []}) => {
              getNestedFieldNames(hit._source, fields, '');
            });
          }
          const mappedFields = [...fields].map(field => ({name: field}));
          setFields(mappedFields);
          return mappedFields;
        }
    });

    const form = useForm<filter>({
        defaultValues: {
            field: filter ? filter.field : '',
            mandate: filter ? filter.mandate : '',
            query: filter ? filter.query : ''
        },
    });

    const onSubmit: SubmitHandler<filter> = (data) => {
        let error = 0;
        let error_string = '';
        if (!data.field) {
            error++;
            error_string = t('You must provide a field.')
        }
        if (!data.mandate) {
            error++;
            error_string = t('You must provide a clause.')
        }
        if (!data.query) {
            error++;
            error_string = t('You must provide a value.')
        }
        if (error) {
            toast({
                title: t('Failed to add Filter.'),
                variant: 'error',
                description: error_string,
            });
            return;
        }
        if (filter) {
            setFilters(filters.map(old_filter => 
                old_filter.id === filter.id ? { ...old_filter, field: data.field, mandate: data.mandate, query: data.query } : old_filter
            ));
            setOpen(false);
            toast({
                title: t('Filter edited.'),
                variant: 'success'
            });
        } else {
            setFilters([
                ...filters,
                { id: `id-filter${Date.now()}`, type: data.type, field: data.field, mandate: data.mandate, query: data.query, active: true }]
            );
            setOpen(false);
            toast({
                title: t('Filter added.'),
                variant: 'success'
            });
        }
    };

    if (!fieldsSummaryQuery.isSuccess) {
        return;
    }

    return (
        <Dialog open={open} onOpenChange={setOpen}>
            {dropdown ? (
                dropdown
            ) : (
                <DialogTrigger asChild>
                    {children}
                </DialogTrigger>
            )}
            {fieldsSummaryQuery.isFetching ? (
                null
            ) : (
                <DialogContent className="sm:max-w-[650px]">
                    <DialogHeader>
                        <DialogTitle>{filter ? t('Edit Filter') : t('Add Filter')}</DialogTitle>
                        {!filter && <DialogDescription>{t('Use this to add a new Filter!')}</DialogDescription>}
                    </DialogHeader>
                    <Form {...form}>
                        <form onSubmit={form.handleSubmit(onSubmit)} className="grid gap-4 py-2">
                            <div className="flex items-center gap-2">
                                <FormField
                                    control={form.control}
                                    name="field"
                                    render={({ field }) => (
                                        <FormItem className="">
                                            <FormControl>
                                                <Popover open={popoverOpen} onOpenChange={setPopoverOpen}>
                                                    <PopoverTrigger asChild>
                                                        <FormControl>
                                                            <Button
                                                                variant="secondary"
                                                                role="combobox"
                                                                className={cn(
                                                                    "w-[200px] justify-between",
                                                                    !field.value && "text-muted-foreground"
                                                                )}
                                                                >
                                                                <div className="max-w-[180px] truncate">
                                                                    {field.value
                                                                        ? fieldsSummaryQuery.data.find(
                                                                            (filterfield) => filterfield.name === field.value
                                                                        )?.name
                                                                        : t("Select field")}
                                                                </div>
                                                                <ChevronsUpDown className="ml-2 h-4 w-4 shrink-0 opacity-50" />
                                                            </Button>
                                                        </FormControl>
                                                    </PopoverTrigger>
                                                    <PopoverContent className="w-max p-0">
                                                        <Command>
                                                            <CommandInput placeholder={t("Search field") + '...'} />
                                                            <CommandList>
                                                            <CommandEmpty>{t("No fields found.")}</CommandEmpty>
                                                            <CommandGroup>
                                                                {fieldsSummaryQuery.data.map((filterfield) => (
                                                                    <CommandItem
                                                                        value={filterfield.name}
                                                                        key={filterfield.name}
                                                                        onSelect={() => {
                                                                            form.setValue("field", filterfield.name);
                                                                            setPopoverOpen(false);
                                                                        }}
                                                                    >
                                                                        <Check
                                                                        className={cn(
                                                                            "mr-2 h-4 w-4",
                                                                            filterfield.name === field.value
                                                                            ? "opacity-100"
                                                                            : "opacity-0"
                                                                        )}
                                                                        />
                                                                        {filterfield.name}
                                                                    </CommandItem>
                                                                ))}
                                                            </CommandGroup>
                                                            </CommandList>
                                                        </Command>
                                                    </PopoverContent>
                                                </Popover>
                                            </FormControl>
                                        </FormItem>
                                    )}
                                />
                                <FormField
                                    control={form.control}
                                    name="mandate"
                                    render={({ field }) => (
                                        <FormItem className="">
                                            <FormControl>
                                                <Select value={field.value} onValueChange={field.onChange} defaultValue={"must"}>
                                                    <SelectTrigger className="w-fit gap-2 bg-secondary hover:bg-secondary/80 outline-none ring-0 ring-offset-0 focus:ring-offset-0 focus:ring-0">
                                                        <SelectValue defaultValue="must" placeholder={field.value} className="whitespace-nowrap"/>
                                                    </SelectTrigger>
                                                    <SelectContent>
                                                        <SelectGroup>
                                                            <SelectItem value="must" >{t("must")}</SelectItem>
                                                            <SelectItem value="must_not">{t("must not")}</SelectItem>
                                                            <SelectItem value="should">{t("should")}</SelectItem>
                                                        </SelectGroup>
                                                    </SelectContent>
                                                </Select>
                                            </FormControl>
                                        </FormItem>
                                    )}
                                />
                                <FormField
                                    control={form.control}
                                    name="query"
                                    render={({ field }) => (
                                        <FormItem className="grow">
                                            <FormControl>
                                                <Input {...field} className="min-w-[200px] bg-secondary" />
                                            </FormControl>
                                        </FormItem>
                                    )}
                                />
                            </div>
                            <DialogFooter>
                                <Button type="submit" className="justify-end">
                                    {filter ? t('Edit Filter') : t('Add Filter')}
                                </Button>
                            </DialogFooter>
                        </form>
                    </Form>
                </DialogContent>
            )}
        </Dialog>
    );
};
