import { Button } from '@/components/shadcn/button';
import{ filter } from '@/dashboard/filters/types';
import AddIcon from '@mui/icons-material/Add';
import { useDashboardContext } from '@/dashboard/contexts/DashboardContext';
import { Checkbox } from "@/components/shadcn/checkbox";
import MoreVertIcon from '@mui/icons-material/MoreVert';
import { ChevronDown } from "lucide-react";
import {
  AccordionContent,
  AccordionItem,
  AccordionTrigger,
} from "@/components/shadcn/accordion";
import {
  DropdownMenu,
  DropdownMenuContent,
  DropdownMenuItem,
  DropdownMenuTrigger,
} from "@/components/shadcn/dropdown-menu";
import { useTranslation } from 'react-i18next';
import { DialogTrigger } from '@/components/shadcn/dialog';
import { cn } from '@/lib/utils';
import { lazy } from 'react';
import React from 'react';

const AddEditFilterDialog = lazy(() => import('@/dashboard/filters/AddEditFilterDialog'))

const FilterBox = ({ filter, filters, setFilters }: { filter: filter; filters: filter[]; setFilters: (filters: filter[]) => void;}) => {
  const { t } = useTranslation();
  const dropdown = (
    <DropdownMenu modal={false}>
      <DropdownMenuTrigger asChild>
        <Button size="icon" variant="ghost" className="h-7 w-7">
          <MoreVertIcon fontSize="small"/>
        </Button>
      </DropdownMenuTrigger>
      <DropdownMenuContent>
        <DialogTrigger asChild>
          <DropdownMenuItem>{t("Edit")}</DropdownMenuItem>
        </DialogTrigger>
        <DropdownMenuItem onClick={() =>
          setFilters(
            filters.filter(q =>
              q.id !== filter.id
            )
          )}>{t("Delete")}
        </DropdownMenuItem>
      </DropdownMenuContent>
    </DropdownMenu>
  );
  const handleToggleFilter = (id: string) => {
    const newFilters = [...filters];
    const item = newFilters.find(
      a => a.id === id
    );
    item && (item.active = !item.active);
    setFilters(newFilters);
  }
  return (
    <div className="flex items-center space-x-2 px-2">
      <Checkbox
        id={filter.id}
        checked={filter.active}
        onCheckedChange={() => handleToggleFilter(filter.id)}
      />
      <div className="w-full flex items-center justify-between">
        <label
          htmlFor={filter.id}
          className="text-sm font-medium leading-none peer-disabled:cursor-not-allowed peer-disabled:opacity-70 flex items-center space-x-2"
        >
          <div className="max-w-[62px] truncate">{filter.field}</div>
          <span className={cn("rounded-md p-1 border", filter.mandate === "must" ? "text-success bg-success-background border-success" : filter.mandate === "must_not" ? "text-error bg-error-background border-error" : "text-warning bg-warning-background border-warning")}>
            {filter.mandate.replace(/_/g, ' ')}
          </span>
          <div className={cn("truncate", filter.mandate === 'must' ? "max-w-[83px]" : filter.mandate === "must_not" ? "max-w-[53.5px]" : "max-w-[71px]")}>{filter.query}</div>
        </label>
        <React.Suspense fallback={<div className="bg-accent h-7 w-7 rounded-md"></div>}>
          <AddEditFilterDialog filter={filter} filters={filters} setFilters={setFilters} dropdown={dropdown} />
        </React.Suspense>
      </div>
    </div>
  )
}

export const FilterToolbar = () => {

  const { filters, setFilters } = useDashboardContext().dashboardInfo;
  const { t } = useTranslation();

  return (
    <AccordionItem value="filters">
      <AccordionTrigger className="p-2 mt-2 mb-4 text-sm flex gap-2 hover:no-underline justify-between">
        <ChevronDown className="h-5 w-5 shrink-0 transition-transform duration-200 hover:cursor-pointer" />
        <p className="">{t("Filters")}</p>
        <div className="ml-auto" onClick={(e) => e.stopPropagation()}>
          <React.Suspense fallback={<div className="bg-accent h-7 w-7 rounded-md"></div>}>
            <AddEditFilterDialog filters={filters} setFilters={setFilters}>
              <Button size="icon" variant="ghost" className="h-7 w-7">
                  <AddIcon fontSize="small" />
              </Button>
            </AddEditFilterDialog>
          </React.Suspense>
        </div>
      </AccordionTrigger>
      <AccordionContent>
        <div className="w-full flex flex-col gap-2">
          {filters.map((filter) => (
              <FilterBox key={filter.id} filter={filter} filters={filters} setFilters={setFilters}></FilterBox>
          ))}
        </div>
      </AccordionContent>
    </AccordionItem>
  )
}