import { TermConfig, TermPanelData } from '@/components/charts/Term';
import { TextPanelData, TextConfig } from '@/components/charts/Text';
import { HistogramConfig, HistogramPanelData } from '@/components/charts/BaseHistogram';
import { PanelConfig, PanelConfigData } from '@/lib/types';
import { StatsConfig, StatsPanelData } from '@/components/charts/Stats';
import { DataBoxConfig, dataBoxData } from '@/components/charts/DataBox';
import { LogTableConfig, LogTablePanelData } from '@/components/log-table/log-table';
import { TwoWeekConfig, TwoWeekPanelData } from '@/components/charts/other-charts/log_entries_2_weeks';
import { useTranslation } from 'react-i18next';
import {
    Sheet,
    SheetContent,
    SheetDescription,
    SheetHeader,
    SheetTitle,
    SheetTrigger,
} from "@/components/shadcn/sheet";
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from '@/components/shadcn/select';
import { dashboardStore } from '@/dashboard/Dashboard';
import { row } from '@/lib/types';
import { useState } from 'react';
import { MapChartConfig, MapPanelData } from '@/components/charts/map/MapChart';
import { ScrollArea } from '@/components/shadcn/scroll-area';

interface PanelData {
    name: string,
    description: string,
    defaultWidth: number,
    defaultHeight: number,
    icon?: string
}

interface PanelType {
    [key:string]: PanelData
}

const panelTypes: PanelType = {
//    'goal': {'name': 'Goal', defaultWidth: 4, defaultHeight: 4, description: 'Displays the progress towards a fixed goal on a pie chart'},
    'histogram': {'name': 'Histogram', defaultWidth: 16, defaultHeight: 12, description: 'A bucketed time series chart of the current query or queries. Uses the OpenSearch date_histogram facet. If using time stamped indices this panel will query them sequentially to attempt to apply the lighest possible load to your OpenSearch cluster', icon: 'fa-solid fa-chart-simple'},
//    'hits': {'name': 'Hits', defaultWidth: 4, defaultHeight: 4, description: 'The total hits for a query or set of queries. Can be a pie chart, bar chart, list, or absolute total of all queries combined'},
    'map': {'name': 'Map', defaultWidth: 12, defaultHeight: 12, description: 'Displays a map of shaded regions using a field containing a 2 letter country, or US state, code. Regions with more hit are shaded darker. Node that this does use the OpenSearch terms facet, so it is important that you set it to the correct field.'},
//    'sparklines': {'name': 'Sparklines', defaultWidth: 4, defaultHeight: 4, description: 'Sparklines are tiny, simple, time series charts, shown separately. Because sparklines are uncluttered by grids, axis markers and colors, they are perfect for spotting change in a series'},
    'stats': {'name': 'Stats', defaultWidth: 12, defaultHeight: 8, description: 'A statistical panel for displaying aggregations using the Open Search statistical facet query', icon: ''},
    'table': {'name': 'Table', defaultWidth: 16, defaultHeight: 12, description: 'A paginated table of records matching your query or queries. Click on a row to expand it and review all of the fields associated with that document.', icon: 'fa-thin fa-table'},
    'term': {'name': 'Terms', defaultWidth: 8, defaultHeight: 13, description: 'Displays the results of an OpenSearch facet as a pie chart, bar chart, or a table', icon: 'fa-thin fa-chart-pie'},
    'text': {'name': 'Text', defaultWidth: 12, defaultHeight: 4, description: 'A static text panel that can use plain text, markdown, or (sanitized) HTML', icon: 'fa-thin fa-text'},
    'twoweeklog': {'name': 'Two Week Log', defaultWidth: 8, defaultHeight: 13, description: 'Displays the total log entries per index, sorted by week day', icon: 'fa-thin fa-chart-pie'},
    'databox' : {'name': 'Log Server Info', defaultWidth: 4, defaultHeight: 4, description: 'Displays selected metrics describing the Log Server instance'}
    //    'trends': {'name': 'Trends', defaultWidth: 4, defaultHeight: 4, description: 'A stock-ticker style representation of how queries are moving over time. For example, if the time is 1:10pm, your time picker was set to Last 10m, and the Time ago parameter was set to 1h, the panel would show how much the query results have changed since 12:00-12:10pm'}
};

export const PanelSettings = ({ panel, updateWidgetData }: { panel: PanelConfig; updateWidgetData: (config: PanelConfigData) => void }) => {

    const renderSettings = () => {
        switch (panel.type) {
        case 'goal':
            return (
                <div>

                </div>
            )
        case 'histogram':
            return (
                <HistogramConfig
                    config={(panel.data) as HistogramPanelData}
                    updateWidgetData={updateWidgetData}
                />
            )
        case 'hits':
            return (
                <div>

                </div>
            )
        case 'map':
            return (
                <MapChartConfig
                    config={(panel.data) as MapPanelData}
                    updateWidgetData={updateWidgetData}
                />
            )
        case 'sparklines':
            return (
                <div>

                </div>
            )
        case 'stats':
            return (
                <StatsConfig
                    config={(panel.data) as StatsPanelData}
                    updateWidgetData={updateWidgetData}
                />
            )
        case 'table':
            return (
                <div>
                    <LogTableConfig
                        config={(panel.data) as LogTablePanelData}
                        updateWidgetData={updateWidgetData}
                    />
                </div>
            )
        case 'term':
            return (
                <TermConfig
                    config={(panel.data) as TermPanelData}
                    updateWidgetData={updateWidgetData}
                />
            )
        case 'text':
            return (
                <TextConfig
                    config={(panel.data) as TextPanelData}
                    updateWidgetData={updateWidgetData}
                />
            )
        case 'twoweeklog':
            return (
                <TwoWeekConfig
                    config={(panel.data) as TwoWeekPanelData}
                    updateWidgetData={updateWidgetData}
                />
            );
        case 'trends':
            return (
                <div>

                </div>
            )
        case 'databox':
            return (
                <DataBoxConfig
                    config={(panel.data) as dataBoxData}
                    updateWidgetData={updateWidgetData}
                />
            )
        default:
            return (<div/>);
        }
    }

    return (
        <div>
            { renderSettings() }
        </div>
      );
}

export const EditPanel = ({ panel, savePanelSettings }: { panel: PanelConfig, savePanelSettings: (newPanel: PanelConfigData) => void }) => {

    return (
        <PanelSettings panel={panel} updateWidgetData={savePanelSettings} />
    );
}

export const AddPanel = ({ rowId, children }: { rowId: string, children: React.ReactNode }) => {
    const { definition, setDefinition } = dashboardStore();
    const { t } = useTranslation();
    const MAX_ROW_WIDTH = 12;
    const [open, setOpen] = useState(false);
    const BlankPanelData: PanelConfig = {
        id: '',
        x: 0,
        y: 0,
        width: 12,
        height: 12,
        type: 'histogram',
        data: {
            title: 'New Panel',
            subQueryChoice: 'all',
            subQueries: [],
            description: ''
        }
    };
    
    const [panel, setPanel] = useState(BlankPanelData);

    const getRowHeight = (row: row): number => {
        var height = 0;
        row.panels.forEach((panel) => {
            if (panel.y + panel.height > height) {
                height = panel.y + panel.height;
            }
        });

        return height;
    }

    const getOpenSpaceInRow = (row: row): boolean[][] => {
        const rowHeight = getRowHeight(row);
        if (rowHeight == 0) {
            return [];
        }

        const openSpace: boolean[][] = [];

        for (var i = 0; i < rowHeight; i++) {
            openSpace[i] = [];
            for (var j = 0; j < MAX_ROW_WIDTH; j++) {
                openSpace[i][j] = true;
            }
        }

        for (var p = 0; p < row.panels.length; p++) {
            const panel = row.panels[p];
            for (var i = panel.y; i < panel.y + panel.height; i++) {
                for (var j = panel.x; j < panel.x + panel.width && j < MAX_ROW_WIDTH; j++) {
                    openSpace[i][j] = false;
                }
            }
        }

        return openSpace;
    }

    const hasEnoughSpace = (openSpaceMap: boolean[][], x: number, y: number, width: number, height: number) => {
        if (x + width > MAX_ROW_WIDTH) {
            return false;
        }

        if (openSpaceMap.length == 0) {
            return true;
        }

        for (var i = x; i < x + width && i < MAX_ROW_WIDTH; i++) {
            for(var j = y; j < y + height && j < openSpaceMap.length; j++) {
                if (!openSpaceMap[j][i]) {
                    return false;
                }
            }
        }
        return true;
    }

    const findOpenSpaceInRow = (row: row, width: number, height: number) => {
        const openSpace = getOpenSpaceInRow(row);
        const rowHeight = getRowHeight(row);
        const maxY = rowHeight - height;

        var bestX = MAX_ROW_WIDTH;
        var bestY = rowHeight;

        // Start in the bottom left and work upwards, all the way to the left
        for (var j = maxY - 1; j >= 0; j--) {
            // If the cell we're looking doesn't have sufficient space, we're done with this iteration
            if (!hasEnoughSpace(openSpace, 0, j, width, height)) {
            break;
            } else {
            // Otherwise keep working upwards.
            bestX = 0;
            bestY = j;
            }
        }

        // After we've exhausted the first column and found space, we're done:
        if (bestX != MAX_ROW_WIDTH) {
            return {x: bestX, y: bestY};
        }

        // Otherwise move over to the right hand side on the bottom and work leftwards then upwards
        for (var i = MAX_ROW_WIDTH - width - 1; i >= 0; i--) {
            // If the cell we're looking doesn't have sufficient space, we're done with this iteration
            if (!hasEnoughSpace(openSpace, i, rowHeight - 1, width, height)) {
            break;
            } else {
            // Otherwise keep working upwards.
            bestX = i;
            bestY = MAX_ROW_WIDTH - width;
            }
        }

        // After we've exhausted the lowest row we can use and found space, we're done:
        if (bestX != MAX_ROW_WIDTH) {
            return {x: bestX, y: bestY};
        }

        return null;
        }

        const getRowById = (rowId: string): number => {
        for(var i = 0; i < definition.rows.length; i++) {
            if (definition.rows[i].id == rowId) {
            return i;
            }
        }

        return -1;
    }

    const addPanel = (config: PanelConfig) => {
        const newPanelId = `panel${Date.now()}`;

        const rowIndex = getRowById(rowId);

        if (rowIndex == -1) {
            return;
        }

        const row = definition.rows[rowIndex];

        const openLocation = findOpenSpaceInRow(row, config.width, config.height);

        var newX = 0;
        var newY = getRowHeight(row);

        if (openLocation) {
            newX = openLocation.x;
            newY = openLocation.y;
        }

        const newPanel = {
            id: newPanelId,
            type: config.type,
            width: config.width,
            height: config.height,
            x: newX,
            y: newY,
            data: config.data,
        };
        
        const newRows = definition.rows.map(row => {
            if (row.id === rowId) {
                return {...row, panels: [...row.panels, newPanel]};
            }
            return row;
        });
        
        setDefinition({...definition, rows: newRows});
        setOpen(false);
    };

    const updateWidgetData = (config: PanelConfigData) => {
        addPanel({...panel, data: config});
    }

    const onPanelTypeChange = (newPanelType: string) => {
        
        // Set up some defaults:
        const newWidgetSettings: PanelConfig = {
            id: '',
            width: panelTypes[newPanelType].defaultWidth,
            height: panelTypes[newPanelType].defaultHeight,
            type: newPanelType,
            x: 0,
            y: 0,
            data: {
                title: 'New Panel',
                subQueryChoice: 'all',
                subQueries: [],
                description: ''
            }
        };
        setPanel(newWidgetSettings);
    }

    return (
        <Sheet open={open} onOpenChange={setOpen}>
            <SheetTrigger asChild>
                {children}
            </SheetTrigger>
            <SheetContent className="pr-0">
                <SheetHeader>
                    <SheetTitle>{t('Add Panel')}</SheetTitle>
                    <SheetDescription>
                        {t('Configure your new panel here.')}
                    </SheetDescription>
                </SheetHeader>
                <ScrollArea className="h-full max-h-[calc(100%-70px)]">
                    <div className="grid grid-cols-3 items-center gap-4 space-y-0 mt-6 pr-6">
                        <label className="text-right text-sm font-medium leading-none">{t('Panel Type')}</label>
                        <Select value={panel.type} onValueChange={onPanelTypeChange}>
                            <SelectTrigger className="border-border col-span-2 w-full">
                                <SelectValue placeholder="Select Panel Type..." defaultValue="histogram"/>
                            </SelectTrigger>
                            <SelectContent>
                            {
                                Object.keys(panelTypes).map((option) => (
                                    <SelectItem key={option} value={option}>
                                        {panelTypes[option].name}
                                    </SelectItem>
                                ))
                            }
                            </SelectContent>
                        </Select>
                    </div>
                    <PanelSettings panel={panel} updateWidgetData={updateWidgetData}/>
                </ScrollArea>
            </SheetContent>
        </Sheet>
    );
}