import { PanelConfig, PanelConfigData } from "@/lib/types";
import { useDashboardContext } from "@/dashboard/contexts/DashboardContext";
import { useTranslation } from "react-i18next";
import { BaseHistogram, HistogramPanelData } from "@/components/charts/BaseHistogram";
import { Hits } from "@/components/charts/Hits";
import { Stats, StatsPanelData } from "@/components/charts/Stats";
import { LogTable, LogTablePanelData } from "@/components/log-table/log-table";
import { Term, TermPanelData } from "@/components/charts/Term";
import { Text, TextPanelData } from "@/components/charts/Text";
import { TotalLogEntries2Weeks, TwoWeekPanelData } from "@/components/charts/other-charts/log_entries_2_weeks";
import { DataBox, dataBoxData } from "@/components/charts/DataBox";
import {
    Card,
    CardDescription,
    CardHeader,
    CardTitle,
} from "@/components/shadcn/card";
import {
    Sheet,
    SheetContent,
    SheetDescription,
    SheetHeader,
    SheetTitle,
    SheetTrigger,
} from "@/components/shadcn/sheet";
import {
    DropdownMenu,
    DropdownMenuContent,
    DropdownMenuItem,
    DropdownMenuTrigger,
} from "@/components/shadcn/dropdown-menu";
import MoreHorizIcon from '@mui/icons-material/MoreHoriz';
import { EditPanel } from '@/dashboard/panels/AddEditPanel';
import { dashboardStore } from "@/dashboard/Dashboard";
import { MapChart, MapPanelData } from "@/components/charts/map/MapChart";
import { ErrorBoundary } from 'react-error-boundary';
import { AxiosError } from "axios";
import ErrorOutlineOutlinedIcon from '@mui/icons-material/ErrorOutlineOutlined';
import {
  Alert,
  AlertDescription,
  AlertTitle,
} from "@/components/shadcn/alert";
import { ScrollArea } from "@/components/shadcn/scroll-area";

export const Panel = ({ panel, rowId }: { panel: PanelConfig; rowId: string; }) => {
    const { t } = useTranslation();
    const { dashboardInfo, dashboardMetaInfo } = useDashboardContext();
    const { filters, subQueries, range, setRange } = dashboardInfo;
    const { isReport } = dashboardMetaInfo;
    const { definition, setDefinition } = dashboardStore();

    const deletePanel = () => {
      const newRows = definition.rows.map(row => {
        if (row.id === rowId) {
          return {...row, panels: row.panels.filter(panelToRemove => panelToRemove.id !== panel.id)};
        }
        return row;
      });
      setDefinition({...definition, rows: newRows});
    }

    const savePanelSettings = (newPanel: PanelConfigData) => {
      const newRows = definition.rows.map((theRow) => {
        if (theRow.id == rowId) {
          theRow.panels = theRow.panels.map((panelToEdit) => {
            if (panelToEdit.id == panel.id) {
              return {...panelToEdit, data: newPanel};
            }
            return panelToEdit;
          })
          return theRow;
        }
        return theRow;
      });
      setDefinition({...definition, rows: newRows});
    }
  
    const renderChart = () => {
      switch (panel.type) {
        // Panels we are replicating from Kibana
        case 'goal':
          return null;
        case 'histogram':
          return (
            <BaseHistogram
              panelConfig={panel.data as HistogramPanelData}
              range={range}
              setRange={setRange}
              queries={subQueries}
              filters={filters}
              panelId={panel.id}
            />
          );
        case 'hits':
          return (
            <Hits
              panelConfig={panel.data as HistogramPanelData}
              range={range}
              panelId={panel.id}
              queries={subQueries}
              filters={filters}
            />
          )
        case 'map':
          return (
            <MapChart
              panelConfig={panel.data as MapPanelData}
              range={range}
              panelId={panel.id}
              queries={subQueries}
              filters={filters}
            />
          );
        case 'sparklines':
          return null;
        case 'stats':
          return (
            <Stats
              panelConfig={panel.data as StatsPanelData}
              range={range}
              queries={subQueries}
              filters={filters}
              panelId={panel.id}
            />
          )
        // Leave log-table here until testing is done.
        case 'log-table':
        case 'table':
          return (
            <LogTable 
              panelConfig={panel.data as LogTablePanelData}
              setPanelConfig={savePanelSettings}
              range={range}
              queries={subQueries}
              filters={filters}
              panelId={panel.id}
            />
          );
        case 'term':
          return (
            <Term
              panelConfig={panel.data as TermPanelData}
              range={range}
              queries={subQueries}
              filters={filters}
              panelId={panel.id}
            />
          );
        case 'databox':
          return (
            <DataBox 
              panelConfig={panel.data as dataBoxData}
              panelId={panel.id}
            /> 
          );
        case 'text':
          return (
            <Text 
              panelConfig={panel.data as TextPanelData}
              panelId={panel.id}
            /> 
            )
        case 'twoweeklog':
          return (
            <TotalLogEntries2Weeks
              panelConfig={panel.data as TwoWeekPanelData}
              panelId={panel.id}
            />
          )
        case 'trends':
            return null;
        // // Potentially new panels 
        // case 'line':
        //   return (
        //     <LineChartCustomLabel />
        //   );
        // case 'pie':
        //   return (
        //     <PieChartLabel />
        //   );
        // case 'numeric':
        //   return (
        //     <NumericChart 
        //       data={panel.data.value} 
        //       title={panel.data.title} 
        //       description={panel.data.description} 
        //       link={panel.data.link} 
        //     />
        //   );
        // case 'bar':
        //   return (
        //     <BarChartMultiple />
        //   );
        // case 'radar':
        //   return (
        //     <RadarChartFilled />
        //   );
        default:
          return (<div/>);
      }
    };

    const PanelFallback = ({ error }: { error: AxiosError }) => {
      return (
        <Card className="w-full h-full overflow-auto bg-secondary">
          <CardHeader className="pb-0 flex w-full justify-end items-center flex-row px-4 py-2">
            {!isReport &&
              <DropdownMenu modal={false}>
                  <DropdownMenuTrigger asChild>
                    <div className="p-1 hover:bg-secondary cursor-pointer rounded transition-colors">
                      <MoreHorizIcon />
                    </div>
                  </DropdownMenuTrigger>
                  <DropdownMenuContent>
                    <DropdownMenuItem onClick={deletePanel}>Delete</DropdownMenuItem>
                  </DropdownMenuContent>
              </DropdownMenu>
            }
          </CardHeader>
          <div className="flex items-center justify-center w-full">
            <Alert variant="destructive" className="w-5/6">
              <ErrorOutlineOutlinedIcon fontSize="small" style={{top: "0.85rem"}}/>
              <AlertTitle>{t(`Uh oh! Your ${panel.data.title} panel has encountered an error.`)}</AlertTitle>
              <AlertDescription>
                <div className="flex flex-col gap-4 py-4 whitespace-nowrap overflow-auto">
                  {error.message}
                  <div>{t("Try refreshing or deleting the the panel.")}</div>
                </div>
              </AlertDescription>
            </Alert>
          </div>
        </Card>
      )
    }
  
    return (
      <ErrorBoundary FallbackComponent={PanelFallback} onError={(error: Error) => console.log(error)}>
        <Card className="border-border/50 w-full h-full overflow-auto bg-secondary">
          <CardHeader className={"draggable pb-0 flex w-full justify-between items-center flex-row px-4 py-2" + (!panel.uneditable? " cursor-move":"") + " transition-colors hover:bg-background/70 border-b border-border/50" }>
            <CardTitle className="font-normal text-xl">{panel.data.title}</CardTitle>
              <Sheet>
                {!dashboardMetaInfo.isReport && !panel.uneditable &&
                  <DropdownMenu modal={false}>
                      <DropdownMenuTrigger asChild>
                        <div className="p-1 hover:bg-secondary cursor-pointer rounded transition-colors" style={{marginTop: 0}}>
                          <MoreHorizIcon />
                        </div>
                      </DropdownMenuTrigger>
                      <DropdownMenuContent>
                        <SheetTrigger asChild>
                          <DropdownMenuItem>
                            {t('Settings')}
                          </DropdownMenuItem>
                        </SheetTrigger>
                        <DropdownMenuItem onClick={deletePanel}>Delete</DropdownMenuItem>
                      </DropdownMenuContent>
                  </DropdownMenu>
                }
                <SheetContent className="focus-visible:outline-none pr-0">
                    <SheetHeader>
                      <SheetTitle>{t('Edit Panel')}</SheetTitle>
                      <SheetDescription>
                          {t('Make changes to your panel here.')}
                      </SheetDescription>
                    </SheetHeader>
                    <ScrollArea className="h-full max-h-[calc(100%-70px)]">
                      <EditPanel panel={panel} savePanelSettings={savePanelSettings}/>
                    </ScrollArea>
                </SheetContent>
            </Sheet>
            {panel.data.displayName && <CardDescription>{panel.data.displayName}</CardDescription>}
          </CardHeader>
          {renderChart() || <div/>}
        </Card>
      </ErrorBoundary>
    );
  };