import { Dialog, DialogContent, DialogFooter, DialogHeader, DialogTitle, DialogTrigger } from '@/components/shadcn/dialog';
import { ReactNode, useState } from 'react';
import { useForm } from 'react-hook-form';
import { useTranslation } from 'react-i18next';
import { useMutation, useQueryClient } from '@tanstack/react-query';
import { z } from 'zod';
import { zodResolver } from '@hookform/resolvers/zod';
import { importQuery } from '@/dashboard/queries/queries';
import { toast } from '@/components/shadcn/use-toast';
import { Form, FormControl, FormField, FormItem, FormMessage } from '@/components/shadcn/form';
import { Input } from '@/components/shadcn/input';
import { Button } from '@/components/shadcn/button';
import { AxiosError } from 'axios';
import { Checkbox } from '@/components/shadcn/checkbox';

export const ImportQueryDialog = ({ children }: { children: ReactNode }) => {
    const { t } = useTranslation();
    const queryClient = useQueryClient();
    const [open, setOpen] = useState(false);

    const importFormSchema = z.object({
        import: z.instanceof(File).refine((file) => {
            return file.type == 'application/json';
        }, t('File Is Not a Valid Query File!')),
        global: z.boolean()
    });

    const importForm = useForm<z.infer<typeof importFormSchema>>({
        resolver: zodResolver(importFormSchema),
        mode: 'onChange',
        defaultValues: {
            import: undefined,
            global: false,
        },
    });

    const importQueryMutation = useMutation({
        mutationFn: importQuery,
        onSuccess: () => {
            queryClient.invalidateQueries({ queryKey: ['query'] });
            setOpen(false);
            toast({
                title: t('Query Imported.'),
                variant: 'success',
                description: t('Access your new query below.'),
            });
        },
        onError: (error: AxiosError<{ message: string }>) => {
            toast({
                title: t('Failed to import query.'),
                variant: 'error',
                description: error.response?.data.message,
            });
        },
    });

    const onImportSubmit = async (data: z.infer<typeof importFormSchema>) => {
        const text = await data.import.text();
        importQueryMutation.mutate({ import: text, global: data.global});
    };

    return (
        <Dialog open={open} onOpenChange={setOpen}>
            <DialogTrigger>{children}</DialogTrigger>
            <DialogContent>
                <DialogHeader>
                    <DialogTitle>{t('Import a Query')}</DialogTitle>
                </DialogHeader>
                <Form {...importForm}>
                    <form className="flex flex-col gap-2" onSubmit={importForm.handleSubmit(onImportSubmit)}>
                        <div className="flex items-center gap-4">
                            <div className="grow">
                                <FormField
                                    control={importForm.control}
                                    name="import"
                                    render={({ field: { value, onChange, ...fieldProps } }) => (
                                        <FormItem>
                                            <FormControl>
                                                {/* Why this looks strange https://github.com/shadcn-ui/ui/discussions/2137 */}
                                                <Input {...fieldProps} type="file" onChange={(event) => onChange(event.target.files && event.target.files[0])} className="file:text-foreground file:font-bold border-border"/>
                                            </FormControl>
                                            <FormMessage />
                                        </FormItem>
                                    )}
                                />
                            </div>
                            <FormField
                            control={importForm.control}
                            name="global"
                            render={({ field }) => (
                                <FormItem>
                                    <FormControl>
                                        <div className="flex items-center space-x-2">
                                        <Checkbox id="importglobal" 
                                            checked={field.value}
                                            onCheckedChange={field.onChange}
                                            defaultChecked={false}
                                        />
                                        <label
                                            htmlFor="importglobal"
                                            className="text-sm font-medium leading-none peer-disabled:cursor-not-allowed peer-disabled:opacity-70"
                                        >
                                            Global
                                        </label>
                                        </div>
                                    </FormControl>
                                </FormItem>
                            )}
                        />
                        </div>
                        <DialogFooter>
                            <Button type="submit">{t('Import')}</Button>
                        </DialogFooter>
                    </form>
                </Form>
            </DialogContent>
        </Dialog>
    );
};
