import { Dialog, DialogClose, DialogContent, DialogDescription, DialogFooter, DialogHeader, DialogTitle, DialogTrigger } from '@/components/shadcn/dialog';
import { Input } from '@/components/shadcn/input';
import { Button } from '@/components/shadcn/button';
import { ReactNode, useState } from 'react';
import { useTranslation } from 'react-i18next';
import { toast } from '@/components/shadcn/use-toast';
import { query } from '@/dashboard/queries/types';
import { AxiosError } from 'axios';
import { addQuery, getQueries } from '@/dashboard/queries/queries';
import { useMutation, useQuery, useQueryClient } from '@tanstack/react-query';
import { DataTable } from './data-table';
import { columns } from './columns';
import { useForm, SubmitHandler } from 'react-hook-form';
import { Form, FormControl, FormField, FormItem } from '@/components/shadcn/form';
import { Checkbox } from '@/components/shadcn/checkbox';
import { subQuery } from '@/dashboard/queries/types';
import { filter } from '@/dashboard/filters/types';
import { useDashboardContext } from '@/dashboard/contexts/DashboardContext';
import { useDashboardQueryContext } from '@/dashboard/contexts/DashboardQueryContext';

/**
 * The query management dialog.
 */
export const ManageQueriesDialog = ({ children }: { children: ReactNode }) => {
    const [open, setOpen] = useState(false);
    const { t } = useTranslation();
    const queryClient = useQueryClient();
    const dashboardQuery = useDashboardQueryContext();
    const { filters, setFilters, subQueries, setSubQueries } = useDashboardContext().dashboardInfo;

    const runQuery = (subQueries: subQuery[], filters: filter[]) => {
        setSubQueries(subQueries);
        setFilters(filters);
    }

    const queriesSummaryQuery = useQuery<query[], AxiosError>({
        queryKey: ['query'],
        queryFn: async () => {
            const response = await getQueries();
            return response.data;
        },
    });

    const addQueryMutation = useMutation({
        mutationFn: addQuery,
        onSuccess: () => {
            queryClient.invalidateQueries({ queryKey: ['query'] });
            toast({
                title: t('Query added.'),
                variant: 'success',
            });
        },
        onError: (error: AxiosError<{ message: string }>) => {
            toast({
                title: t('Failed to add query.'),
                variant: 'error',
                description: error.response?.data.message || t('An unknown error occured.'),
            });
        },
    });

    const form = useForm<query>({
        defaultValues: {
            name: '',
            show_everyone: false
        },
    });

const onSubmit: SubmitHandler<query> = (data) => {
        addQueryMutation.mutate({ name: data.name, raw: JSON.stringify(dashboardQuery), subQueries: subQueries, filters: filters, global: data.show_everyone });
    };

    if (!queriesSummaryQuery.isSuccess) {
        return;
    }

    return (
        <Dialog open={open} onOpenChange={setOpen}>
            <DialogTrigger asChild>
                {children}
            </DialogTrigger>
            <DialogContent className="w-fit min-w-fit h-[600px] min-h-[600px]">
                <DialogHeader>
                    <DialogTitle>{t('Manage Dashboard Queries')}</DialogTitle>
                    <DialogDescription>{t('Use this to save your current dashboard query or to manage your existing dashboard queries.')}</DialogDescription>
                </DialogHeader>
                <Form {...form}>
                    <form onSubmit={form.handleSubmit(onSubmit)} className="flex items-center gap-4 py-2">
                        <div className="grow">
                            <FormField
                                control={form.control}
                                name="name"
                                render={({ field }) => (
                                    <FormItem className="">
                                        <FormControl>
                                            <Input {...field} className="border-border" placeholder={t("Save current dashboard query as") + '...'}/>
                                        </FormControl>
                                    </FormItem>
                                )}
                            />
                        </div>
                         <FormField
                            control={form.control}
                            name="show_everyone"
                            render={({ field }) => (
                                <FormItem>
                                    <FormControl>
                                        <div className="flex items-center space-x-2">
                                        <Checkbox id="global" 
                                            checked={field.value}
                                            onCheckedChange={field.onChange}
                                            defaultChecked={false}
                                        />
                                        <label
                                            htmlFor="global"
                                            className="text-sm font-medium leading-none peer-disabled:cursor-not-allowed peer-disabled:opacity-70"
                                        >
                                            Global
                                        </label>
                                        </div>
                                    </FormControl>
                                </FormItem>
                            )}
                        />
                        <Button type="submit" className="justify-end">
                            {t('Create')}
                        </Button>
                    </form>
                </Form>
                <DataTable columns={columns} data={queriesSummaryQuery.data} runQuery={runQuery}/>
                <DialogFooter>
                    <DialogClose asChild>
                        <Button type="button" variant="outline">
                            Close
                        </Button>
                    </DialogClose>
                </DialogFooter>
            </DialogContent>
        </Dialog>
    );
};
