import { Input } from '@/components/shadcn/input';
import {
  Tooltip,
  TooltipContent,
  TooltipTrigger,
} from "@/components/shadcn/tooltip";
import { Badge } from "@/components/shadcn/badge";
import { subQuery } from '@/dashboard/queries/types';
import { useEffect, useState } from 'react';
import { getNaturalLanguageQuery } from '@/dashboard/queries/queries';
import { useQuery } from '@tanstack/react-query';
import { AxiosError } from 'axios';
import { useDashboardContext } from '@/dashboard/contexts/DashboardContext';
import { cn } from '@/lib/utils';
import { t } from 'i18next';
import ScienceOutlinedIcon from '@mui/icons-material/ScienceOutlined';
import { colors } from '@/dashboard/queries/store';

const NLQSearchbar = ({ subQueries, setSubQueries }: { subQueries: subQuery[], setSubQueries: (subQueries: subQuery[]) => void }) => {

  const [inputValue, setInputValue] = useState('');
  const [placeholderValue, setPlaceholderValue] = useState('');
  const { fields } = useDashboardContext().dashboardInfo;
  const [loading, setLoading] = useState(false);
  const [placeholderIndex, setPlaceholderIndex] = useState(0);
  const [characterIndex, setCharacterIndex] = useState(0);
  const [backspacing, setBackspacing] = useState(false);
  const [intervalRate, setIntervalRate] = useState(25);
  const placeholders = [
    "Show me logs from yesterday",
    "Login Attempts, use Windows Event IDs",
    "Apache 404 errors",
    "I want to see system logs from 192.168.0.12",
    "An advanced, robust, complicated query for sshd errors.",
    "session starts (pam)"
  ];

  useEffect(() => {
    const interval = setInterval(() => {
      if (!backspacing) {
        if (characterIndex < placeholders[placeholderIndex].length) {
          setPlaceholderValue(placeholders[placeholderIndex].substring(0, characterIndex + 1));
          setCharacterIndex(characterIndex + 1);
        } else {
          setIntervalRate(1000);
          setBackspacing(true);
        }
      } else {
        if (characterIndex > 0) {
          setIntervalRate(10);
          setCharacterIndex(characterIndex - 1);
          setPlaceholderValue(placeholders[placeholderIndex].substring(0, characterIndex - 1));
        } else {
          setIntervalRate(25);
          setBackspacing(false);
          setPlaceholderIndex((placeholderIndex + 1) % 5);
        }
      }
    }, intervalRate);
    return () => clearInterval(interval);
  }, [placeholderValue, backspacing]);

  const naturalLanguageQuerySummary = useQuery<string, AxiosError>({
    queryKey: ['nlq'],
    queryFn: async () => {
        const response = await getNaturalLanguageQuery(inputValue, JSON.stringify(fields.map(field => field.name)));
        const newSubQuery: subQuery = {id: `id${Date.now()}-${Math.floor(Math.random() * 1000)}`, query: response.data.query, color: colors[subQueries.length % 6]};
        if (subQueries.length === 1 && subQueries[0].query === '*') {
          setSubQueries([newSubQuery]);
        } else {
          setSubQueries([...subQueries, newSubQuery]);
        }
        return response.data;
    },
    enabled: false
  });

  useEffect(() => {
    !naturalLanguageQuerySummary.isLoading && setLoading(false);
  }, [naturalLanguageQuerySummary.isLoading, subQueries]);

  const handleInputEnter = (key: string) => {
    if (key === 'Enter') {
      setLoading(true);
      naturalLanguageQuerySummary.refetch();
    }
  };

  const handleInputChange = (value: string) => {
    setInputValue(value);
  };

  return (
      <div className="flex items-center pl-[0.6rem] pr-4 gap-2 w-full h-9">
        {/* <AutoAwesomeIcon fontSize="small" className={cn("text-primary m-2", loading && "animate-spin")} /> */}
        <Tooltip>
          <TooltipTrigger asChild>
            <svg xmlns="http://www.w3.org/2000/svg" height="24" viewBox="0 0 24 24" width="24" fill="hsl(var(--primary))" className="mx-[0.4rem]">
                <path d="M19 9l1.25-2.75L23 5l-2.75-1.25L19 1l-1.25 2.75L15 5l2.75 1.25L19 9z" className={cn(loading && "animate-pulse")} />
                <path d="M11.5 9.5L9 4 6.5 9.5 1 12l5.5 2.5L9 20l2.5-5.5L17 12l-5.5-2.5z" className={cn(loading && "animate-pulse delay-500")} />
                <path d="M19 15l-1.25 2.75L15 19l2.75 1.25L19 23l1.25-2.75L23 19l-2.75-1.25L19 15z" className={cn(loading && "animate-pulse delay-1000")} />
            </svg>
          </TooltipTrigger>
          <TooltipContent>
            <div className="flex gap-2 flex-nowrap items-center">
              <p>{t("Natural Language Queries")}</p>
              <Badge className="text-xs gap-1">
                <ScienceOutlinedIcon className="max-h-4 max-w-4" />
                {t("Experimental")}
              </Badge>
            </div>
          </TooltipContent>
        </Tooltip>
        <Input
          type="text"
          placeholder={placeholderValue}
          className="bg-background border border-border rounded-md flex-grow"
          value={inputValue}
          onChange={(e) => handleInputChange(e.target.value)}
          onKeyDown={(e) => handleInputEnter(e.key)}
        />
      </div>
  )
}

export const NLQToolbar = () => {
    const {subQueries, setSubQueries} = useDashboardContext().dashboardInfo;

    return (
      <NLQSearchbar subQueries={subQueries} setSubQueries={setSubQueries}/>
    )
}