import {
    AlertDialog,
    AlertDialogContent,
    AlertDialogDescription,
    AlertDialogFooter,
    AlertDialogHeader,
    AlertDialogTrigger,
} from '@/components/shadcn/alert-dialog';
import { AlertDialogAction, AlertDialogCancel, AlertDialogTitle } from '@radix-ui/react-alert-dialog';
import { Button } from '@/components/shadcn/button';
import { ReactNode } from 'react';
import { useTranslation } from 'react-i18next';
import { toast } from '@/components/shadcn/use-toast';
import { AxiosError } from 'axios';
import { useMutation, useQueryClient } from '@tanstack/react-query';
import { overwriteQuery } from '@/dashboard/queries/queries';
import { useDashboardContext } from '@/dashboard/contexts/DashboardContext';

/**
 * The easybutton dialog. Passing in an easybutton makes it an edit easybutton dialog. Otherwise it's an add easybutton dialog.
 */
export const OverwriteQueryDialog = ({ id, name, children }: { id: string; name: string; children: ReactNode }) => {
    const { t } = useTranslation();
    const queryClient = useQueryClient();
    const { filters, subQueries } = useDashboardContext().dashboardInfo;

    const overwriteQueryMutation = useMutation({
        mutationFn: overwriteQuery,
        onSuccess: () => {
            queryClient.invalidateQueries({ queryKey: ['query'] });
            toast({
                title: t('Query overwritten.'),
                variant: 'success',
            });
        },
        onError: (error: AxiosError<{ message: string }>) => {
            toast({
                title: t('Failed to overwrite query.'),
                variant: 'error',
                description: error.response?.data.message || t('An unknown error occured.'),
            });
        },
    });

    return (
        <AlertDialog>
        <AlertDialogTrigger className="w-full">{children}</AlertDialogTrigger>
        <AlertDialogContent>
            <AlertDialogHeader>
                <AlertDialogTitle>{t('Confirm Query Overwrite')}</AlertDialogTitle>
                <AlertDialogDescription>{t(`Are you sure you want to overwrite ${name} with the current dashboard query?`)}</AlertDialogDescription>
            </AlertDialogHeader>
            <AlertDialogFooter>
                <AlertDialogCancel>
                    <Button variant="outline">{t('Cancel')}</Button>
                </AlertDialogCancel>
                <AlertDialogAction>
                    <Button onClick={() => overwriteQueryMutation.mutate({ id: id, subQueries: subQueries, filters: filters })}>
                        {t('Overwrite')}
                    </Button>
                </AlertDialogAction>
            </AlertDialogFooter>
        </AlertDialogContent>
    </AlertDialog>
    );
};
