import { Button } from '@/components/shadcn/button';
import { Input } from '@/components/shadcn/input';
import { subQuery } from '@/dashboard/queries/types';
import AddIcon from '@mui/icons-material/Add';
import DeleteOutlineOutlinedIcon from '@mui/icons-material/DeleteOutlineOutlined';
import { useDashboardContext } from '@/dashboard/contexts/DashboardContext';
import React, { useState, useRef } from 'react';
import { useTranslation } from 'react-i18next';
import { cn } from '@/lib/utils';
import { HexColorPicker } from "react-colorful";
import { Tooltip, TooltipContent, TooltipTrigger } from '@/components/shadcn/tooltip';
import {
  Popover,
  PopoverContent,
  PopoverTrigger,
} from "@/components/shadcn/popover";
import { colors } from '@/dashboard/queries/store';
import { Token, SimpleSearch, retokenize } from '../../SimpleSearch';
import { type TokenInputRef }  from 'react-customize-token-input';
import { Search } from "lucide-react";


interface QToolbarProps {
  isSimple?: boolean
  simpleTokens?: string | null
}


const QueryBox = ({subQuery, subQueries, setSubQueries}: {subQuery: subQuery, subQueries: subQuery[], setSubQueries: (subQueries: subQuery[]) => void}) => {
  
  const [inputValue, setInputValue] = useState(subQuery.query);
  const [pressedEnter, setPressedEnter] = useState(false);
  const [pickedColor, setPickedColor] = useState(subQuery.color);
  const [open, setOpen] = useState(false);

  const handleInputEnter = (key: string) => {
    if (key === 'Enter') {
      setPressedEnter(true);
      setTimeout(() => {
        setPressedEnter(false);
      }, 150);
      setSubQueries(subQueries.map(theSubQuery => 
        theSubQuery.id === subQuery.id ? { ...theSubQuery, query: inputValue } : theSubQuery
      ));
    }
  };

  const handleInputChange = (value: string) => {
    setInputValue(value);
  };

  const handleColorChange = (newColor: string) => {
    setPickedColor(newColor);
  }

  const handleColorSubmit = () => {
    setSubQueries(subQueries.map(theSubQuery => 
      theSubQuery.id === subQuery.id ? { ...theSubQuery, color: pickedColor} : theSubQuery
    ));
    setOpen(false);
  }
  
  return (
    <div className="min-w-40 grow flex items-center bg-secondary rounded-md border border-border">
      <Popover open={open} onOpenChange={setOpen}>
        <PopoverTrigger asChild>
          <div
            className="h-3 w-3 rounded-full m-3 shrink-0 transition-all duration-300 hover:cursor-pointer hover:ring-2 hover:ring-foreground"
            style={{backgroundColor: `${pickedColor || '#1cd98c'}`}}
          ></div>
        </PopoverTrigger>
        <PopoverContent className="p-0 w-auto">
          <div className="flex w-[200px] flex-col">
            <HexColorPicker color={pickedColor} onChange={(color) => handleColorChange(color)}/>
            <div className="flex items-center m-2 gap-2">
              <Input className="border border-border shrink" value={pickedColor} onChange={(e) => handleColorChange(e.target.value)} />
              <Button onClick={handleColorSubmit}>
                Submit
              </Button>
            </div>
          </div>
        </PopoverContent>
      </Popover>
      <Input
        type="text"
        placeholder="Enter search query"
        className={cn("grow h-9 border-l border-r border-b-0 border-t-0 border-border rounded-none transition-[background] bg-gradient-to-r to-background via-background from-background via-50% bg-[length:200%_200%] bg-pos-100", pressedEnter && "via-secondary bg-pos-0", subQueries.length === 1 && "rounded-r-md border-r-0")}
        value={inputValue}
        onChange={(e) => handleInputChange(e.target.value)}
        onKeyDown={(e) => handleInputEnter(e.key)}
      />
      {subQueries.length !== 1 &&
        <Button
          key={subQuery.id}
          size="icon"
          variant="ghost"
          className="shrink-0 rounded-l-none"
          onClick={() => 
            setSubQueries(
              subQueries.filter(q =>
                q.id !== subQuery.id
              )
            )
          }
        >
          <DeleteOutlineOutlinedIcon fontSize="small" />
        </Button>
      }
    </div>
  )
}


const SimpleQuery = ({subQuery, subQueries, setSubQueries, searchTokens}: {subQuery: subQuery, subQueries: subQuery[], setSubQueries: (subQueries: subQuery[]) => void, searchTokens: string | null}) => {
  const tokenInputRef = useRef<TokenInputRef>(null);
  const [tokenValues, setTokenValues] = useState<Token[]>(retokenize(searchTokens));
  const [preventSubmit, setPreventSubmit] = useState<boolean>(false);

  // format tokens and pass to search
  const handleQuerySubmit = (event:React.FormEvent) => {
    event.preventDefault()
    if (preventSubmit) { 
      setPreventSubmit(false); 
      return; 
    }
    // format tokens into query
    let detoken;
    detoken = tokenValues.map((t) => {
      if (t.tType == 'query'){
        return t.query;
      }
      return '"' + t.value.replace(/"/g, "'") + '"';
    });
    detoken = detoken.join(" AND ");
    if (detoken == "") {
      detoken = "*";
    }
    // set new queries to search
    setSubQueries(subQueries.map(theSubQuery => 
      theSubQuery.id === subQuery.id ? { ...theSubQuery, query: detoken } : theSubQuery
    ));
  }

  return (
    <form className="w-full" onSubmit={handleQuerySubmit}>
      <div className="flex items-center w-full relative"
          onClick={()=>{tokenInputRef.current?.focus()}}>
        <SimpleSearch tokenInputRef={tokenInputRef} 
                      tokenValues={tokenValues} 
                      setTokenValues={setTokenValues}
                      setPreventSubmit={setPreventSubmit}
                      initString={subQuery.query} />
        <button id="simple-search" type="submit" className="absolute inset-y-0 right-3">
            <Search className="material-symbols-outlined tw-p-3 text-secondary-foreground/70" />
        </button>
      </div>
    </form>
  )
}


export const QueryToolbar: React.FC<QToolbarProps> = (props:QToolbarProps) => {
    const { subQueries, setSubQueries } = useDashboardContext().dashboardInfo;
    const { t } = useTranslation();

    return (
      <div className="py-2 pl-[0.6rem] pr-4 mt-1 w-full flex items-start gap-2">
        {!props.isSimple && <Tooltip>
          <TooltipTrigger asChild>
            <Button
              size="icon"
              className="min-w-9"
              variant="ghost"
              onClick={() => 
              setSubQueries([
                ...subQueries,
                { id: `id${Date.now()}-${Math.floor(Math.random() * 1000)}`, query: '*', color: colors[subQueries.length % 6]}]
              )
          }
          >
          <AddIcon fontSize="small" />
          </Button>
          </TooltipTrigger>
            <TooltipContent>
              {t("Subquery")}
              </TooltipContent>
          </Tooltip> 
        }
        <div className="flex gap-2 items-center flex-wrap w-full max-h-80 overflow-y-auto pl-4 pr-3">
          {subQueries.map((subQuery) => (
            props.isSimple? 
            <SimpleQuery key={subQuery.id} subQuery={subQuery} subQueries={subQueries} setSubQueries={setSubQueries} searchTokens={props.simpleTokens? props.simpleTokens : null} /> :
            <QueryBox key={subQuery.id} subQuery={subQuery} subQueries={subQueries} setSubQueries={setSubQueries} />
          ))}
        </div>
        {!props.isSimple && 
          <Button
            variant="secondary"
            className="border border-border"
            onClick={() =>
              setSubQueries([
                { id: `id${Date.now()}-${Math.floor(Math.random() * 1000)}`, query: '*', color: '#1cd98c'}
              ])
            }
          >
            {t("Clear")}
          </Button>
        }
      </div>
    )
  }