"use client"

import * as React from "react"
import {
    ColumnDef,
    ColumnFiltersState,
    RowData,
    SortingState,
    flexRender,
    getCoreRowModel,
    getFilteredRowModel,
    getPaginationRowModel,
    getSortedRowModel,
    useReactTable,
} from "@tanstack/react-table";

import {
  Table,
  TableBody,
  TableCell,
  TableHead,
  TableHeader,
  TableRow,
} from "@/components/shadcn/table";

import { Input } from "@/components/shadcn/input";
import { ImportQueryDialog } from "@/dashboard/queries/ImportQueryDialog";
import { Button } from "@/components/shadcn/button";
import { useTranslation } from 'react-i18next';
import { subQuery } from '@/dashboard/queries/types';
import { filter } from '@/dashboard/filters/types';

interface DataTableProps<TData, TValue> {
  columns: ColumnDef<TData, TValue>[]
  data: TData[],
  runQuery: (subQueries: subQuery[], filters: filter[]) => void
}

declare module '@tanstack/table-core' {
  interface TableMeta<TData extends RowData> {
    runQuery: (subQueries: subQuery[], filters: filter[]) => void
  }
}

export function DataTable<TData, TValue>({
    columns,
    data,
    runQuery,
  }: DataTableProps<TData, TValue>) {
    const [sorting, setSorting] = React.useState<SortingState>([])
    const [columnFilters, setColumnFilters] = React.useState<ColumnFiltersState>(
      []
    )

    const { t } = useTranslation();

    const table = useReactTable({
      data,
      columns,
      onSortingChange: setSorting,
      getCoreRowModel: getCoreRowModel(),
      getPaginationRowModel: getPaginationRowModel(),
      getSortedRowModel: getSortedRowModel(),
      onColumnFiltersChange: setColumnFilters,
      getFilteredRowModel: getFilteredRowModel(),
      state: {
        sorting,
        columnFilters,
      },
      meta: {
        runQuery: runQuery
      },
      initialState: {
        pagination: {
          pageSize: data.length
        }
      }
    })

  return (
    <div>
        <div className="flex items-center py-4 gap-4">
            <Input
            placeholder="Filter dashboard queries..."
            value={(table.getColumn("name")?.getFilterValue() as string) ?? ""}
            onChange={(event) =>
                table.getColumn("name")?.setFilterValue(event.target.value)
            }
            className="grow border border-border"
            />
            <ImportQueryDialog>
              <Button>{t('Import')}</Button>
            </ImportQueryDialog>
        </div>
        <div className="rounded-md border overflow-auto max-h-[310px]">
          <Table>
              <TableHeader>
              {table.getHeaderGroups().map((headerGroup) => (
                  <TableRow key={headerGroup.id}>
                  {headerGroup.headers.map((header) => {
                      return (
                      <TableHead key={header.id}>
                          {header.isPlaceholder
                          ? null
                          : flexRender(
                              header.column.columnDef.header,
                              header.getContext()
                              )}
                      </TableHead>
                      )
                  })}
                  </TableRow>
              ))}
              </TableHeader>
              <TableBody>
              {table.getRowModel().rows?.length ? (
                  table.getRowModel().rows.map((row) => (
                  <TableRow
                      key={row.id}
                      data-state={row.getIsSelected() && "selected"}
                  >
                      {row.getVisibleCells().map((cell) => (
                      <TableCell key={cell.id}>
                          {flexRender(cell.column.columnDef.cell, cell.getContext())}
                      </TableCell>
                      ))}
                  </TableRow>
                  ))
              ) : (
                  <TableRow>
                  <TableCell colSpan={columns.length} className="h-24 text-center">
                      No results.
                  </TableCell>
                  </TableRow>
              )}
              </TableBody>
          </Table>
        </div>
    </div>
  )
}
