import { Textarea } from "@/components/shadcn/textarea";
import { Button } from "@/components/shadcn/button";
import {
  Dialog,
  DialogContent,
  DialogDescription,
  DialogFooter,
  DialogHeader,
  DialogTitle,
  DialogTrigger,
} from "@/components/shadcn/dialog";
import {
  Select,
  SelectContent,
  SelectGroup,
  SelectItem,
  SelectTrigger,
  SelectValue,
} from "@/components/shadcn/select";
import { RadioGroup, RadioGroupItem } from "@/components/shadcn/radio-group";
import { Input } from "@/components/shadcn/input";
import { useForm, SubmitHandler } from 'react-hook-form';
import { Form, FormLabel, FormControl, FormField, FormItem } from '@/components/shadcn/form';
import { useTranslation } from 'react-i18next';
import { ReactNode, useEffect, useState } from 'react';
import { scheduleReport } from '@/dashboard/reports/queries';
import { useMutation, useQuery, useQueryClient } from "@tanstack/react-query";
import axios, { AxiosError } from "axios";
import { toast } from "@/components/shadcn/use-toast";
import { email } from '@/dashboard/reports/types';
import { z } from "zod";
import { zodResolver } from "@hookform/resolvers/zod";
import { Loader2 } from "lucide-react";

export const EmailReportDialog = ({ dashboardName, dashboardId, children }: { dashboardName: string, dashboardId: string, children: ReactNode }) => {
  const [open, setOpen] = useState(false);
  const { t } = useTranslation();
  const queryClient = useQueryClient();
  const [emailData, setEmailData] = useState<email>();
  const [filename, setFilename] = useState('');
  const [isGenerating, setIsGenerating] = useState(false);

  const attachmentQuery = useQuery({
    queryKey: ['emailReport'],
    queryFn: async () => {
      const response = await axios.get(`${window.location.origin}/nagioslogserver/reports/check_attachment/${filename}`);
      if (response.data.status === 'success') {
        emailData && emailReportMutation.mutate({email: emailData, filename: filename});
        setIsGenerating(false);
      } else {
        console.log(t('Failed to find report.'));
      }
      return response.data;
    },
    enabled: false,
    retry: true,
    retryDelay: 2000
  });

  function exportReport(type: string) {
    const name = `report-${Math.floor(Date.now() / 1000)}.${type}`;
    const url = `${window.location.origin}/nagioslogserver/reports/view/${dashboardId}?export=${type}&filename=${name}`;
    setFilename(name);
    setIsGenerating(true);

    fetch(url)
      .then(() => {
        attachmentQuery.refetch();
      })
      .catch(error => {
        console.log(t('Error during report export:'), error);
      })
  }

  const Overlay = () => {
    return (
      <div className="fixed inset-0 z-[51] bg-black/80">
        <div className="relative top-[45%] left-1/2 z-[51] flex items-center gap-2 text-background dark:text-foreground">
          <Loader2 className="animate-spin" />
          {t("Generating Attachment")}
        </div>
      </div>
    )
  }

  const formSchema = z.object({
    dashboard_id: z.string().optional(),
    name: z.string().min(1, t('Name is required')),
    frequency: z.object({
      which: z.string().min(1, t('Frequency is required')),
      minute: z.string().optional(),
      hour: z.string().optional(),
      weekday: z.string().optional(),
      monthday: z.string().optional(),
      ampm: z.string().optional()
    }),
    recipients: z.string().min(1, t('Recipients are required')),
    subject: z.string().min(1, t('Subject is required')),
    attachment: z.string().optional(),
    message: z.string().optional()
  })

  const emailReportMutation = useMutation({
    mutationFn: scheduleReport,
    onSuccess: () => {
        queryClient.invalidateQueries({ queryKey: ['scheduled_reports'] });
        setOpen(false);
        toast({
          title: form.watch('frequency.which') === 'once' ? t('Email sent!') : t('Report scheduled!'),
          variant: 'success',
          description: form.watch('frequency.which') !== 'once' && t('Manage your scheduled reports in the Scheduled Reports page.'),
        });
        form.reset();
    },
    onError: (error: AxiosError<{ message: string }>) => {
        toast({
            title: t('Failed to schedule report.'),
            variant: 'error',
            description: error.response?.data.message || t('An unknown error occured.'),
        });
    },
  });
  
  const hours: string[] = [];
  const minutes: string[] = [];
  const months: string[] = [];

  for (let i = 1; i <= 12; i++) {
    hours.push(i.toString().padStart(2, '0'));
  }

  for (let i = 0; i <= 59; i++) {
    minutes.push(i.toString().padStart(2, '0'));
  }

  for (let i = 1; i <= 31; i++) {
    months.push(i.toString());
  }

  const form = useForm<email>({
    resolver: zodResolver(formSchema),
    defaultValues: {
      dashboard_id: dashboardId,
      name: '',
      frequency: {
        which: "once"
      },
      attachment: 'none',
      recipients: '',
      subject: dashboardName,
      message: ''
    },
  });

  const onSubmit: SubmitHandler<email> = (data) => {
    if (data.frequency.which === 'once' && (data.attachment !== 'none')) {
      setEmailData(data);
      data.attachment === 'pdf' ? exportReport("pdf") : exportReport("jpg");
    } else {
      emailReportMutation.mutate({email: data});
    }
  };

  useEffect(() => {
    if (emailReportMutation.isSuccess) {
      form.reset();
    }
  }, [emailReportMutation.status, form.reset]);

  return (
    <Dialog open={open} onOpenChange={setOpen}>
      {isGenerating &&
        <Overlay />
      }
      <DialogTrigger asChild>
          {children}
      </DialogTrigger>
        <DialogContent className="sm:max-w-[425px]">
          <DialogHeader>
              <DialogTitle>{t('Email Report')}</DialogTitle>
              <DialogDescription>{t('Use this to schedule or immediately email your report!')}</DialogDescription>
          </DialogHeader>
          <Form {...form}>
              <form onSubmit={form.handleSubmit(onSubmit)} className="grid gap-4 py-2">
                  <FormField
                      control={form.control}
                      name="name"
                      render={({ field }) => (
                          <FormItem className="grid grid-cols-4 items-center gap-4 space-y-0">
                              <FormLabel className="text-right">{t('Name')}</FormLabel>
                              <FormControl>
                                  <Input {...field} className="col-span-3 border-border" />
                              </FormControl>
                          </FormItem>
                      )}
                  />
                  <FormField
                      control={form.control}
                      name="frequency.which"
                      render={({ field }) => (
                          <FormItem className="grid grid-cols-4 items-center gap-4 space-y-0">
                              <FormLabel className="text-right">{t('Frequency')}</FormLabel>
                              <FormControl>
                                  <Select onValueChange={field.onChange} defaultValue={field.value}>
                                      <SelectTrigger className="col-span-3 border-border">
                                          <SelectValue placeholder={t('Select a frequency')} defaultValue="once"/>
                                      </SelectTrigger>
                                      <SelectContent>
                                          <SelectGroup>
                                            <SelectItem value="once">{t("Once")}</SelectItem>
                                            <SelectItem value="daily">{t("Daily")}</SelectItem>
                                            <SelectItem value="weekly">{t("Weekly")}</SelectItem>
                                            <SelectItem value="monthly">{t("Monthly")}</SelectItem>
                                          </SelectGroup>
                                      </SelectContent>
                                  </Select>
                              </FormControl>
                          </FormItem>
                      )}
                  />
                  {(form.watch("frequency.which") === 'daily' || form.watch("frequency.which") === 'weekly' || form.watch("frequency.which") === 'monthly') &&
                    <div className="grid grid-cols-4 items-center gap-4 space-y-0">
                      <FormLabel className="text-right">{t('Time')}</FormLabel>
                      <FormField
                        control={form.control}
                        name="frequency.hour"
                        render={({ field }) => (
                            <FormItem>
                                <FormControl>
                                    <Select onValueChange={field.onChange} defaultValue={"01"}>
                                        <SelectTrigger className="border-border">
                                            <SelectValue defaultValue="01"/>
                                        </SelectTrigger>
                                        <SelectContent>
                                            <SelectGroup>
                                              {hours.map((item) => (
                                                <SelectItem value={item}>{item}</SelectItem>
                                              ))}
                                            </SelectGroup>
                                        </SelectContent>
                                    </Select>
                                </FormControl>
                            </FormItem>
                        )}
                      />
                      <FormField
                        control={form.control}
                        name="frequency.minute"
                        render={({ field }) => (
                            <FormItem>
                                <FormControl>
                                    <Select onValueChange={field.onChange} defaultValue={"00"}>
                                        <SelectTrigger className="border-border">
                                            <SelectValue defaultValue="00"/>
                                        </SelectTrigger>
                                        <SelectContent>
                                            <SelectGroup>
                                              {minutes.map((item) => (
                                                <SelectItem value={item}>{item}</SelectItem>
                                              ))}
                                            </SelectGroup>
                                        </SelectContent>
                                    </Select>
                                </FormControl> 
                            </FormItem>
                        )}
                      />
                      <FormField
                        control={form.control}
                        name="frequency.ampm"
                        render={({ field }) => (
                            <FormItem>
                                <FormControl>
                                    <Select onValueChange={field.onChange} defaultValue={"am"}>
                                        <SelectTrigger className="border-border">
                                            <SelectValue defaultValue="am"/>
                                        </SelectTrigger>
                                        <SelectContent>
                                            <SelectGroup>
                                              <SelectItem value="am">AM</SelectItem>
                                              <SelectItem value="pm">PM</SelectItem>
                                            </SelectGroup>
                                        </SelectContent>
                                    </Select>
                                </FormControl>
                            </FormItem>
                        )}
                      />
                    </div>
                  }
                  {(form.watch("frequency.which") === 'weekly') &&
                    <FormField
                      control={form.control}
                      name="frequency.weekday"
                      render={({ field }) => (
                          <FormItem className="grid grid-cols-4 items-center gap-4 space-y-0">
                              <FormLabel className="text-right">{t('Weekday')}</FormLabel>
                              <FormControl>
                                  <Select onValueChange={field.onChange} defaultValue={"0"}>
                                      <SelectTrigger className="col-span-3 border-border">
                                          <SelectValue defaultValue="0"/>
                                      </SelectTrigger>
                                      <SelectContent>
                                          <SelectGroup>
                                            <SelectItem value="0">{t("Sunday")}</SelectItem>
                                            <SelectItem value="1">{t("Monday")}</SelectItem>
                                            <SelectItem value="3">{t("Tuesday")}</SelectItem>
                                            <SelectItem value="4">{t("Wednesday")}</SelectItem>
                                            <SelectItem value="5">{t("Thursday")}</SelectItem>
                                            <SelectItem value="6">{t("Friday")}</SelectItem>
                                            <SelectItem value="7">{t("Saturday")}</SelectItem>
                                          </SelectGroup>
                                      </SelectContent>
                                  </Select>
                              </FormControl>
                          </FormItem>
                      )}
                    />
                  }
                  {(form.watch("frequency.which") === 'monthly') &&
                    <FormField
                      control={form.control}
                      name="frequency.monthday"
                      render={({ field }) => (
                          <FormItem className="grid grid-cols-4 items-center gap-4 space-y-0">
                              <FormLabel className="text-right">{t('Day')}</FormLabel>
                              <FormControl>
                                  <Select onValueChange={field.onChange} defaultValue={"1"}>
                                      <SelectTrigger className="col-span-3 border-border">
                                          <SelectValue defaultValue="1"/>
                                      </SelectTrigger>
                                      <SelectContent>
                                          <SelectGroup>
                                              {months.map((item) => (
                                                <SelectItem value={item}>{item}</SelectItem>
                                              ))}
                                          </SelectGroup>
                                      </SelectContent>
                                  </Select>
                              </FormControl>
                          </FormItem>
                      )}
                    />
                  }
                  <FormField
                      control={form.control}
                      name="attachment"
                      render={({ field }) => (
                        <FormItem className="grid grid-cols-4 items-start gap-4 space-y-0">
                          <FormLabel className="text-right">{t('Attachment')}</FormLabel>
                          <FormControl>
                            <RadioGroup
                              onValueChange={field.onChange}
                              defaultValue={field.value}
                              className="flex flex-col space-y-1"
                            >
                              <FormItem className="flex items-center space-x-3 space-y-0">
                                <FormControl>
                                  <RadioGroupItem value="pdf" className="text-foreground border-foreground" />
                                </FormControl>
                                <FormLabel className="font-normal">
                                  PDF
                                </FormLabel>
                              </FormItem>
                              <FormItem className="flex items-center space-x-3 space-y-0">
                                <FormControl>
                                  <RadioGroupItem value="jpg" className="text-foreground border-foreground" />
                                </FormControl>
                                <FormLabel className="font-normal">
                                  JPG
                                </FormLabel>
                              </FormItem>
                              <FormItem className="flex items-center space-x-3 space-y-0">
                                <FormControl>
                                  <RadioGroupItem value="none" className="text-foreground border-foreground" />
                                </FormControl>
                                <FormLabel className="font-normal">
                                  {t("None")}
                                </FormLabel>
                              </FormItem>
                            </RadioGroup>
                          </FormControl>
                        </FormItem>
                      )}
                  />
                  <FormField
                      control={form.control}
                      name="recipients"
                      render={({ field }) => (
                          <FormItem className="grid grid-cols-4 items-center gap-4 space-y-0">
                              <FormLabel className="text-right">{t('Recipients')}</FormLabel>
                              <FormControl>
                                  <Input {...field} className="col-span-3 border-border" />
                              </FormControl>
                          </FormItem>
                      )}
                  />
                  <FormField
                      control={form.control}
                      name="subject"
                      render={({ field }) => (
                          <FormItem className="grid grid-cols-4 items-center gap-4 space-y-0">
                              <FormLabel className="text-right">{t('Subject')}</FormLabel>
                              <FormControl>
                                  <Input {...field} className="col-span-3 border-border" />
                              </FormControl>
                          </FormItem>
                      )}
                  />
                  <FormField
                      control={form.control}
                      name="message"
                      render={({ field }) => (
                          <FormItem className="grid grid-cols-4 items-center gap-4 space-y-0">
                              <FormLabel className="text-right">{t('Message')}</FormLabel>
                              <FormControl>
                                  <Textarea {...field} className="col-span-3 border-border" />
                              </FormControl>
                          </FormItem>
                      )}
                  />
                  {/* <FormField
                      control={form.control}
                      name="hide"
                      render={({ field }) => (
                        <FormItem className="grid grid-cols-4 items-center gap-4 space-y-0">
                          <FormLabel className="text-right">{t('Hide')}</FormLabel>
                          <FormControl>
                            <div className="flex items-center space-x-2 col-span-3">
                              <Checkbox id="hide" 
                                  checked={field.value}
                                  onCheckedChange={field.onChange}
                                  defaultChecked={false}
                              />
                              <label
                                htmlFor="hide"
                                className="text-sm font-medium leading-none peer-disabled:cursor-not-allowed peer-disabled:opacity-70"
                              >
                                {t("Do not show this report in your Scheduled Reports")}
                              </label>
                            </div>
                          </FormControl>
                        </FormItem>
                      )}
                  /> */}
                  <DialogFooter>
                      <Button type="submit" className="justify-end">
                          {t('Submit')}
                      </Button>
                  </DialogFooter>
                </form>
            </Form>
        </DialogContent>
      </Dialog>
    );
  }